/********************************************************************
//        NAME- CollapsableRowGroup
// DESCRIPTION- Controls a group of table rows that can be collapsed
//              (hidden) or expanded (revealed) as a group
********************************************************************/

/********************************************************************
// CollapsableRowGroup(header) : constructor
//   header : Element
// Creates a new CollapsableRowGroup and adds the controller widget
//   to the element
********************************************************************/
function CollapsableRowGroup(header)
{
  this.rows   = []; // an array of the rows in this group
  this.state  = 'empty'; // the current state of this group
  this.widget = new CollapsingWidget(this); // the controller widget

  // get the valid representation of the element
  if (isElement(header))
    this.header = header;
  else if (typeof header == 'string')
    this.header = getElement(header);
  else
    throw('CollabsableRowGroup: invalid header');

  this.init(); // initialize this object
}

/********************************************************************
// CollapsableRowGroup.init() : function
// Initializes the object by inserting the widget element
********************************************************************/
CollapsableRowGroup.prototype.init = function init()
{
  if (this.header.hasChildNodes())
  {
    var container = this.header.childNodes.item(0);
    if (container.hasChildNodes())
      container.insertBefore(this.widget.element, container.firstChild);
  }
}

/********************************************************************
// CollapsableRowGroup.expand() : function
// Expands the object, displaying all of its rows
********************************************************************/
CollapsableRowGroup.prototype.expand = function expand()
{
  var rows = this.rows; // local scope this.rows
  var numberOfRows = rows.length;
  for (var i=0; i<numberOfRows; i++)
  {
    // extend each row if necessary
    if (rows[i].expanded == undefined)
      extendElement(rows[i]);
    rows[i].show(); // display the given row
  }
  this.widget.setState('expanded'); // update the widget's state
}

/********************************************************************
// CollapsableRowGroup.collapse() : function
// Collapses the row, hiding all of its rows
********************************************************************/
CollapsableRowGroup.prototype.collapse = function collapse()
{
  var rows = this.rows; // local scope this.rows
  var numberOfRows = rows.length;
  for (var i=0; i<numberOfRows; i++)
  {
    // extend each row if necessary
    if (rows[i].expanded == undefined)
      extendElement(rows[i]);
    rows[i].hide(); // hide the given row
  }
  this.widget.setState('collapsed'); // update the widget's state
}

/********************************************************************
// CollapsableRowGroup.addRow(row) : function
//   row : Element
// Adds row to the rows array
********************************************************************/
CollapsableRowGroup.prototype.addRow = function addRow(row)
{
  // only add the row if it is not already contained in the row array
  if (!this.rows.contains(row))
    this.rows.push(row);

  // if there are more than 0 rows, set our state to expanded
  if (this.rows.length > 0)
    this.widget.setState('expanded');
  // if there are 0 rows, set our state to empty
  else
    this.widget.setState('empty');
}

/********************************************************************
// CollapsableRowGroup.removeRow(row) : function
//   row : Element
// Removes row from the rows array
********************************************************************/
CollapsableRowGroup.prototype.removeRow = function removeRow(row)
{
  // only try to remove the row if it already exists in the array
  if (this.rows.contains(row))
    this.rows.remove(row);

  // if there are 0 rows, set the state to empty
  if (this.rows.length <= 0)
    this.widget.setState('empty');
}
