/********************************************************************
//        NAME- CompareConfigController
// DESCRIPTION- controls all the functions of the compareConfig
//              panel
********************************************************************/

/********************************************************************
// CompareConfigController() : constructor
// Instantiates the CompareConfigController object
********************************************************************/
function CompareConfigController()
{
  this.defaultCategory = 'overview';
  this.currentCategory = null;
  this.currentViewLevel = 'summary';
  this.changesOnly = false;
  this.areCategoriesLoaded = false;

  this.viewURL = 'views/compareConfigs.php';
  this.saveCommand = 'commands/saveCompareConfig.php';

  this.viewController = new ViewController('compareConfigsFrame');

  this.controlDiv = extendElement(getElement('compareConfigControls'));
  this.categoryControl = extendElement(getElement('compareConfigCategory'));
  this.compareResultsButton = extendElement(getElement('viewComparisonResultsButton'));
  this.viewControls = extendElement(getElement('compareConfigViewControls'));
  this.leftFileControl = extendElement(getElement('compareConfigLeftFile' )); // the select element for our left file
  this.rightFileControl = extendElement(getElement('compareConfigRightFile')); // the select element for our right file
  this.changesOnlyControl = getElement('viewChangesOnly'); // the checkbox that sets the changes only setting

  this.saveButton = getElement('saveCompareConfig');
  this.printButton = getElement('printCompareConfig');
  
  this.configRequest = new ServerRequest('commands/getCapturedConfigurations.php');
  this.categoryRequest = new ServerRequest('commands/getConfigCategories.php');
  
  this.shadowBox = this.viewController.shadowBox;

  // initialize the object
  this.init();
}

/********************************************************************
// CompareConfigController.init() : function
// Initializes the object by assigning behaviors to the appropriate
// elements
********************************************************************/
CompareConfigController.prototype.init = function init()
{
  // levelControl changes the View Level
  var levelControl = getElement('compareConfigViewLevel');
  levelControl.controller = this;
  levelControl.onchange = function ()
  {
    this.controller.setViewLevel(this.value);
  }

  // categoryControl changes the currently visible category
  var categoryControl = getElement('compareConfigCategory');
  categoryControl.controller = this;
  categoryControl.onchange = function ()
  {
    this.controller.currentCategory = this.value;
    this.controller.runComparison(false);
  }

  // changesOnlyControl sets whether to show only changed items or all items
  this.changesOnlyControl.controller = this;
  this.changesOnlyControl.onclick = function ()
  {
    this.controller.setChangesOnly(this.checked);
  }

  // compareResultsButton calls the comparison command
  this.compareResultsButton.controller = this;
  this.compareResultsButton.action = function ()
  {
    this.controller.runComparison(true);
  }

  // saveButton opens the save dialog
  this.saveButton.controller = this;
  this.saveButton.action = function ()
  {
    this.controller.save();
  }
  this.saveButton.disable();
  
  this.printButton.controller = this;
  this.printButton.action = function ()
  {
    this.controller.viewController.getContentWindow().print();
  }
  this.printButton.disable();
  
  this.configRequest.controller = this;
  this.configRequest.callback = function (data)
  {
    this.controller.receiveConfigs(data);
  }
  
  this.categoryRequest.controller = this;
  this.categoryRequest.callback = function (data)
  {
    this.controller.receiveCategories(data);
  }

  this.compareResultsButton.disable();
  this.viewControls.cloak();
  
  return this;
}

/********************************************************************
// CompareConfigController.setViewLevel(value) : function
//   value : String
// Sets whether or not to display advanced entries
********************************************************************/
CompareConfigController.prototype.setViewLevel = function setViewLevel(value)
{
  this.currentViewLevel = value;
  this.runComparison(false);
}

/********************************************************************
// CompareConfigController.setChangesOnly(bool) : function
//   bool : Boolean
// Sets whether or not to display all entries or only changes entries
********************************************************************/
CompareConfigController.prototype.setChangesOnly = function setChangesOnly(bool)
{
  this.changesOnly = bool;
  this.runComparison(false);
}

/********************************************************************
// CompareConfigController.updateSurveyList() : function
// Gets a list of comparable surveys and populates the appropriate
//   select elements
********************************************************************/
CompareConfigController.prototype.getConfigs = function getConfigs()
{
  this.compareResultsButton.disable();
  // set leftControl to loading state
  var leftControl = this.leftFileControl; // local scope this.leftFileControl
  leftControl.hide(); // selects must be hidden when options are changed or they cease functioning
  clearChildren(leftControl); // remove all children of leftControl
  leftControl.appendChild(newElement('option', {'text':translate('MSG_LOADING')}, false)); // append an option labeled 'Loading'
  leftControl.disabled = true; // disable the select during loading
  leftControl.show(); // show the updated select

  // set rightControl to loading state
  var rightControl = this.rightFileControl; // local scope this.rightFileControl
  rightControl.hide(); // selects must be hidden when options are changed or they cease functioning
  clearChildren(rightControl); // remove all children of rightControl
  rightControl.appendChild(newElement('option', {'text':translate('MSG_LOADING')}, false)); // append an option labeled 'Loading'
  rightControl.disabled = true;// disable the select during loading
  rightControl.show(); // show the updated select
  
  this.configRequest.send();
}

/********************************************************************
// CompareConfigController.receiveSurveyList(surveyList) : function
//   surveyList : Object
// Called by the ServerRequest sent in updateSurveyList. Updates the
//   file controls to represent configurations in surveyList
********************************************************************/
CompareConfigController.prototype.receiveConfigs = function receiveConfigs(configData)
{
  var key = configData.key; // translation key for data
  var options = configData.data; // actual survey data
  var numberOfOptions = options.length;

  var leftControl = this.leftFileControl;
  leftControl.hide();
  clearChildren(leftControl);
  // create an option in leftControl for the original configuration
  leftControl.appendChild(newElement('option', {'value':'original','text':translate('MSG_ORIGINAL_CONFIGURATION')}, false));

  var rightControl = this.rightFileControl;
  rightControl.hide();
  clearChildren(rightControl);
  // create an option in rightControl for the current configuration
  rightControl.appendChild(newElement('option', {'value':'current','text':translate('MSG_CURRENT_CONFIGURATION')}, false));

  for (var i=0; i<numberOfOptions; i++)
  {
    var option = options[i]; // local scope appropriate option
    // create an option element in each file control representing this option
    leftControl.appendChild(newElement('option', {'value':option[key['filename']],'text':option[key['timestamp']]}, false));
    rightControl.appendChild(newElement('option', {'value':option[key['filename']],'text':option[key['timestamp']]}, false));
  }

  // enable and display each file control
  leftControl.disabled = false;
  leftControl.show();
  rightControl.disabled = false;
  rightControl.show();
  // refresh div content so selects work again
  this.controlDiv.cloak();
  this.controlDiv.uncloak();

  this.compareResultsButton.enable();
}

/********************************************************************
// CompareConfigController.getCategories() : function
// Sends the ServerRequest to retrieve the list of categories
//   available for viewing
********************************************************************/
CompareConfigController.prototype.getCategories = function getCategories()
{
  if (this.areCategoriesLoaded == false)
  {
    this.categoryControl.cloak();
    clearChildren(this.categoryControl);
    this.categoryControl.appendChild(newElement('option', {'text':translate('MSG_LOADING')}));
    this.categoryControl.disabled = true;
    this.categoryControl.uncloak();
    this.categoryRequest.send();
  }
}

/********************************************************************
// CompareConfigController.receiveCategories(categoryData) : function
//   categoryData : Object
// Receives the data from the server containing available categories
//   and displays them in the categoryControl
********************************************************************/
CompareConfigController.prototype.receiveCategories = function receiveCategories(categoryData)
{
  var key = categoryData.key;
  var categories = categoryData.data;
  var numberOfCategories = categories.length;
  this.categoryControl.cloak();
  clearChildren(this.categoryControl);
  this.categoryControl.appendChild(newElement('option', {'value':'all','text':translate('MSG_ALL')}));
  var useDefault = false;
  for (var i=0; i<numberOfCategories; i++)
  {
    var category = categories[i];
    var name = category[key['name']];
    var caption = category[key['caption']];
    var option = this.categoryControl.appendChild(newElement('option', {'value':name, 'text':caption}));
    if (name == this.defaultCategory)
    {
      useDefault = true;
      option.selected = true;
    }
  }
  this.categoryControl.disabled = false;
  this.categoryControl.uncloak();

  this.areCategoriesLoaded = true;
}

/********************************************************************
// CompareConfigController.appendSaveInfo(request) : function
//   request : ServerRequest
// Called by the save dialog, updates the request with specific
//   variables for this save mode
********************************************************************/
CompareConfigController.prototype.appendSaveInfo = function appendSaveInfo(request)
{
  request.addVariable('category', this.currentCategory);
  if (this.currentViewLevel == 'advanced')
    request.addVariable('advanced', 'yes');
  else
    request.addVariable('advanced', 'no');
  var leftValue = this.leftFileControl.value;
  if (leftValue == 'original')
  {
    request.addVariable('leftFile', 'surveybase.xml');
    request.addVariable('leftFileName', 'original');
  }
  else
  {
    request.addVariable('leftFile', this.leftFileControl.value);
    request.addVariable('leftFileName', this.leftFileControl.getElementsByTagName('option').item(this.leftFileControl.selectedIndex).innerHTML);
  }
  var rightValue = this.rightFileControl.value;
  if (rightValue == 'current')
  {
    request.addVariable('rightFile', 'survey.lastwebsession.xml');
    request.addVariable('rightFileName', 'current');
  }
  else
  {
    request.addVariable('rightFile', this.rightFileControl.value);
    request.addVariable('rightFileName', this.rightFileControl.getElementsByTagName('option').item(this.rightFileControl.selectedIndex).innerHTML);
  }
  request.addVariable('changesOnly', ((this.changesOnlyControl.checked) ? 'yes' : 'no'));
}

/********************************************************************
// CompareConfigController.save() : function
// Called by saveButton, determines which type of save to perform
//   Offline Mode : use older server-side save system
//   Online Mode : use newer client-side save system
********************************************************************/

CompareConfigController.prototype.save = function save()
{
  var header = (translate('MSG_SAVE') + ' ' + translate('MSG_SURVEY_COMPARISON'));
  if (CONFIG.IS_OFFLINE)
  {
    Frontend.saveDialog.openDialog(header, 'comparesurvey', this);
  }
  else
  {
    var query = 
    {
      attachment  : 1,
      filename    : 'comparesurvey',
      category    : this.currentCategory,
      advanced    : (this.currentViewLevel == 'advanced') ? 'yes' : 'no',
      changesOnly : (this.changesOnlyControl.checked) ? 'yes' : 'no'
    }
    var leftValue = this.leftFileControl.value;
    if (leftValue == 'original')
    {
      query['leftFile'] = 'surveybase.xml';
      query['leftFileName'] = 'original';
    }
    else
    {
      query['leftFile'] = leftValue;
      query['leftFileName'] = this.leftFileControl.getElementsByTagName('option').item(this.leftFileControl.selectedIndex).innerHTML
    }
    var rightValue = this.rightFileControl.value;
    if (rightValue == 'current')
    {
      query['rightFile'] = 'survey.lastwebsession.xml';
      query['rightFileName'] = 'current';
    }
    else
    {
      query['rightFile'] = rightValue;
      query['rightFileName'] = this.rightFileControl.getElementsByTagName('option').item(this.rightFileControl.selectedIndex).innerHTML;
    }
    Frontend.clientSave(this.saveCommand, query, header);
  }
}

/********************************************************************
// CompareConfigController.runComparison() : function
// Builds the variable array and calls refresh()
********************************************************************/
CompareConfigController.prototype.runComparison = function runComparison(run)
{
  this.printButton.disable();
  this.saveButton.disable();
  this.compareResultsButton.disable();
  var mode = (run != undefined && run === true) ? 'run' : 'read';
  if (this.currentCategory == undefined)
    this.currentCategory = this.defaultCategory;
  var leftValue = this.leftFileControl.value;
  if (leftValue == 'original')
  {
    this.viewController.setParameter('leftFilepath', 'surveybase.xml');
    this.viewController.setParameter('leftFilename', 'original');
  }
  else
  {
    this.viewController.setParameter('leftFilepath', this.leftFileControl.value);
    this.viewController.setParameter('leftFilename', this.leftFileControl.getElementsByTagName('option').item(this.leftFileControl.selectedIndex).innerHTML);
  }
  var rightValue = this.rightFileControl.value;
  if (rightValue == 'current')
  {
    this.viewController.setParameter('rightFilepath', 'survey.lastwebsession.xml');
    this.viewController.setParameter('rightFilename', 'current');
  }
  else
  {
    this.viewController.setParameter('rightFilepath', this.rightFileControl.value);
    this.viewController.setParameter('rightFilename', this.rightFileControl.getElementsByTagName('option').item(this.rightFileControl.selectedIndex).innerHTML);
  }
  this.viewController.setParameter('changesOnly', ((this.changesOnly == true) ? 'yes' : 'no'));
  this.viewController.setParameter('advanced', ((this.currentViewLevel == 'advanced') ? 'yes' : 'no'));
  this.viewController.setParameter('category', this.currentCategory);
  this.viewController.setParameter('mode', mode);
  this.viewController.setURL(this.viewURL);
}