/********************************************************************
//        NAME- DataRetriever
// DESCRIPTION- an object that contacts the server via a 
//              ServerRequest to get information and dynamically
//              update part of the UI
//       NOTES- intended as a superclass of many other components
********************************************************************/

/********************************************************************
// DataRetriever() : constructor
// Instantiates the DataRetriever
********************************************************************/
function DataRetriever(element)
{
  this.dataSource = ''; // the url that this object will recieve info from, should be overloaded in child classes
  this.data       = []; // the data recieved from the server, imperitive for updating UI
  this.key        = []; // the key for the data to dereference object labels
  this.variables  = []; // an array of variables to append to the ServerRequest

  // get the appropriate reference to the element
  if (isElement(element))
    this.element = element;
  else if (typeof element == 'string')
    this.element = getElement(element);
  else
    throw('DataRetriever: invalid element - '+element);
}

/********************************************************************
// DataRetriever.addVariable(name, value) : function
//   name : String
//   value : String
// Adds a name:value pair to the variable list
********************************************************************/
DataRetriever.prototype.addVariable = function addVariable(name, value)
{
  this.variables.push({'name':name, 'value':value});
  return this;
}

/********************************************************************
// DataRetriever.refresh() : function
// Sends a ServerRequest to get the latest data from the server
********************************************************************/
DataRetriever.prototype.refresh = function refresh()
{
  var variables = this.variables; // local scope of this.variables
  var numberOfVariables = variables.length;
  // create a new ServerRequest and instantiate it
  var request = new ServerRequest(this.dataSource);
  request.controller = this;
  request.callback = function (data)
  {
    this.controller.receiveData(data);
  }
  // iterate through this's variables and assign them to the ServerRequest
  for (var i=0; i<numberOfVariables; i++)
  {
    var variable = variables[i];
    request.addVariable(variable.name, variable.value);
  }
  request.send();
  return this;
}

/********************************************************************
// DataRetriever.receiveData(data) : function
//   data : Object
// Called from the ServerRequest after it gets new data
********************************************************************/
DataRetriever.prototype.receiveData = function receiveData(data)
{
  // if the data is a server error, report it
  if (data.name && data.name.indexOf('Error') >= 0)
  {
    Frontend.reportError(data.name+': '+data.value);
  }
  // if the data is not formatted as required {key, data}, call handleUnformattedData
  else if (data.data == undefined || data.key == undefined)
  {
    this.handleUnformattedData(data);
  }
  // store the data and update the UI
  else
  {
    this.data = data.data; // store the data to this.data
    this.key  = data.key; // store the key to this.key
    clearChildren(this.element);
    this.update(); // updates the UI with the new data
  }
  return this;
}

/********************************************************************
// DataRetriever.update() : function
// Should be overwritten by child classes, this redraws the object's
//   element with the new data
********************************************************************/
DataRetriever.prototype.update = function update()
{
  return false;
}

/********************************************************************
// DataRetriever.handleUnformattedData(data) : function
//   data : Object / String
// Allows a child class to perform alternate actions if the server
//   returns an alternate response (for example: '0' for success)
// If not overwritten by child classes, we inform the user of the
//   error
********************************************************************/
DataRetriever.prototype.handleUnformattedData = function handleUnformattedData(data)
{
  Frontend.inform('DataRetriever: data from dataSource is not formatted');
}
