/********************************************************************
//        NAME- SystemIDController
// DESCRIPTION- Constructor for the SystemIDController object
//      PASSED- messageElement (HTMLElement / string): this is the
//                  object that will contain the status messages
//              progressBar (ProgressBar): this is the progress bar
//                  that will be updated with the percent completion
//                  of the system ID
//              completeFunction (Function): the function that will
//                  be executed when the systemID is complete
//     RETURNS- 
//       NOTES- controls the running and status checking of a
//              system ID. Hooks into a message element and a
//              progress bar in order to display ID status
********************************************************************/
function SystemIDController(messageElement, progressBar, progressTable, completeFunction)
{
  this.startCommand  = 'commands/startSystemSurvey.php';
  this.statusCommand = 'commands/getProcessStatus.php';
  this.saveCommand   = 'commands/saveSystemSurvey.php';

  this.busyMessage    = translate('MSG_AI_IS_BUSY');
  this.runningMessage = (translate('MSG_PLEASE_WAIT')+' ... '+translate('MSG_SCANNING_HARDWARE'));
  this.savingMessage  = (translate('MSG_PROCESSING_RESULTS')+' ...');

  this.progressBar      = progressBar;
  this.progressTable    = progressTable;
  this.completeFunction = completeFunction;

  this.hasCrashedChild  = false;

  if (isElement(messageElement))
    this.messageElement = messageElement;
  else if (typeof messageElement == 'string')
    this.messageElement = getElement(messageElement);
  else
    throw('SystemIDController: bad messageElement');
}

/********************************************************************
//        NAME- SystemIDController.prototype.run
// DESCRIPTION- sends the ServerRequest to begin the system ID and
//              starts the status checking function
//      PASSED- 
//     RETURNS- 
//       NOTES- 
********************************************************************/
SystemIDController.prototype.run = function run()
{
  var request = new ServerRequest(this.startCommand);
  request.controller = this;
  request.callback = function(data)
  {
    this.controller.runResponse(data);
  }
  //Frontend.sendRequest(request);
  request.send();
  return this;
}

/********************************************************************
//        NAME- SystemIDController.prototype.runResponse
// DESCRIPTION- this is called by the initial ServerRequest created
//              by run()
//      PASSED- data (Object): this is the response from the server
//     RETURNS- 
//       NOTES- checks data for an error, if none found it extracts
//              the PID and runs beginUpdating()
********************************************************************/
SystemIDController.prototype.runResponse = function runResponse(data)
{
  if (data.name && data.name.indexOf('Error') >= 0)
  {
    clearChildren(this.messageElement);
    this.messageElement.appendChild(newText(data.name+': '+data.value));
  }
  else
  {
    switch (data.state)
    {
    case 'busy':
      clearChildren(this.messageElement);
      this.messageElement.appendChild(newText(this.busyMessage));
      break;
    default:
      this.processID = data.processID;
      this.beginUpdating();
    }
  }
  return this;
}

/********************************************************************
//        NAME- SystemIDController.prototype.beginUpdating
// DESCRIPTION- begins checking the status of the system ID
//      PASSED- 
//     RETURNS- 
//       NOTES- initializes the progress bar and message element,
//              then sends the first update ServerRequest, request
//              will call update() when finished
********************************************************************/
SystemIDController.prototype.beginUpdating = function beginUpdating()
{
  this.progressBar.setPercent(0);
  clearChildren(this.messageElement);
  this.messageElement.appendChild(newText(this.runningMessage));
  this.statusRequest = new ServerRequest(this.statusCommand);
  this.statusRequest.addVariable('processID', this.processID);
  this.statusRequest.controller = this;
  this.statusRequest.callback = function(data)
  {
    this.controller.update(data);
  }
  //Frontend.sendRequest(this.statusRequest);
  this.statusRequest.send();
  return this;
}

/********************************************************************
//        NAME- SystemIDController.prototype.update
// DESCRIPTION- updates the hooked message element and progress bar
//              with the latest status of the System ID
//      PASSED- data (Object): the response from the server
//     RETURNS- 
//       NOTES- checks data for an error, finding none it extracts
//              the state and percent complete. If state is a final
//              state, the result is saved to idResult.xml, otherwise
//              updating continues
********************************************************************/
SystemIDController.prototype.update = function update(data)
{
  if (data.name && data.name.indexOf('Error') >= 0)
  {
    clearChildren(this.messageElement);
    this.messageElement.appendChild(newText(data.name+': '+data.value));
  }
  else
  {
    var key = data.key;
    var data = data.data;
    var state = data[key['state']];
    var runningStates = ['canceling', 'running'];
    if (runningStates.contains(state))
    {
      clearChildren(this.messageElement);
      this.messageElement.appendChild(newText(this.runningMessage));
      this.progressBar.setPercent(data[key['percentComplete']]);
      this.updateProgressTable(key, data[key['children']]);
      var request = this.statusRequest;
      setTimeout(function()
      {
        request.send();
      }, 500);
    }
    else if (state == 'crashed')
    {
      clearChildren(this.messageElement);
      this.messageElement.appendChild(newText('(NT) CRASHED!'));
      this.updateProgressTable(key, data[key['children']]);
      this.getResult();
    }
    else
    {
      this.progressBar.setPercent(100);
      this.updateProgressTable(key, data[key['children']]);
      this.saveFile();
    }
  }
  return this;
}

SystemIDController.prototype.updateProgressTable = function updateProgressTable(key, data)
{
  if (this.progressTable != undefined)
  {
    clearChildren(this.progressTable);
    var numberOfChildren = data.length;
    for (var i=0; i<numberOfChildren; i++)
    {
      var child = data[i];
      var component = child[key['component']];
      var state = child[key['state']];
      var percentComplete = child[key['percentComplete']];
      var row = newElement('tr');
      if (state == 'crashed')
      {
        this.hasCrashedChild = true;
        addClass(row, 'crashed');
      }
      row.appendChild(newElement('td', {'text':component}));
      row.appendChild(newElement('td', {'text':state}));
      var progressCell = row.appendChild(newElement('td'));
      var progressBar = new ProgressBar().draw(progressCell);
      progressBar.setPercent(percentComplete);
      this.progressTable.appendChild(row);
    }
  }
}

/********************************************************************
//        NAME- SystemIDController.prototype.getResult
// DESCRIPTION- sends a ServerRequest to get the result of the ID,
//              in a normal completion, this is done in saveFile(),
//              but we can perform getResult() in case of an abnormal
//              terminiation here to clear the shared memory
//      PASSED-
//     RETURNS- 
//       NOTES-
********************************************************************/
SystemIDController.prototype.getResult = function getResult()
{
  var request = new ServerRequest(this.resultCommand);
  request.addVariable('processID',this.processID);
  request.callback = function(){};
  //Frontend.sendRequest(request);
  request.send();
  return this;
}

/********************************************************************
//        NAME- SystemIDController.prototype.saveFile
// DESCRIPTION- sends a ServerRequest to save the ID results to
//              idResult.xml
//      PASSED- 
//     RETURNS- 
//       NOTES- 
********************************************************************/
SystemIDController.prototype.saveFile = function saveFile()
{
  clearChildren(this.messageElement);
  this.messageElement.appendChild(newText(this.savingMessage));
  var request = new ServerRequest(this.saveCommand);
  request.addVariable('processID', this.processID);
  request.controller = this;
  request.callback = function(data)
  {
    this.controller.saveComplete(data);
  }
  //Frontend.sendRequest(request);
  request.send();
  return this;
}

/********************************************************************
//        NAME- SystemIDController.prototype.saveComplete
// DESCRIPTION- called after the file is saved
//      PASSED- data (Object): the response from the server
//     RETURNS- 
//       NOTES- checks for an error, finding none it calls complete()
********************************************************************/
SystemIDController.prototype.saveComplete = function saveComplete(data)
{
  if (data.name && data.name.indexOf('Error') >= 0)
  {
    clearChildren(this.messageElement);
    this.messageElement.appendChild(newText('(NT) '+data.name+': '+data.value));
  }
  else if (data == 0)
  {
    this.complete();
  }
  return this;
}

/********************************************************************
//        NAME- SystemIDController.prototype.complete
// DESCRIPTION- executes the completeFunction if there is one
//      PASSED- 
//     RETURNS- 
//       NOTES- 
********************************************************************/
SystemIDController.prototype.complete = function complete()
{
  if (this.completeFunction != undefined)
    this.completeFunction();
  return this;
}
