/********************************************************************
//        NAME- TestList
// DESCRIPTION- Constructor for the TestList class
//      PASSED- element (HTMLElement / string): the ul element
//              that will contain the data from the datasource
//              modeControl (HTMLElement / string): the ModeControl
//                  object that controls this list's mode
//              durationControl (HTMLElement / string): the 
//                  DurationControl object that controls the duration
//                  to run this list's Tests
//              errorControl (HTMLElement / string): the checkbox
//                  used to indicate 'Stop on First Error' setting
//     RETURNS- 
//       NOTES- Controls the appearance, selection and submission of
//              tests, inherited by QuickTestController and
//              CompleteTestController
********************************************************************/
function TestList(element, interactiveModeControl, unattendedModeControl, durationControl, errorControl, startButton, scheduleButton, scheduleTitle)
{
  inherit(this, new DataRetriever(element));

  if (isElement(interactiveModeControl))
    this.interactiveModeControl = interactiveModeControl;
  else if (typeof interactiveModeControl == 'string')
    this.interactiveModeControl = getElement(interactiveModeControl);
  else
    throw('TestList: invalid interactiveModeControl - ' + interactiveModeControl);

  if (isElement(unattendedModeControl))
    this.unattendedModeControl = unattendedModeControl;
  else if (typeof unattendedModeControl == 'string')
    this.unattendedModeControl = getElement(unattendedModeControl);
  else
    throw('TestList: invalid unattendedModeControl - ' + unattendedModeControl);

  if (isElement(errorControl))
    this.errorControl = errorControl;
  else if (typeof errorControl == 'string')
    this.errorControl = getElement(errorControl);
  else
    throw('TestList: invalid errorControl - ' + errorControl);

  if (isElement(startButton))
    this.startButton = startButton;
  else if (typeof startButton == 'string')
    this.startButton = getElement(startButton);
  else
    throw('TestList: invalid startButton - ' + startButton);

  if (CONFIG.IS_ONLINE == true)
  {
    if (isElement(scheduleButton))
      this.scheduleButton = scheduleButton;
    else if (typeof scheduleButton == 'string')
      this.scheduleButton = getElement(scheduleButton);
    else
      throw('TestList: invalid scheduleButton - ' + scheduleButton);
  }

  this.scheduleTitle = scheduleTitle;

  this.durationControl = durationControl;

  this.startCommand = 'commands/startStandardTests.php';
  this.testType     = '';

  this.allItems         = [];
  this.interactiveItems = [];
  this.mixedItems       = [];
  this.unattendedItems  = [];
}

/********************************************************************
//        NAME- TestList.prototype.init
// DESCRIPTION- Initializes mode controls
//      PASSED- 
//     RETURNS- 
//       NOTES- 
********************************************************************/
TestList.prototype.init = function init()
{
  this.interactiveModeControl.controller = this;
  this.interactiveModeControl.onclick = function ()
  {
    this.controller.setMode('interactive');
  }
  this.interactiveModeControl.checked = false;

  this.unattendedModeControl.controller = this;
  this.unattendedModeControl.onclick = function ()
  {
    this.controller.setMode('unattended');
  }
  this.unattendedModeControl.checked = true;

  this.startButton.controller = this;
  this.startButton.action = function ()
  {
    this.controller.beginTesting();
  }
  if (CONFIG.IS_ONLINE == true)
  {
    this.scheduleButton.controller = this;
    this.scheduleButton.action = function ()
    {
      Frontend.scheduleDialog.openDialog(this.controller.scheduleTitle, this.controller);
    }
  }

  this.currentMode = 'unattended';
  this.addVariable('mode', this.testType);
  return this;
}

/********************************************************************
//        NAME- TestList.prototype.update
// DESCRIPTION- Called after data has changed, redraws the list
//      PASSED- 
//     RETURNS- 
//       NOTES- calls drawAllDevicesItem and drawDeviceItem if data
//              contains testable devices
********************************************************************/
TestList.prototype.update = function update()
{
  this.allItems         = [];
  this.interactiveItems = [];
  this.mixedItems       = [];
  this.unattendedItems  = [];
  var data = this.data;
  var numberOfDevices = data.length;
  if (numberOfDevices > 0)
  {
    this.drawAllDevicesItem();
    for (var i=0; i<numberOfDevices; i++)
      this.drawDeviceItem(data[i]);
  }
  this.setMode(this.currentMode); // ensure we are displaying the appropriate tests
}

/********************************************************************
//        NAME- TestList.prototype.drawAllDevicesItem
// DESCRIPTION- Draws the 'All Devices' selection item
//      PASSED- 
//     RETURNS- 
//       NOTES- 
********************************************************************/
TestList.prototype.drawAllDevicesItem = function drawAllDevicesItem()
{
  var li = newElement('li', {'controller':this});
  this.allWidget = li.appendChild(newElement('input', {'type':'checkbox','controller':this}));
  li.appendChild(newText(translate('MSG_ALL_DEVICES')));
  li.widget = this.allWidget;
  this.allWidget.onclick = function ()
  {
    this.controller.allItemsToggle(this.checked);
  }
  this.element.appendChild(li);
}

/********************************************************************
//        NAME- TestList.prototype.drawDeviceItem
// DESCRIPTION- Draws an individual list item for the given device
//      PASSED- device (Object): the device info to use when drawing
//                  the LI element, referenced to in the LI element
//     RETURNS- 
//       NOTES- 
********************************************************************/
TestList.prototype.drawDeviceItem = function drawDeviceItem(device)
{
  var key = this.key;
  var li = newElement('li', {'controller':this, 'device':device});
  this.allItems.push(li);
  var interactivityLevel = device[key['interactivity']];
  var classArray = [];
  if (interactivityLevel == key['mixedInteractive'])
  {
    this.mixedItems.push(li);
    addClass(li, 'mixed');
    addClass(li, 'interactive');
    addClass(li, 'hidden');
  }
  else if (interactivityLevel == key['interactiveOnly'])
  {
    this.interactiveItems.push(li);
    addClass(li, 'interactive');
    addClass(li, 'hidden');
  }
  else
  {
    this.unattendedItems.push(li);
    addClass(li, 'unattended');
  }
  li.widget = li.appendChild(newElement('input', {'type':'checkbox','controller':this}));
  li.widget.onclick = function ()
  {
    this.controller.itemToggled(this);
  }
  var prefix = (hasClass(li, 'interactive')) ? ('(' + translate('MSG_INTERACTIVE') + ') ') : '';
  var span = li.appendChild(newElement('span', {'text':prefix}));
  li.prefixSpan = span;
  li.appendChild(newText(device[key['caption']] + ' - ' + device[key['description']]));
  this.element.appendChild(li);
}

/********************************************************************
//        NAME- TestList.prototype.allItemsToggle
// DESCRIPTION- Called from the allWidget, this toggles all of the
//              available devices on or off depending on the state
//              of the allWidget
//      PASSED- bool (boolean): this is the state that all items
//                  should be set to
//     RETURNS- 
//       NOTES- This is normally only called by the allWidget and it
//              passes the allWidget.checked state
********************************************************************/
TestList.prototype.allItemsToggle = function allItemsToggle(bool)
{
  if (this.currentMode == 'interactive')
    var items = this.allItems;
  else
    var items = this.unattendedItems;
  var numberOfItems = items.length;
  for (var i=0; i<numberOfItems; i++)
    items[i].widget.checked = bool;
}

/********************************************************************
//        NAME- TestList.prototype.itemToggled
// DESCRIPTION- Called when a widget other than the allWidget is
//              selected
//      PASSED- widget (HTMLElement): a reference to the selected
//                  widget, can be used in descendent classes for
//                  whatever means
//     RETURNS- 
//       NOTES- If not overloaded, this function toggles off the
//              allWidget when any other widget is checked, but does
//              not call the allItemsToggle
********************************************************************/
TestList.prototype.itemToggled = function itemToggled(widget)
{
  this.allWidget.checked = false;
}

/********************************************************************
//        NAME- TestList.prototype.setMode
// DESCRIPTION- Set's the list's current mode
//      PASSED- mode (String): the mode to switch the list to, valid
//                  choices are 'interactive' and 'unattended'
//     RETURNS- 
//       NOTES- This should only be called by the modeControl
********************************************************************/
TestList.prototype.setMode = function setMode(mode)
{
  this.currentMode = mode;
  if (mode == 'unattended')
  {
    var items = this.interactiveItems;
    var numberOfItems = items.length;
    for (var i=0; i<numberOfItems; i++)
    {
      var item = items[i];
      addClass(item, 'hidden');
      item.widget.checked = false;
    }
    var items = this.mixedItems;
    var numberOfItems = items.length;
    for (var i=0; i<numberOfItems; i++)
    {
      var item = items[i];
      addClass(item.prefixSpan, 'hidden');
      removeClass(item, 'interactive');
    }
  }
  else
  {
    var items = this.interactiveItems;
    var numberOfItems = items.length;
    for (var i=0; i<numberOfItems; i++)
    {
      var item = items[i];
      removeClass(item, 'hidden');
    }
    var items = this.mixedItems;
    var numberOfItems = items.length;
    for (var i=0; i<numberOfItems; i++)
    {
      var item = items[i];
      removeClass(item.prefixSpan, 'hidden');
      addClass(item, 'interactive');
      item.widget.checked = false;
    }
  }
}

/********************************************************************
//        NAME- TestList.prototype.beginTesting
// DESCRIPTION- Checks all selected tests and submits them
//      PASSED- 
//     RETURNS- 
//       NOTES- 
********************************************************************/
TestList.prototype.beginTesting = function beginTesting()
{
  // only allow beginTesting once. Do nothing if it has been set to false.
  // It will be set to false in this function immediately before sending the request.
  // It will be reset to true by the status controller once status monitoring has begun.
  // See request.callback below (within this function body)
  if ( Frontend.allowBeginTesting == false )
    return;

  this.request = new ServerRequest(this.startCommand, 'POST');
  this.request.controller = this;
  this.request.callback = function (data)
  {
    this.controller.testingBegan(data);
    Frontend.allowBeginTesting = true; //< re-enable beginTesting
  }
  var testResults = this.appendTestVariables(this.request);
  if (testResults === false)
  {
    var that = this;
    var dialog = Frontend.dialog;
    dialog.toggleHeader(true);
    if (this.testType == 'quick')
      dialog.setHeader(translate('MSG_QUICK_TEST'));
    else if (this.testType == 'complete')
      dialog.setHeader(translate('MSG_COMPLETE_TEST'));
    else
      dialog.toggleHeader(false);
    dialog.toggleBadge(false);
    dialog.setContent(translate('MSG_SELECT_DEVICE'));
    dialog.clearButtons();
    dialog.addButton(translate('MSG_OK'), null, function ()
    {
      dialog.hide();
    });
    dialog.show();
  }
  else if (testResults == 'durationError')
  {
    var that = this;
    var dialog = Frontend.dialog;
    dialog.toggleHeader(true);
    if (this.testType == 'quick')
      dialog.setHeader(translate('MSG_QUICK_TEST'));
    else if (this.testType == 'complete')
      dialog.setHeader(translate('MSG_COMPLETE_TEST'));
    else
      dialog.toggleHeader(false);
    dialog.toggleBadge(false);
    dialog.setContent(translate('MSG_LOOP_NUMBER_AND_TEST_TIME_MUST_BE_WHOLE'));
    dialog.clearButtons();
    dialog.addButton(translate('MSG_OK'), null, function ()
    {
      dialog.hide();
    });
    dialog.show();
  }
  else if (testResults == 'tooMuchTime')
  {
    var that = this;
    var dialog = Frontend.dialog;
    dialog.toggleHeader(true);
    if (this.testType == 'quick')
      dialog.setHeader(translate('MSG_QUICK_TEST'));
    else if (this.testType == 'complete')
      dialog.setHeader(translate('MSG_COMPLETE_TEST'));
    else
      dialog.toggleHeader(false);
    dialog.toggleBadge(false);
    dialog.setContent(translate('MSG_DURATION_TIME_TOO_HIGH'));
    dialog.clearButtons();
    dialog.addButton(translate('MSG_OK'), null, function ()
    {
      dialog.hide();
    });
    dialog.show();
  }
  else if (testResults == 'needsMedia')
  {
    var that = this;
    var dialog = Frontend.dialog;
    dialog.toggleHeader(true);
    if (this.testType == 'quick')
      dialog.setHeader(translate('MSG_QUICK_TEST'));
    else if (this.testType == 'complete')
      dialog.setHeader(translate('MSG_COMPLETE_TEST'));
    else
      dialog.toggleHeader(false);
    dialog.toggleBadge(false);
    dialog.setContent(translate('MSG_PLEASE_INSERT_MEDIA'));
    dialog.clearButtons();
    dialog.addButton(translate('MSG_OK'), null, function ()
    {
      Frontend.allowBeginTesting = false; //< turn off beginTesting in case button clicked multiple times.
      that.request.send();
      dialog.hide();
    });
    dialog.show();
  }
  else
  {
    Frontend.allowBeginTesting = false; //< turn off beginTesting in case button clicked multiple times.
    this.request.send();
  }
}

TestList.prototype.testingBegan = function testingBegan(data)
{
  if (data.name && data.name.indexOf('Error') >= 0)
  {
    alert(data.name + ': ' + data.value);
  }
  else
  {
    Frontend.statusController.startMonitoring(data, this);
  }
}

TestList.prototype.appendTestVariables = function appendTestVariables(request)
{
  var items = this.allItems;
  var needsMedia = false;
  var selectedItems = [];
  var numberOfItems = items.length;
  for (var i=0; i<numberOfItems; i++)
  {
    var item = items[i];
    if (item.widget.checked == true)
      selectedItems.push(item);
  }
  var numberOfDevices = selectedItems.length;
  if (numberOfDevices <= 0)
  {
    return false;
  }
  else
  {
    var key = this.key;
    request.addVariable('type', this.testType);
    request.addVariable('mode', this.currentMode);
    request.addVariable('stopOnError', (this.errorControl.checked) ? 'yes' : 'no');
    request.addVariable('durationType',  this.durationControl.type);
    request.addVariable('durationValue', this.durationControl.getValue());
    for (var i=0; i<numberOfDevices; i++)
    {
      request.addVariable('devices[]', selectedItems[i].device[key['component']] + ':::' + selectedItems[i].device[key['name']]);
      if (selectedItems[i].device[key['needsMedia']] == true)
        needsMedia = true;
    }
  }
  if (this.durationControl.getValue() <= 0 || isNaN(this.durationControl.getValue()) || this.durationControl.getValue().indexOf('.') > -1)
    return 'durationError';
  else if (this.durationControl.type == 'time' && this.durationControl.getValue() > 5760)
    return 'tooMuchTime';
  else if (needsMedia == true)
    return 'needsMedia';
  else
    return true;
}

TestList.prototype.appendScheduleInfo = function appendScheduleInfo(request)
{
  request.addVariable('command', this.testType);
  return this.appendTestVariables(request);
}
//*/
