#!/bin/sh

#
# Copyright (c) 2009-2010 Hewlett-Packard Development Company, L.P. <dannf@hp.com>
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
#     * Redistributions of source code must retain the above
#       copyright notice, this list of conditions and the following
#       disclaimer.
#
#     * Redistributions in binary form must reproduce the above
#       copyright notice, this list of conditions and the following
#       disclaimer in the documentation and/or other materials
#       provided with the distribution.
#
#     * Neither the name of the Hewlett-Packard Co. nor the names
#       of its contributors may be used to endorse or promote
#       products derived from this software without specific prior
#       written permission.
#
# THIS SOFTWARE IS PROVIDED BY HEWLETT-PACKARD DEVELOPMENT COMPANY
# ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
# NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
# FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
# HEWLETT-PACKARD DEVELOPMENT COMPANY BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

DISTROID_VERSION=1.4

is_redhat() {
    if [ ! -f "$ROOT/etc/redhat-release" ]; then
	return 1
    fi

    # Red Hat Enterprise Linux ES release 4 (Nahant Update 8)
    if grep -q '^Red Hat' "$ROOT/etc/redhat-release"; then
	return 0
    fi
}

redhat_vendor() {
    echo "Red Hat"
}

redhat_name() {
    echo "Red Hat"
}

redhat_flavor() {
    if grep -q "Enterprise Linux" "$ROOT/etc/redhat-release"; then
	echo "Enterprise Linux"
    fi
}

redhat_major() {
    if grep -q Nahant "$ROOT/etc/redhat-release"; then
	echo 4
    elif grep -q Tikanga "$ROOT/etc/redhat-release"; then
	echo 5
    fi
}

redhat_version() {
    local major="$(redhat_major)"
    if [ "$major" -eq 4 ]; then
	local minor=$(sed -r 's/^Red Hat Enterprise Linux ES release 4 \(Nahant Update ([0-9]+)\)$/\1/' < "$ROOT/etc/redhat-release")
	echo 4u${minor}
    elif [ "$major" -eq 5 ]; then
	sed -r 's/^Red Hat Enterprise Linux Server release ([0-9]+(\.[0-9]+)*) \(Tikanga\)$/\1/' < "$ROOT/etc/redhat-release"
    fi
}

redhat_description() {
    cat "$ROOT/etc/redhat-release"
}

redhat_uid() {
    local distro="rh"
    local version="$(redhat_version)"
    local flavor="$(redhat_flavor)"
    local flavabbrev=""

    if [ "$flavor" = "Enterprise Linux" ]; then
	flavabbrev="el"
    elif [ "$flavor" = "Advanced Server" ]; then
	flavabbrev="as"
    elif [ "$flavor" = "Advanced Workstation" ]; then
	flavabbrev="aw"
    fi

    echo "${distro}${flavabbrev}${version}"
}

is_debian() {
    if [ ! -f "$ROOT/etc/debian_version" ]; then
	return 1
    fi
    if is_ubuntu; then
	return 2
    fi
    return 0
}

debian_vendor() {
    echo "Debian"
}

debian_name() {
    echo "Debian GNU/Linux"
}

debian_flavor() {
    echo ""
}

debian_major() {
    # Debian uses a x.y major
    if grep -E -q '^[0-9]+\.[0-9]+\.*' "$ROOT/etc/debian_version"; then
	sed -r 's/^([0-9]+\.[0-9]+)\..*$/\1/' < "$ROOT/etc/debian_version"
    fi
}

debian_version() {
    # In testing/sid, /etc/debian_version contains codenames.
    # Check that we have a valid major number first so to avoid
    # emitting output for those releases.
    if [ -n "$(debian_major)" ]; then
	cat "$ROOT/etc/debian_version"
    fi
}

debian_description() {
    echo "Debian GNU/Linux $(cat $ROOT/etc/debian_version)"
}

debian_uid() {
    local distro="deb"
    local version="$(debian_version)"
    if [ -n "$version" ]; then
	echo "${distro}${version}"
    fi
}

is_suse() {
    test -f "$ROOT/etc/SuSE-release"
}

suse_vendor() {
    echo "Novell"
}

suse_name() {
    echo "SuSE Linux"
}

suse_flavor() {
    if grep -q "SUSE Linux Enterprise Server" "$ROOT/etc/SuSE-release"; then
	echo "Enterprise Server"
    fi
}

suse_major() {
    grep ^VERSION "$ROOT/etc/SuSE-release" | sed -r 's/^VERSION = ([0-9]+)$/\1/'
}

suse_version() {
    local major="$(suse_major)"
    local patchlevel="$(grep ^PATCHLEVEL $ROOT/etc/SuSE-release | \
                        sed -r 's/^PATCHLEVEL = ([0-9]+)$/\1/')"

    if [ "$patchlevel" -eq 0 ]; then
	echo "$major"
    else
	echo "${major}SP${patchlevel}"
    fi
}

suse_description() {
    head -1 "$ROOT/etc/SuSE-release" | sed 's/ (.*)$//'
}

suse_uid() {
    local distro="sl"
    local version="$(suse_version)"
    local flavor="$(suse_flavor)"
    local flavabbrev=""

    if [ "$flavor" = "Enterprise Server" ]; then
	flavabbrev="es"
    fi

    echo "${distro}${flavabbrev}${version}"
}

is_ubuntu() {
    if [ -f "$ROOT/etc/lsb-release" ] && \
	grep -q '^DISTRIB_ID=Ubuntu' "$ROOT/etc/lsb-release"; then
	return 0
    else
	return 1
    fi
}

ubuntu_vendor() {
    echo "Ubuntu"
}

ubuntu_name() {
    echo "Ubuntu"
}

ubuntu_flavor() {
    echo ""
}

ubuntu_major() {
    grep ^DISTRIB_RELEASE "$ROOT/etc/lsb-release" | \
	sed -r 's/^DISTRIB_RELEASE=(.*)$/\1/'
}

ubuntu_version() {
    ubuntu_major
}

ubuntu_description() {
    grep ^DISTRIB_DESCRIPTION "$ROOT/etc/lsb-release" | \
	sed -r 's/^DISTRIB_DESCRIPTION="(.*)"$/\1/'g
}

ubuntu_uid() {
    echo "ubu$(ubuntu_version)"
}

is_vmwesx() {
    if [ -f "$ROOT/etc/vmware-release" ] && \
	[ -x "$ROOT/usr/bin/vmware" ]; then
	return 0
    else
	return 1
    fi
}

vmwesx_vendor() {
    echo "VMware"
}

vmwesx_name() {
    echo "VMware ESX"
}

vmwesx_flavor() {
    local s="$($ROOT/usr/bin/vmware -v | head -1)"
    local flavor=""

    if echo "$s" | grep -E -q '^VMware ESX (.*) ([0-9]+)(\.[0-9]+)* build-[0-9]+$'; then
	flavor="$(echo $s | sed -r 's/^VMware ESX (.*) ([0-9]+)(\.[0-9]+)* build-[0-9]+$/\1/')"
    fi

    echo "$flavor"
}

vmwesx_major() {
    local maj=""

    if vmwesx_version | grep -E -q '^([0-9]+)(\.[0-9]+)*$'; then
	maj=$(vmwesx_version | sed -r 's/^([0-9]+)(\.[0-9]+)*$/\1/')
    fi
    echo "$maj"
}

vmwesx_version() {
    local s="$($ROOT/usr/bin/vmware -v | head -1)"
    local ver=""

    if echo "$s" | grep -E -q '^VMware ESX(.*) ([0-9]+)(\.[0-9]+)* build-[0-9]+$'; then
	ver="$(echo $s | sed -r 's/^VMware ESX(.*) (([0-9]+)(\.[0-9]+)*) build-[0-9]+$/\2/')"
    fi

    echo "$ver"
}

vmwesx_description() {
    "$ROOT/usr/bin/vmware" -v
}

vmwesx_uid() {
    echo "vmwesx$(vmwesx_version)"
}


is_xenserver() {
    if [ -f "$ROOT/etc/xensource-inventory" ]; then
    	return 0
    else
	return 1
    fi
}

xenserver_vendor() {
    echo "Citrix"
}

xenserver_name() {
    echo "Citrix XenServer"
}

xenserver_flavor() {
    grep "PRODUCT_NAME" /etc/xensource-inventory | cut -d= -f 2 | cut -d\' -f 2 | cut -d. -f 1
}

xenserver_major() {
    grep "PRODUCT_VERSION" /etc/xensource-inventory | cut -d= -f 2 | cut -d\' -f 2 | cut -d. -f 1
}

xenserver_version() {
    local ver="$(grep "PRODUCT_VERSION" /etc/xensource-inventory | cut -d= -f 2 | cut -d\' -f 2)"
    local build="$(grep "BUILD_NUMBER" /etc/xensource-inventory | cut -d= -f 2 | cut -d\' -f 2 | cut -d. -f 1)"
    echo "$ver-$build"
}

xenserver_description() {
    cat "$ROOT/etc/redhat-release"
}

xenserver_uid() {
    echo "xenserver$(xenserver_version)"
}

is_ax() {
    if [ -f "$ROOT/etc/asianux-release" ]; then
	return 0
    else
	return 1
    fi
}

ax_vendor() {
    echo "Asianux"
}

ax_name() {
    echo "Asianux"
}

ax_flavor() {
    local flavor=""

    if grep -E -q '^Asianux (.*) (([0-9]+)(\.[0-9]+)*) (\([a-zA-Z ]+( SP([0-9]+))?\))$' "$ROOT/etc/asianux-release"; then
	flavor="$(sed -r 's/^Asianux (.*) (([0-9]+)(\.[0-9]+)*) (\([a-zA-Z ]+( SP([0-9]+))?\))$/\1/' < $ROOT/etc/asianux-release)"
    fi

    echo "$flavor"
}

ax_major() {
    local major=""

    if grep -E -q '^Asianux (.*) (([0-9]+)(\.[0-9]+)*) (\([a-zA-Z ]+( SP([0-9]+))?\))$' "$ROOT/etc/asianux-release"; then
	major="$(sed -r 's/^Asianux (.*) (([0-9]+)(\.[0-9]+)*) (\([a-zA-Z ]+( SP([0-9]+))?\))$/\3/' < $ROOT/etc/asianux-release)"
    fi

    echo "$major"
}

ax_version() {
    local ver=""
    local spver=""

    if grep -E -q '^Asianux (.*) (([0-9]+)(\.[0-9]+)*) (\([a-zA-Z ]+( SP([0-9]+))?\))$' "$ROOT/etc/asianux-release"; then
	ver="$(sed -r 's/^Asianux (.*) (([0-9]+)(\.[0-9]+)*) (\([a-zA-Z ]+( SP([0-9]+))?\))$/\2/' < $ROOT/etc/asianux-release)"
	spver="$(sed -r 's/^Asianux (.*) (([0-9]+)(\.[0-9]+)*) (\([a-zA-Z ]+( SP([0-9]+))?\))$/\7/' < $ROOT/etc/asianux-release)"
    fi

    if [ -n "$spver" ]; then
	ver="${ver}SP${spver}"
    fi

    echo "$ver"
}

ax_description() {
    cat $ROOT/etc/asianux-release
}

ax_uid() {
    uid="ax"

    if [ "$(ax_flavor)" = "Server" ]; then
	uid="${uid}s"
    fi

    uid="${uid}$(ax_version)"

    echo "$uid"
}

## boilerplate for adding new distros
## cut & paste, then search & replace "name" with a distro

## This function is for internal use only. It returns 0 if the distribution
## is of class "name", non-zero otherwise
# is_name() {
# }

## This function outputs a string representing the vendor. This should
## be a company/organization name - e.g. "Red Hat", "Novell"
# name_vendor() {
# }

## This function outputs a string representing the distribution name.
## It should not distinguist between flavors.
## Good: "Red Hat", "SuSE Linux", "Debian GNU/Linux"
## Bad: "Red Hat Enterprise Linux", "SLES", "Debian"
# name_name() {
# }

## This function outputs a string that describes a specific flavor of
## the distribution - e.g., "Enterprise Server" or "Advanced Workstation"
# name_flavor() {
# }

## This function outputs a string that represents the major version of
## the distribution. For example, on RHEL 4.8 it would output "4", on
## Debian 5.0.3 it would output "5.0".
# name_major() {
# }

## This function outputs a string that represents the complete version
## of a distribution.
# name_version() {
# }

## This function outputs a single line of text that is used to describe
## the distribution. This is freeform text that may include codenames/etc.
## This is meant to be a pretty name exposed to users in management tools,
## not something that should be parsed.
# name_description() {
# }

## This should output a short string consisting of no whitespace that uniquely
## identifies this distro/flavor/version. The basically exists to create
## short tokens to be used in scripting/filenames.
# name_uid() {
# }

class=""

for c in redhat debian suse ubuntu vmwesx ax xenserver; do
    if is_${c}; then
	class="${c}"
    fi
done

if [ -z "${class}" ]; then
    echo "Error: Could not identify distribution" >&2
    exit 1
fi

for field in vendor name flavor major version description uid; do
    echo "$field: $(${class}_${field})"
done
