/********************************************************************
//        NAME- SurveyTable
// DESCRIPTION- Constructor for the SurveyTable class
//      PASSED- element (HTMLElement / string): the table element
//                  that will contain the data from the datasource
//              categoryControl (HTMLElement / string): the dropdown
//                  list that will contain our available dropdown
//     RETURNS- 
//       NOTES- the SurveyTable encapsulates the controls for 
//              displaying and controlling the data of the two 
//              tables in the Survey tab. Controls the switching of
//              categories and view levels
//              -- Inherits DataTable
********************************************************************/
function SurveyTable(element, categoryControl)
{
  inherit(this, new DataRetriever(element));

  this.defaultCategory  = 'overview';
  this.currentCategory  = '';
  this.currentViewLevel = 'summary';
  this.dataPadding      = 30;

  this.categories = [];
  this.shadowBox = new ShadowBox(this, this.element);

  if (isElement(categoryControl))
    this.categoryControl = extendElement(categoryControl);
  else if (typeof categoryControl == 'string')
    this.categoryControl = extendElement(getElement(categoryControl));
  else
    throw('SurveyTable: Invalid categoryControl');
}

/********************************************************************
//        NAME- SurveyTable.prototype.refresh
// DESCRIPTION- Overloaded version of refresh to include shadowBox
//      PASSED- 
//     RETURNS- 
//       NOTES- 
********************************************************************/
SurveyTable.prototype.refresh = function refresh()
{
  this.shadowBox.show();
  var variables = this.variables;
  var numberOfVariables = variables.length;
  var request = new ServerRequest(this.dataSource);  
  request.controller = this;
  request.callback = function (data)
  {
    this.controller.shadowBox.hide();
    this.controller.receiveData(data);
  }
  for (var i=0; i<numberOfVariables; i++)
  {
    var variable = variables[i];
    request.addVariable(variable.name, variable.value);
  }
  request.send();
  return this;
}

/********************************************************************
//        NAME- SurveyTable.prototype.setViewLevel
// DESCRIPTION- Changes the viewLevel of the SurveyTable
//      PASSED- level (string): the level we are setting the table
//                  to, values are 'advanced' or 'summary'
//     RETURNS- 
//       NOTES- 
********************************************************************/
SurveyTable.prototype.setViewLevel = function setViewLevel(level)
{
  this.shadowBox.show();
  var that = this;
  backgroundTask(function()
  {
    if (level == 'advanced')
      removeClass(that.element, 'hideAdvanced');
    else
      addClass(that.element, 'hideAdvanced');
    that.shadowBox.hide();
  });
  this.currentViewLevel = level;
}

/********************************************************************
//        NAME- SurveyTable.prototype.update
// DESCRIPTION- Called after data has changed, redraws the table
//      PASSED- 
//     RETURNS- 
//       NOTES- Calls updateCategories and setCategory with the
//              default category
********************************************************************/
SurveyTable.prototype.update = function update()
{
  this.currentCategory = '';
  this.updateCategories();
  this.setViewLevel('summary');
  this.setCategory(this.defaultCategory);
}

/********************************************************************
//        NAME- SurveyTable.prototype.updateCategories
// DESCRIPTION- Extracts category information from data and creates
//              category array
//      PASSED- 
//     RETURNS- 
//       NOTES- Calls updateCategoryControl on completion
********************************************************************/
SurveyTable.prototype.updateCategories = function updateCategories()
{
  var nameLabel    = this.key['name'];
  var captionLabel = this.key['caption'];
  var data = this.data;
  var categories = [];
  var numberOfCategories = data[this.key['categories']].length;
  clearChildren(this.element);
  for (var i=0; i<numberOfCategories; i++)
  {
    var categoryData = data[this.key['categories']][i];
    var category =
    {
      'name'    : categoryData[nameLabel],
      'caption' : categoryData[captionLabel],
    //'table'   : new HTMLTag('table', {'class':'surveyCategory'}), //this.element.appendChild(newElement('tbody', {'class':'surveyCategory ui_hidden'})),
      'div'     : newElement('div', {'class':'surveyCategory ui_hidden'}),
      'drawn'   : false,
      'index'   : i
    }
    categories.push(category);
    this.element.appendChild(category.div);
  }
  this.categories = categories;
  this.updateCategoryControl();
  return this;
}

/********************************************************************
//        NAME- SurveyTable.prototype.updateCategoryControl
// DESCRIPTION- Populates the categoryControl dropdown list with all
//              of the current category options
//      PASSED- 
//     RETURNS- 
//       NOTES- Calls updateCategoryControl on completion
********************************************************************/
SurveyTable.prototype.updateCategoryControl = function updateCategoryControl()
{
  var categoryControl = this.categoryControl;
  var categories = this.categories;
  var numberOfCategories = categories.length;
  categoryControl.hide();
  clearChildren(categoryControl);
  categoryControl.appendChild(newElement('option', {'value':'all', 'text':translate('MSG_ALL')}));
  for (var i=0; i<numberOfCategories; i++)
  {
    var category = categories[i];
    var elementOptions =
    {
      'value' : category.name,
      'text'  : category.caption
    }
    if (category.name == this.defaultCategory)
      elementOptions['selected'] = 'selected';
    categoryControl.appendChild(newElement('option', elementOptions));
  }
  categoryControl.show();
  return this;
}

/********************************************************************
//        NAME- SurveyTable.prototype.setCategory
// DESCRIPTION- Changes the displayed category
//      PASSED- categoryName (string): the name of the category to
//              display
//     RETURNS- 
//       NOTES- To speed up the interface, categories are only drawn
//              when they are first displayed, so the first display
//              of a single category will require loading, but all
//              subsequent loads of that category will be instant.
//              The logic of this function is contained in a 
//              timeout in order to process in the background.
********************************************************************/
SurveyTable.prototype.setCategory = function setCategory(categoryName)
{
  if (this.currentCategory != categoryName)
  {
    var that = this;
    var categories = this.categories;
    var numberOfCategories = categories.length;
    for (var i=0; i<numberOfCategories; i++)
    {
      var category = categories[i];
      if ((categoryName == 'all' && category.name != 'overview') || category.name == categoryName)
      {
        if (category.drawn == false)
          this.drawCategory(category);
        removeClass(category.div, 'ui_hidden');
      }
      else
      {
        addClass(category.div, 'ui_hidden');
      }
    }
    this.currentCategory = categoryName;
  }
}

/********************************************************************
//        NAME- SurveyTable.prototype.drawCategory
// DESCRIPTION- Converts the category data into HTMLElements
//      PASSED- category (Object): the category object that needs to
//                  be drawn
//     RETURNS- 
//       NOTES- This class should be overloaded by descendent classes
//              in order to parse the data specifically
********************************************************************/
SurveyTable.prototype.drawCategory = function drawCategory(category)
{
  category.drawn = true;
}



SurveyTable.prototype.appendSaveInfo = function appendSaveInfo(request)
{
  request.addVariable('category', this.currentCategory);
  if (this.currentViewLevel == 'advanced')
    request.addVariable('advanced', 'yes');
  else
    request.addVariable('advanced', 'no');
}

/*
SurveyTable.prototype.setChangesOnly = function setChangesOnly(bool, loading)
{
  this.element.parentNode.controller.shadowBox.show();
  if (loading == undefined || loading == false)
  {
    var controller = this;
    setTimeout(function()
    {
      controller.setChangesOnly(bool, true);
    },0);
  }
  else
  {
    if (bool == true)
      addClass(this.element, 'hideUnchanged');
    else
      removeClass(this.element, 'hideUnchanged');
    this.element.parentNode.controller.shadowBox.hide();
  }
}
*/
/*
SurveyTable.prototype.refresh = function refresh()
{
  var request = new ServerRequest(this.dataSource);
  request.controller = this;
  request.callback = function (data)
  {
    this.controller.receiveData(data);
  }
  request.send();
  return this;
}

SurveyTable.prototype.receiveData = function receiveData(data)
{
  if (data.name && data.name.indexOf('Error') >= 0)
  {
    alert(data.name+': '+data.value);
  }
  else
  {
    this.data = data.data;
    this.key  = data.key;
  }
}

/********************************************************************
//        NAME- SurveyTable.prototype.
// DESCRIPTION- 
//      PASSED- 
//     RETURNS- 
//       NOTES- 
********************************************************************/
/*
SurveyTable.prototype.getSectionChildren = function getSectionChildren(header)
{
  var sectionChildren = [];
  var foundNextHeader = false;
  var nextRow = header.nextSibling;
  do
  {
    if (isElement(nextRow))
    {
      if (hasClass(nextRow, 'header'))
        foundNextHeader = true;
      else
        sectionChildren.push(nextRow);
    }
    if (nextRow && nextRow.nextSibling)
      nextRow = nextRow.nextSibling;
    else
      nextRow = null;
  }
  while(nextRow != null && foundNextHeader == false);
  return sectionChildren;
}
*/
/*
SurveyTable.prototype.initTable = function initTable()
{
  this.element.controller = this;
  var headerRows = getTagsByClassName('tr', 'header', this.element);
  for (var i=0; i<headerRows.length; i++)
  {
    var header = headerRows[i];
    var childCount = this.getSectionChildren(header).length;
    var widget = newElement('img', {'class':'ui_treeControl', 'control':this, 'header':header});
    //var widget = document.createElement('img');
    //addClass(widget, 'ui_treeControl');
    //widget.control = this;
    //widget.header = header;
    if (childCount > 0)
    {
      widget.src = 'images/tree_collapse.gif';
      widget.onclick = function()
      {
        this.control.collapseSection(this.header, this)
      };
    }
    else
    {
      widget.src = 'images/tree_empty.gif';
      addClass(widget, 'ui_disabledControl');
    }
    header.getElementsByTagName('th')[0].insertBefore(widget, header.getElementsByTagName('th')[0].firstChild);
  }
}*/
/*
SurveyTable.prototype.collapseSection = function collapseSection(header, widget)
{
  if (!hasClass(widget, 'ui_disabledControl'))
  {
    var sectionChildren = this.getSectionChildren(header);
    for(var i=0; i<sectionChildren.length; i++)
      addClass(sectionChildren[i], 'ui_hidden');
    widget.src = 'images/tree_expand.gif';
    widget.onclick = function()
    {
      this.control.expandSection(this.header, this)
    };
    header.style.borderBottom = '1px solid #000000';
    return this;
  }
}

SurveyTable.prototype.expandSection = function expandSection(header, widget)
{
  if (!hasClass(widget, 'ui_disabledControl'))
  {
    var sectionChildren = this.getSectionChildren(header);
    for(var i=0; i<sectionChildren.length; i++)
      removeClass(sectionChildren[i], 'ui_hidden');
    widget.src = 'images/tree_collapse.gif';
    widget.onclick = function()
    {
      this.control.collapseSection(this.header, this)
    };
    header.style.borderBottom = '';
  }
  return this;
}
*/
