/********************************************************************
//        NAME- CustomTestController
// DESCRIPTION- controls all the functions of the customTest panel
//       NOTES- This Test Controller does not inherit from the 
//              TestList class as the others do. The functionality of
//              the CustomTestController necesitates a seperate class
********************************************************************/

/********************************************************************
// CustomTestController() : constructor
// Instantiates the CustomTestController object
********************************************************************/
function CustomTestController()
{
  // inherit from DataRetriever
  inherit(this, new DataRetriever('customTestList'));

  this.dataSource   = 'commands/getCustomTests.php'; // this is where we get our test and parameter info
  this.startCommand = 'commands/startCustomTests.php'; // we call this command to start tests

  this.testType = 'custom';
  this.currentMode = 'unattended';

  // setup a TestDurationController
  this.durationController = new TestDurationController('customTestLoopMode', 'customTestNumberOfLoops', 'customTestTimeMode', 'customTestNumberOfMinutes');

  // assign various Elements used by this object
  this.expandCollapseButton   = getElement('expandCollapseCustomButton'); // this is the button that expands or collapses the entire test list
  this.testCount              = getElement('customTestCount'); // this is the element that displays the amount of selected tests
  this.advancedCount          = getElement('customTestAdvancedCount'); // this is the element that contains the number of advanced tests
  this.advancedCountContainer = getElement('customTestAdvanced'); // this is the element that displays the advanced count
  this.interactiveModeControl = getElement('customTestInteractiveMode'); // this is the interactive mode radio button
  this.unattendedModeControl  = getElement('customTestUnattendedMode'); // this is the unattended mode radio button
  this.errorControl           = getElement('customTestStopOnError'); // this is the stop on errors checkbox
  this.parameterElement       = getElement('customTestParameters'); // this is the element that parameters are displayed in
  this.startButton            = getElement('beginCustomTesting'); // this is the button that begins the tests

  this.branches    = []; // an array that will contain all lists containing sublists
  this.leaves      = []; // an array that will contain all lists with no sublists
  this.interactive = []; // an array that will contain all of the interactive tests
  this.expanded    = false; // a boolean showing whether or not the entire list has been expanded or not

  // initialize the object
  this.init();
}

/********************************************************************
// CustomTestController.init() : function
// Initializes the object by assigning behaviors to the appropriate
//   elements
********************************************************************/
CustomTestController.prototype.init = function init()
{
  // extend and cloak the advancedCountContainer initially, will be uncloaked if the number is greater than 0
  extendElement(this.advancedCountContainer);
  this.advancedCountContainer.cloak();
  // expandCollapseButton toggles the 'Expand All', 'Collapse All' functionality
  this.expandCollapseButton.controller = this;
  this.expandCollapseButton.action = function ()
  {
    this.controller.toggleExpandCollapse();
  }
  // checkAll selected every test in the list
  var checkAll = getElement('customCheckAll');
  checkAll.controller = this;
  checkAll.action = function ()
  {
    this.controller.selectAll();
  }
  // uncheckAll deselected every test in the list
  var uncheckAll = getElement('customUncheckAll');
  uncheckAll.controller = this;
  uncheckAll.action = function ()
  {
    this.controller.deselectAll();
  }
  // interactiveModeControl displays all available tests
  this.interactiveModeControl.controller = this;
  this.interactiveModeControl.onclick = function ()
  {
    this.controller.setMode('interactive');
  }
  this.interactiveModeControl.checked = false;
  // unattendedModeControl hides and deselectes all interactive tests
  this.unattendedModeControl.controller = this;
  this.unattendedModeControl.onclick = function ()
  {
    this.controller.setMode('unattended');
  }
  this.unattendedModeControl.checked = true;
  // startButton begins the testing
  this.startButton.controller = this;
  this.startButton.action = function ()
  {
    this.controller.beginTesting();
  }
}

/********************************************************************
// CustomTestController.update() : function
// This function is called after the data has been updated
********************************************************************/
CustomTestController.prototype.update = function update()
{
  if (this.data.length > 0)
  {
    clearChildren(this.element); // remove all childNodes from this.element
    this.branches = [];
    this.leaves = [];
    this.expandCollapseButton.innerHTML = translate('MSG_EXPAND');
    this.expanded = false;
    this.drawCategories(); // draws all of the test categories
  }
  this.setMode(this.currentMode); // ensure we are displaying the appropriate tests
}

/********************************************************************
// CustomTestController.drawCategories() : function
// Draws the tests contained in this.data
********************************************************************/
CustomTestController.prototype.drawCategories = function drawCategories()
{
  var key = this.key; // local scope this.key
  var data = this.data; // local scope this.data
  var numberOfCategories = data.length;
  for (var i=0; i<numberOfCategories; i++)
  {
    var category = data[i];
    // create a new CollapsableListItem
    var li = new CollapsableListItem(newElement('li', {'text':category[key['caption']],'controller':this}));
    this.branches.push(li); // this item will contain sublists, so we place it in the branches array
    var devices = category[key['devices']];
    if (devices.length > 0)
    {
      var ul = li.element.appendChild(newElement('ul')); // create the sublist for this category's devices
      li.addChild(ul); // add this sublist to the CollapsableListItem
      this.drawDevices(devices, ul); // draws the devices for this category
    }
    this.element.appendChild(li.element);
  }
}

/********************************************************************
// CustomTestController.drawDevices(devices, canvas) : function
//   devices : Object
//   canvas : Element
// Converts the devices into Elements that are appended to canvas
********************************************************************/
CustomTestController.prototype.drawDevices = function drawDevices(devices, canvas)
{
  var key = this.key; // local scope of key
  var numberOfDevices = devices.length;
  for (var i=0; i<numberOfDevices; i++)
  {
    var device = devices[i];
    // make a new CollapsableListItem for this device
    var li = new CollapsableListItem(canvas.appendChild(newElement('li', {'text':device[key['caption']],'controller':this,'data':device})));
    this.branches.push(li); // this item will contain sublists so we place it in the branches array
    var tests = device[key['tests']];
    if (tests.length > 0)
    {
      var ul = li.element.appendChild(newElement('ul')); // create the sublist for this device's tests
      li.addChild(ul); // add this sublist to the CollapsableListItem
      this.drawTests(tests, ul); // draws the tests for this device
    }
  }
}

/********************************************************************
// CustomTestController.drawTests(tests, canvas) : function
//   tests : Object
//   canvas : Element
// Converts the tests into Elements that are appended to canvas
********************************************************************/
CustomTestController.prototype.drawTests = function drawTests(tests, canvas)
{
  var key = this.key; // local scope of this.key
  var numberOfTests = tests.length;
  for (var i=0; i<numberOfTests; i++)
  {
    var test = tests[i];
    // create an li element for this test
    var li = newElement('li', {'class':'leaf','controller':this,'data':test});
    // determine appropriate prefix for this test
    var prefix = '';
    // indicate if the test is interactive
    if (test[key['isInteractive']] == true)
    {
      addClass(li, 'interactive');
      prefix = ('(' + translate('MSG_INTERACTIVE') + ') ');
      this.interactive.push(li);
      li.hide();
    }
    // indicate if the test is advanced
    if (test[key['isAdvanced']] == true)
    {
      addClass(li, 'destructive');
      prefix = ('(' + translate('MSG_ADVANCED') + ') ');
    }
    // setup behavior for this element
    li.onclick = function ()
    {
      this.controller.selectLeaf(this);
    }
    // create a checkbox widget for this element
    li.widget = li.appendChild(newElement('input', {'type':'checkbox','controller':this}));
    li.widget.onclick = function ()
    {
      this.controller.widgetChecked(this);
    }

    li.appendChild(newText(prefix + test[key['caption']]));

    canvas.appendChild(li);

    this.leaves.push(li); // add this element to the leaves array
  }
}

/********************************************************************
// CustomTestController.beginTesting() : function
// Gathers the selected tests and their parameters
********************************************************************/
CustomTestController.prototype.beginTesting = function beginTesting()
{
  // only allow beginTesting once. Do nothing if it has been set to false.
  // It will be set to false in this function immediately before sending the request.
  // It will be reset to true by the status controller once status monitoring has begun.
  // See request.callback below (within this function body)
  if ( Frontend.allowBeginTesting == false )
    return;

  var key = this.key; // local scope of this.key
  var romTest = false;
  var romTestCanceled = false;
  var leaves = this.leaves; // local scope of this.leaves
  var selectedLeaves = []; // will contain all selectedLeaves
  var numberOfLeaves = leaves.length;
  // iterate through every leaf and see if its widget is checked
  for (var i=0; i<numberOfLeaves; i++)
  {
    var leaf = leaves[i];
    var data = leaf.data;
    if (leaf.widget.checked == true)
    {
      if (data[key['isRomTest']] == true)
      {
        if (confirm(translate('MSG_ROM_TEST_SELECTED')) == true)
        {
          selectedLeaves = [leaf];
          break;
        }
        else
        {
          romTestCanceled = true;
        }
      }
      else
      {
        selectedLeaves.push(leaf);
      }
    }
  }
  var numberOfTests = selectedLeaves.length;
  // if no tests are selected, inform the user
  if (numberOfTests <= 0)
  {
    // only display error if rom test was not canceled, otherwise rom test cancel was cause of 0 selected tests
    if (romTestCanceled === false)
      Frontend.inform(translate('MSG_PLEASE_SELECT_TEST'), translate('MSG_CUSTOM_TEST'));
  }
  else if (this.durationController.getValue() <= 0 || isNaN(this.durationController.getValue()) || this.durationController.getValue().indexOf('.') > -1)
  {
    Frontend.inform(translate('MSG_LOOP_NUMBER_AND_TEST_TIME_MUST_BE_WHOLE'), translate('MSG_CUSTOM_TEST'));
  }
  else if (this.durationController.type == 'time' && this.durationController.getValue() > 5760)
  {
    Frontend.inform(translate('MSG_DURATION_TIME_TOO_HIGH'), translate('MSG_CUSTOM_TEST'));
  }
  else if (this.currentLeaf != undefined && this.verifyParameters(this.currentLeaf) == false)
  {
    Frontend.inform(translate('MSG_INVALID_PARAMETERS')); // inform the user of invalid parameters
  }
  else if (this.currentLeaf == undefined || this.verifyParameters(this.currentLeaf))
  {
    // create and setup a new ServerRequest
    this.request = new ServerRequest(this.startCommand, 'POST');
    this.request.controller = this;
    this.request.addVariable('type', this.testType);
    this.request.addVariable('mode', this.currentMode);
    this.request.addVariable('stopOnError', (this.errorControl.checked) ? 'yes' : 'no');
    this.request.addVariable('durationType', this.durationController.type);
    this.request.addVariable('durationValue', this.durationController.getValue());
    var needsMedia = false;
    var romTest = false;
    for (var i=0; i<numberOfTests; i++)
    {
      // iterate through tests and add them to the ServerRequest
      var leaf = selectedLeaves[i];
      var testData = leaf.data;
      if (testData[key['needsMedia']] == true)
        needsMedia = true;
      if (testData[key['isRomTest']] == true && romTest == false)
        romTest = leaf;
      var testName = testData[key['name']];
      var deviceName = leaf.parentNode.parentNode.data[key['name']];
      var componentName = leaf.parentNode.parentNode.data[key['component']];
      var testID = (componentName + ':::' + deviceName + ':::' + testName);
      this.request.addVariable('tests[]', testID);
      // iterate through the tests' parameters and add them to the ServerRequest
      var parameters = testData[key['parameters']];
      var numberOfParameters = parameters.length;
      for (var j=0; j<numberOfParameters; j++)
      {
        var parameter = parameters[j];
        var name = parameter[key['name']];
        if (parameter[key['type']] == 'enum')
        {
          var index = parameter[key['value']];
          var value = parameter[key['options']][index][key['name']]
        }
        else
        {
          var value = parameter[key['value']];
        }
        this.request.addVariable(('test' + i + 'Parameters[]'), (name + ':' + value));
      }
    }

    this.request.callback = function (data)
    {
      this.controller.testingBegan(data);
      Frontend.allowBeginTesting = true; //< re-enable beginTesting
    }
    if (needsMedia == true)
    {
      var that = this;
      var dialog = Frontend.dialog;
      dialog.setHeader(translate('MSG_CUSTOM_TEST'));
      dialog.toggleHeader(true);
      dialog.toggleBadge(false);
      dialog.setContent(translate('MSG_PLEASE_INSERT_MEDIA'));
      dialog.toggleContent(true);
      dialog.clearButtons();
      dialog.addButton(translate('MSG_OK'), null, function ()
      {
        Frontend.allowBeginTesting = false; //< turn off beginTesting in case button clicked multiple times.
        that.request.send();
        dialog.hide();
      });
      dialog.toggleButtons(true);
      dialog.show();
    }
    else
    {
      Frontend.allowBeginTesting = false; //< turn off beginTesting in case button clicked multiple times.
      this.request.send();
    }
  }
}

/********************************************************************
// CustomTestController.testingBegan(data) : function
//   data : String / Object
// Called from a ServerRequest, data will either be a processID or
//   a server error object
********************************************************************/
CustomTestController.prototype.testingBegan = function testingBegan(data)
{
  // if the data returned is an error object we alert the error
  if (data.name && data.name.indexOf('Error') >= 0)
  {
    alert(data.name + ': ' + data.value);
  }
  // otherwise we tell the statusController to start monitoring the process
  else
  {
    Frontend.statusController.startMonitoring(data, this);
  }
}

/********************************************************************
// CustomTestController.setMode(mode) : function
//   mode : String
// Sets whether or not to display interactive tests.
// mode should be (unattended|interactive)
********************************************************************/
CustomTestController.prototype.setMode = function setMode(mode)
{
  var leaves = this.interactive; // local scope of this.interactive
  var numberOfLeaves = leaves.length;
  // if unattended, deselect and hide all interactive tests
  if (mode == 'unattended')
  {
    for (var i=0; i<numberOfLeaves; i++)
    {
      var leaf = leaves[i];
      leaf.widget.checked = false;
      leaf.hide();
    }
  }
  // otherwise show all tests
  else
  {
    for (var i=0; i<numberOfLeaves; i++)
      leaves[i].show();
  }
  this.currentMode = mode; // set mode
  this.deselectLeaf(); // deselect the selected leaf in case it was a test that got hidden
  this.countTests(); // update the number of tests selected 
}

/********************************************************************
// CustomTestController.expandAll() : function
// Expands all branches in the test list
********************************************************************/
CustomTestController.prototype.expandAll = function expandAll()
{
  var branches = this.branches; // local scope of this.branches
  var numberOfBranches = branches.length;
  for (var i=0; i<numberOfBranches; i++)
    branches[i].expand(); // expand the given branch
}

/********************************************************************
// CustomTestController.collapseAll() : function
// Collapses all branches in the test list
********************************************************************/
CustomTestController.prototype.collapseAll = function collapseAll()
{
  var branches = this.branches; // local scope of this.branches
  var numberOfBranches = branches.length;
  for (var i=0; i<numberOfBranches; i++)
    branches[i].collapse(); // collapse the given branch
}

/********************************************************************
// CustomTestController.toggleExpandCollapse() : function
// Determines how to draw the expandCollapseButton
********************************************************************/
CustomTestController.prototype.toggleExpandCollapse = function toggleExpandCollapse()
{
  // if this is not expanded, expand all list items and set expandCollapseButton to Collapse
  if (this.expanded == false)
  {
    this.expandAll();
    clearChildren(this.expandCollapseButton); 
    this.expandCollapseButton.appendChild(newText(translate('MSG_COLLAPSE')));
    this.expanded = true;
  }
  // otherwise, collapse all list items and set expandCollapseButton to Expand
  else
  {
    this.collapseAll();
    clearChildren(this.expandCollapseButton);
    this.expandCollapseButton.appendChild(newText(translate('MSG_EXPAND')));
    this.expanded = false;
  }
}

/********************************************************************
// CustomTestController.selectAll() : function
// Selects all tests that are not currently hidden
********************************************************************/
CustomTestController.prototype.selectAll = function selectAll()
{
  var key = this.key; // local scope of this.key
  var leaves = this.leaves; // local scope of this.leaves
  var numberOfLeaves = leaves.length;
  for (var i=0; i<numberOfLeaves; i++)
  {
    var leaf = leaves[i];
    if (this.currentMode == 'interactive' || 
        leaf.data[key['isInteractive']] == false)
      leaf.widget.checked = true;
  }
  this.countTests(); // update number of tests selected
}

/********************************************************************
// CustomTestController.deselectAll() : function
// Deselects all tests
********************************************************************/
CustomTestController.prototype.deselectAll = function deselectAll()
{
  var leaves = this.leaves; // local scope of this.leaves
  var numberOfLeaves = leaves.length;
  for (var i=0; i<numberOfLeaves; i++)
    leaves[i].widget.checked = false;
  this.countTests(); // update number of tests selected
}

/********************************************************************
// CustomTestController.widgetChecked(widget) : function
//   widget : Element
// Called whenever a widget controlled by this object is selected.
// Currently, widget is not used
********************************************************************/
CustomTestController.prototype.widgetChecked = function widgetChecked(widget)
{
  this.countTests(); // update number of tests selected
}

/********************************************************************
// CustomTestController.selectLeaf(leaf) : function
//   leaf : Element
// Called when a test is clicked on, loads parameters
********************************************************************/
CustomTestController.prototype.selectLeaf = function selectLeaf(leaf)
{
  // if we have a test selected, check its parameters first, only select a new test if the selected test's parameters are valid
  if (this.currentLeaf == undefined || 
      this.verifyParameters(this.currentLeaf) == true)
  {
    this.deselectLeaf(); // deselect current leaf
    addClass(leaf, 'selected');
    this.currentLeaf = leaf;
    this.drawParameters(leaf); // draw the test parameters for this leaf
  }
}

/********************************************************************
// CustomTestController.deselectLeaf() : function
// Deselects the currently selected test and removes its parameters
********************************************************************/
CustomTestController.prototype.deselectLeaf = function deselectLeaf()
{
  if (this.currentLeaf != undefined)
    removeClass(this.currentLeaf, 'selected');
  this.currentLeaf = null; // clear currentLeaf
  clearChildren(this.parameterElement); // remove parameters
}

/********************************************************************
// CustomTestController.countTests() : function
// Counts the number of selected tests and displays that number in
//   the UI
********************************************************************/
CustomTestController.prototype.countTests = function countTests()
{
  var key = this.key; // local scope of this.key
  var leaves = this.leaves; // local scope of this.leaves
  var numberOfLeaves   = leaves.length;
  var numberOfTests    = 0; // total number of selected tests
  var numberOfAdvanced = 0; // number of selected tests that are advanced
  for (var i=0; i<numberOfLeaves; i++)
  {
    var leaf = leaves[i];
    if (leaf.widget.checked == true)
    {
      numberOfTests++; // increment total number
      var data = leaf.data;
      // if this test is advanced, also increment advanced number
      if (data[key['isAdvanced']] == true)
        numberOfAdvanced++
    }
  }
  clearChildren(this.testCount);
  this.testCount.appendChild(newText(numberOfTests));
  // if there is 1 or more advanced tests selected, display the advanced count
  if (numberOfAdvanced > 0)
  {
    clearChildren(this.advancedCount);
    this.advancedCount.appendChild(newText(numberOfAdvanced));
    this.advancedCountContainer.uncloak();
  }
  // otherwise hide the advanced count
  else
  {
    this.advancedCountContainer.cloak();
  }
}

/********************************************************************
// CustomTestController.drawParameters(leaf) : function
//   leaf : Element
// Extracts the parameters in leaf.data and displays them
********************************************************************/
CustomTestController.prototype.drawParameters = function drawParameters(leaf)
{
  var key = this.key; // local scope of this.key
  var data = leaf.data; // local scope of leaf.data
  var deviceData = leaf.parentNode.parentNode.data; // extract the device data from the leaf's grandparent
  // determine appropriate suffix
  var suffix = '';
  if (data[key['isInteractive']] == true)
    suffix = (' (' + translate('MSG_INTERACTIVE') + ')');
  if (data[key['isAdvanced']] == true)
    suffix = (' (' + translate('MSG_ADVANCED') + ')');
  var testName = (data[key['caption']] + suffix); // the name of the selected test
  var deviceName = (deviceData[key['caption']] + ' - ' + deviceData[key['description']]); // the name of the device that owns this test
  var element = this.parameterElement; // local scope of this.parameterElement
  clearChildren(element);
  // append test info
  element.appendChild(newElement('p', {'class':'customTestName','text':testName}));
  element.appendChild(newElement('p', {'text':deviceName}));
  element.appendChild(newElement('p', {'class':'customTestDescription','text':data[key['description']]}));
  // extract parameters and display them
  var parameters = data[key['parameters']];
  var numberOfParameters = parameters.length;
  // if there are no parameters, signify that to the user
  if (numberOfParameters <= 0)
  {
    element.appendChild(newElement('p', {'class':'noTestParameters','text':(translate('MSG_NO') + ' ' + translate('MSG_TEST_PARAMETERS'))}));
  }
  // otherwise, iterate through each parameter and draw it
  else
  {
    var table = element.appendChild(newElement('table', {'class':'customTestParameterFields'})); // create the table
    var tbody = table.appendChild(newElement('tbody')); // create the tbody within the table
    for (var i=0; i<numberOfParameters; i++)
    {
      var parameter   = parameters[i];
      // extract variables
      var name        = parameter[key['name']];
      var caption     = parameter[key['caption']];
      var description = parameter[key['description']];
      var type        = parameter[key['type']];
      var value       = parameter[key['value']];
      var min         = parameter[key['min']] || null;
      var max         = parameter[key['max']] || null;

      var row = tbody.appendChild(newElement('tr', {'title':description}));
      parameter.tr = row;
      row.appendChild(newElement('th', {'text':caption}));
      var td = row.appendChild(newElement('td'));
      // each parameter is drawn differently based on its type
      switch (type)
      {
      case 'bool':
        // bools are represented with a simple checkbox
        var input = td.appendChild(newElement('input', {'type':'checkbox','controller':parameter}));
        input.checked = value;
        input.onclick = function ()
        {
          this.controller[key['value']] = this.checked;
        }
        break;
      case 'enum':
        // enums are represented with a select drop down box
        var select = td.appendChild(newElement('select', {'size':1,'controller':parameter}));
        var options = parameter[key['options']];
        var numberOfOptions = options.length;
        for (var j=0; j<numberOfOptions; j++)
        {
          var option = options[j];
          select.appendChild(newElement('option', {'value':option[key['name']],'text':option[key['caption']]}));
        }
        select.selectedIndex = value;
        select.onchange = function ()
        {
          this.controller[key['value']] = this.selectedIndex;
        }
        break;
      default:
        // everthing else (ints, text, etc) use a text input
        var input = td.appendChild(newElement('input', {'type':'text','value':value,'controller':parameter}));
        input.onkeyup = function ()
        {
          this.controller[key['value']] = this.value;
        }
        input.onchange = function ()
        {
          if (Frontend.customTests.verifyParameters(Frontend.customTests.currentLeaf))
            this.controller[key['value']] = this.value;
        }
        input.onblur = function ()
        {
          if (Frontend.customTests.verifyParameters(Frontend.customTests.currentLeaf))
            this.controller[key['value']] = this.value;
        }
      }
      var td = row.appendChild(newElement('td'));
      // if the parameter type == int, we display the min and max values (DEBUG)
      if (type == 'int' && (min != max))
      {
        var textArray = [];
        if (min != null)
          textArray.push(min);
        if (max != null)
          textArray.push(max);
        td.appendChild(newText(textArray.join(' - ')));
      }
    }
  }
}

/********************************************************************
// CustomTestController.verifyParameters(leaf) : function
//   leaf : Element
// Validates all of the parameter values given for leaf. Any invalid
//   values are highlighted.
// Validation is based on parameter type:
//   - Int values must fall between a given Min and Max
********************************************************************/
CustomTestController.prototype.verifyParameters = function verifyParameters(leaf)
{
  var returnVal = true; // we return true if all parameters are valid, false otherwise
  var key = this.key; // local scope of this.key
  var parameters = leaf.data[key['parameters']]; // extract parameter info from leaf
  var numberOfParameters = parameters.length;
  var badParameters = []; // will contain all bad parameter elements
  for (var i=0; i<numberOfParameters; i++)
  {
    var parameter = parameters[i];
    removeClass(parameter.tr, 'invalid'); // remove any previous invalid class attributes
    if (parameter[key['type']] == 'int')
    {
      var isBad = false;
      var max = (parameter[key['max']] != undefined)
        ? +parameter[key['max']] // coerce max into a number
        : max = null
      var min = (parameter[key['min']] != undefined)
        ? +parameter[key['min']] // coerce min into a number
        : min = null
      var value = this.convertValueToNumber(parameter[key['value']]);
      //+parameter[key['value']]; // coerce current parameter value into a number

      if (min != max)
      {
        // parameter is invalid if the value is > a given max value
        if (max != null)
        {
          if (value > max)
          {
            badParameters.push(parameter);
            isBad = true;
          }
        }
        // parameter is invalid if the value is < a given min value
        if (min != null && isBad == false)
        {
          if (value < min)
          {
            badParameters.push(parameter);
            isBad = true;
          }
        }
      }
      // parameter is invalid if the value is not a number (NaN)
      if (isNaN(value) && isBad == false)
        badParameters.push(parameter);
    }
  }
  var numberOfBadParameters = badParameters.length;
  if (numberOfBadParameters > 0)
  {
    returnVal = false; // return false if there are any invalid parameters
    for (var i=0; i<numberOfBadParameters; i++)
      addClass(badParameters[i].tr, 'invalid'); // mark all invalid parameters
  }
  return returnVal;
}

CustomTestController.prototype.convertValueToNumber = function convertValueToNumber(value)
{
  var number = value;
  var kilobyteMultiplier = 1024;
  var megabyteMultiplier = 1048576;
  var gigabyteMultiplier = 1073741824;
  if (!isNaN(value))
  {
    var number = +value;
  }
  else
  {
    var lastChar = value.substr(value.length-1).toUpperCase();
    var prefix = value.substr(0, value.length-1);
    if (!isNaN(prefix))
    {
      if (lastChar == 'K')
        number = prefix * kilobyteMultiplier;
      else if (lastChar == 'M')
        number = prefix  * megabyteMultiplier;
      else if (lastChar == 'G')
        number = prefix * gigabyteMultiplier;
    }
  }
  return number;
}