/*************************************************
*  This is a cross-browser shortcut for          *
* document.getElementById(). This will return the*
* result of 1 of 3 possible browser              *
* implementations to get an element by it's ID.  *
* elementID must be a string. Returns undefined  *
* if no implementation exists, or the result of  *
* the found implementation.                      *
*************************************************/

function getElement(elementID)
{
  if (document.getElementById)
    return document.getElementById(elementID);
  else if (document.all)
    return document.all[elementID];
  else if (document.layers)
    return document.layers[elementID];
  return undefined;
}

/*************************************************
*  Creates a new HTML element and assigns it the *
* specified attributes. The attributes argument  *
* should be a JS Object                          *
*************************************************/

function newElement(tagname, attributes, doExtend)
{
  doExtend = (doExtend != undefined) ? doExtend : true;
  var returnVal = null;
  if (attributes == undefined)
    attributes = {};
  if (document.createElement)
  {
    returnVal = document.createElement(tagname);
    var filter = new Object();
    for (var attribute in attributes)
    {
      if (filter[attribute] == undefined) // only add unique values
      {
        switch (attribute)
        {
        case 'class':
          returnVal.className = attributes[attribute];
          break;
        case 'text':
          returnVal.appendChild(newText(attributes[attribute]));
          break;
        case 'style':
          for (var selector in attributes[attribute])
          {
            if (filter[selector] == undefined)
            {
              var rule = attributes[attribute][selector];
              returnVal.style[selector] = rule;
            }
          }
          break;
        default:
          returnVal[attribute] = attributes[attribute];
        }
      }
    }
    if (doExtend)
      extendElement(returnVal);
  }
  return returnVal;
}

/*************************************************
*  Creates a new TextNode with the given value.  *
*************************************************/

function newText(value)
{
  var reutrnVal = null;
  if (document.createTextNode)
    returnVal = document.createTextNode(value);
  return returnVal;
}

/*************************************************
*  Performs an action on every element within an *
* array. Action must be a function reference.    *
* Mode can be "withElement" (default) or         *
* "asElement". "withElement" performs the action *
* as action(element), while "asElement" performs *
* the action as element.action(). Returns an     *
* array containing the result of each action.    *
*************************************************/

function foreach(array, action, mode)
{
  var validModes = ["withElement", "asElement"];
  if (mode == undefined)
    mode = "withElement";

  if (!validModes.contains(mode))
    throw("invalid mode in foreach() - \""+mode+"\"");

  var results = [];
  for(var i=0; i<array.length; i++)
  {
    switch(mode)
    {
    case("withElement"):
      results.push(action(array[i]));
      break;
    case("asElement"):
      if (array[i].action)
        results.push(array[i].action());
      else
        results.push(undefined);
      break;
    }
  }
  return results;
}

/*************************************************
*  Gets all elements contained in element that   *
* have one of the tag names provided. Returns an *
* array of these elements.                       *
*************************************************/

function getElementsByTagNames(element)
{
  var returnVal = [];
  for (var i=1; i<arguments.length; i++)
  {
    var tags = element.getElementsByTagName(arguments[i]);
    for (var j=0; j<tags.length; j++)
      returnVal.push(tags[j]);
  }
  return returnVal;
}

/*************************************************
*  Gets all elements in the given element that   *
* contain the given className. If element is not *
* given, the document is searched.               *
*************************************************/

function getTagsByClassName(tagName, className, element)
{
  var returnVal = new Array();
  if (tagName == undefined || className == undefined)
    return returnVal;
  if (element == undefined)
    element = document;
  var tags = element.getElementsByTagName(tagName);
  var i;
  for (i=0; i<tags.length; i++)
  {
    if (hasClass(tags[i], className))
      returnVal.push(tags[i]);
  }
  return returnVal;
}

/*************************************************
*  Checks to see if object is an element by      *
* seeing if it contains a className. On all HTML *
* elements, className should return a string,    *
* while non-elements (text nodes, etc) will not  *
*************************************************/

function isElement(object)
{
  if (object && (object.className || object.className == ''))
    return true;
  return false;
}

function resolveElement(elementRef)
{
  if (isElement(elementRef))
    return elementRef;
  else if (typeof elementRef == 'string')
    return getElement(elementRef);
  else
    return elementRef;
}

/*************************************************
*  addClass and removeClass add and remove       *
* strings from element.className respectively.   *
* Both functions return the new element.className*
* hasClass returns true if the given element     *
* contains classItem in its className            *
*************************************************/

function addClass(element, classItem)
{
  if (element != undefined)
  {
    var className = element.className.split(" ");
    if (!className.contains(classItem))
      className.push(classItem);
    element.className = className.join(" ");
    return element.className;
  }
}

function removeClass(element, classItem)
{
  if (element != undefined)
  {
    var className = element.className.split(" ");
    if (className.contains(classItem))
      className.removeAll(classItem);
    element.className = className.join(" ");
    return element.className;
  }
}

function replaceClass(element, oldClassItem, newClassItem, doForceAdd)
{
  if (doForceAdd == undefined)
    doForceAdd = false;
  var className = element.className.split(" ");
  if (className.contains(oldClassItem) || doForceAdd)
  {
    className.removeAll(oldClassItem);
    if (!className.contains(newClassItem))
      className.push(newClassItem);
  }
  element.className = className.join(" ");
  return element.className;
}

function hasClass(element, classItem)
{
  var returnVal = false;
  var className = element.className.split(" ");
  if (className.contains(classItem))
    returnVal = true;
  return returnVal;
}

/*************************************************
*  These functions are for making classes and    *
* subclasses in OOJS. makeClass takes a reference*
* to a class function and adds a _classType      *
* variable to it so all objects can be compared  *
* to their Class. makeSubClass sets up a Class to*
* inherit public members and methods from another*
* Class. The parent variable is added that allows*
* the Subclass to call functions directly from   *
* its parent. makeClass is also called on the    *
* new Subclass, so this isn't necessary          *
*************************************************/

function makeClass(newClass)
{
  //newClass.prototype.classType = newClass;
  var functionName = new String(newClass).match(/function[\s]([^\s\(]+)?/);
  if (functionName != null && functionName.length > 1)
  {
    newClass.prototype.classid = functionName[1];
    newClass.prototype.toString = function()
    {
      return '[object '+this.classid+']';
    }
  }
}

function makeSubClass(oldClass, newClass)
{
  newClass.prototype = new oldClass();
  newClass.prototype.parent = new oldClass();
  makeClass(newClass);
}

/*************************************************
*  Allows an object to inherit the properties of *
* another object. This can accept a Class or an  *
* instantiated object.                           *
*************************************************/

inherit = function(target, object)
{
  if (typeof object == 'function')
    object = new object();
  var filter = new Object();
  for (var key in object)
  {
    if (object[key] != filter[key])
    {
      if (target[key] == undefined)
      {
        if (object[key].clone != undefined)
          target[key] = object[key].clone();
        else
          target[key] = object[key];
      }
      else
      {/*
        if (this.super == undefined)
          this.super = {};
        if (object[key].clone != undefined)
          this.super[key] = object[key].clone();
        else
          this.super[key] = object[key];
      */}
    }
  }
  return target;
}

/*************************************************
*  Removes all childNodes from an element        *
*************************************************/

function clearChildren(elementID)
{
  var returnVal = false;
  if (isElement(elementID))
    var element = elementID;
  else
    var element = getElement(elementID);

  if ((element.hasChildNodes && element.hasChildNodes() == true) || (element.childNodes && element.childNodes.length > 0))
  {
    var numberOfChildNodes = element.childNodes.length;
    for(var i=0; i<numberOfChildNodes; i++)
    {
      element.removeChild(element.childNodes[0]);
    }
    returnVal = true;
  }
  return returnVal;
}

/*************************************************
*  Set the selected option of a select box to the*
* option whose value matches the given value. If *
* no such option exists, false is returned,      *
* otherwise true is returned.                    *
*************************************************/

function changeSelectValue(selectID, value)
{
  var returnVal = false;
  if (!document.getElementById(selectID))
    return returnVal;
  var options = document.getElementById(selectID).getElementsByTagName('option');
  var i;
  for(i=0; i<options.length && returnVal == false; i++)
  {
    if (options[i].value == value)
    {
      document.getElementById(selectID).selectedIndex = i;
      returnVal = true;
    }
  }
  return returnVal;
}

/*************************************************
*  Forces javascript code to pause execution for *
* the specified number of milliseconds           *
*************************************************/

function sleep(milliseconds)
{
  timeStamp = new Date();
  var newStamp = null;

  do
    var newStamp = new Date();
  while(newStamp-timeStamp < milliseconds);
} 

/*************************************************
*  Encodes a string into a URL compatible query  *
* string. Uses encodeURIComponent but leaves     *
* spaces as '+' rather than '%20'                *
*************************************************/

function encodeQueryComponent(string)
{
  var URIencode = encodeURIComponent(string);
  return URIencode.replace(/%20/g, "+");
}

function decodeQueryComponent(string)
{
  var altered = string.replace(/\+/g, '%20');
  return decodeURIComponent(altered);
}

/*************************************************
*  Adds additional methods to an Object, used to *
* get HTMLElements in IE to inherit necesarry    *
* controls                                       *
*************************************************/

function extendElement(element)
{
  if (element != Body)
  {
    element.getTop = function getTop()
    {
      var returnVal = 0;
      if (this.offsetTop != undefined)
        returnVal = this.offsetTop;
      if (this.offsetParent != undefined && this.offsetParent.offsetTop != undefined)
        returnVal += this.offsetParent.offsetTop;
      return returnVal;
    }
  
    element.setTop = function setTop(value, unit)
    {
      unit = unit || 'px';
      value = (value < 0 || value == undefined) ? 0 : value;
      if (this.style != undefined)
        this.style.top = (value + unit);
      return this;
    }
    
    element.clearTop = function clearTop()
    {
      this.style.top = "";
      return this;
    }
  
    element.getLeft = function getLeft()
    {
      var returnVal = 0;
      if (this.offsetLeft != undefined)
        returnVal = this.offsetLeft;
      if (this.offsetParent != undefined && this.offsetParent.offsetLeft != undefined)
        returnVal += this.offsetParent.offsetLeft;
      return returnVal;
    }
  
    element.setLeft = function setLeft(value, unit)
    {
      unit = unit || 'px';
      value = (value < 0 || value == undefined) ? 0 : value;
      this.style.left = (value + unit);
      return this;
    }
    
    element.clearLeft = function clearLeft()
    {
      this.style.left = "";
      return this;
    }
  
    element.getRight = function getRight()
    {
      var returnVal = 0;
      if (this.offsetWidth != undefined)
        returnVal = (this.getLeft + this.offsetWidth);
      return returnVal;
    }
  
    element.setRight = function setRight(value, unit)
    {
      unit = unit || 'px';
      value = (value < 0 || value == undefined) ? 0 : value;
      this.style.right = (value + unit);
      return this;
    }
    
    element.clearRight = function clearRight()
    {
      this.style.right = "";
      return this;
    }
  
    element.getBottom = function getBottom()
    {
      var returnVal = 0;
      if (this.offsetHeight != undefined)
        returnVal = (this.getTop() + element.offsetHeight);
      return returnVal;
    }
  
    element.setBottom = function setBottom(value, unit)
    {
      unit = unit || 'px';
      value = (value < 0 || value == undefined) ? 0 : value;
      this.style.bottom = (value + unit);
      return this;
    }
    
    element.clearBottom = function clearBottom()
    {
      this.style.bottom = "";
      return this;
    }
  
    element.getWidth = function getWidth()
    {
      var returnVal = 0;
      if (this.offsetWidth != undefined)
        returnVal = this.offsetWidth;
      return returnVal;
    }
  
    element.setWidth = function setWidth(value, unit)
    {
      unit = unit || 'px';
      value = (value < 0) ? 0 : value;
      this.style.width = (value + unit);
      return this;
    }
    
    element.clearWidth = function clearWidth()
    {
      this.style.width = "";
      return this;
    }
  
    element.getHeight = function getHeight()
    {
      var returnVal = 0;
      if (this.offsetHeight != undefined)
        returnVal = this.offsetHeight;
      return returnVal;
    }
  
    element.setHeight = function setHeight(value, unit)
    {
      unit = unit || 'px';
      value = (value < 0 || value == undefined) ? 0 : value;
      this.style.height = (value + unit);
      return this;
    }
    
    element.clearHeight = function clearHeight()
    {
      this.style.height = "";
      return this;
    }
  
    element.conformTo = function conformTo(element)
    {
      if (element.getWidth  == undefined || 
          element.getHeight == undefined || 
          element.getTop    == undefined ||
          element.getLeft   == undefined   );
        extendElement(element);
      this.setWidth(element.getWidth());
      this.setHeight(element.getHeight());
      this.setTop(element.getTop());
      this.setLeft(element.getLeft());
      return this;
    }
  
    element.center = function center()
    {
      // get left position
      var elementHorizontalCenter = (Body.getLeft() + (Body.getWidth() / 2));
      var thisHorizontalCenter    = (this.getWidth() / 2);
      var leftPosition = (elementHorizontalCenter - thisHorizontalCenter);
      // get top position
      var elementVerticalCenter = (Body.getTop() + (Body.getHeight() / 2));
      var thisVerticalCenter    = (this.getHeight() / 2);
      var topPosition = (elementVerticalCenter - thisVerticalCenter);
      // center this
      this.setTop(topPosition);
      this.setLeft(leftPosition);
    }

    element.centerHorizontally = function centerHorizontally()
    {
      // get left position
      var elementHorizontalCenter = (Body.getLeft() + (Body.getWidth() / 2));
      var thisHorizontalCenter    = (this.getWidth() / 2);
      var leftPosition = (elementHorizontalCenter - thisHorizontalCenter);
      this.setLeft(leftPosition);
    }
  
    element.centerTo = function centerTo(element)
    {
      if (isElement(element))
        element = element;
      else if (typeof element == 'string')
        element = getElement(element);
      else
        throw('centerTo: invalid element - ' + element);
      if (element.extended == undefined)
        extendElement(element);
      // get left position
      var elementHorizontalCenter = (element.getLeft() + (element.getWidth() / 2));
      var thisHorizontalCenter    = (this.getWidth() / 2);
      var leftPosition = (elementHorizontalCenter - thisHorizontalCenter);
      // get top position
      var elementVerticalCenter = (element.getTop() + (element.getHeight() / 2));
      var thisVerticalCenter    = (this.getHeight() / 2);
      var topPosition = (elementVerticalCenter - thisVerticalCenter);
      // center this
      this.setTop(topPosition);
      this.setLeft(leftPosition);
    }
  
    element.show = function show()
    {
      removeClass(this, 'hidden');
      return this;
    }
  
    element.hide = function hide()
    {
      addClass(this, 'hidden');
      return this;
    }

    element.flicker = function flicker()
    {
      this.hide();
      var that = this;
      backgroundTask(function ()
      {
        that.show();
      });
    }
  
    element.cloak = function cloak()
    {
      addClass(this, 'invisible');
      return this;
    }
  
    element.uncloak = function uncloak()
    {
      removeClass(this, 'invisible');
      return this;
    }

    element.flash = function flash()
    {
      this.cloak();
      var that = this;
      backgroundTask(function ()
      {
        that.uncloak();
      });
    }
  
    element.fade = function fade(transparency)
    {
      var opacity = 100.0 - transparency;
      if (opacity < 0)
        opacity = 0.0;
      if (opacity > 100)
        opacity = 100.0;
      if (this.style != undefined)
      {
        //if (this.style.opacity != undefined)
          this.style.opacity = opacity/100; // W3C Standards Compliant
        //else if (this.filters)
          this.style.filter = 'alpha(opacity='+opacity+')'; // IE Proprietary
        //else
          this.style.MozOpacity = opacity/100; // Mozilla Proprietary
      }
      return this;
    }
  
    element.extended = true;
  }
  return element;
}

function backgroundTask(task)
{
  setTimeout(task,0);
  return true;
}

function addEvent(element, eventType, eventAction)
{
  if (typeof element.addEventListener != 'undefined')
  {
    element.addEventListener(eventType, eventAction, false);
  }
  else if (typeof element.attachEvent != 'undefined')
  {
    var closure = function ()
    {
      eventAction.call(element, window.event);
    }
    eventType = ('on' + eventType);
    element.attachEvent(eventType, closure);
  }
  else
  {
    eventType = ('on' + eventType);
    if (typeof element[eventType] != 'undefined')
    {
      var oldEventAction = element[eventType];
      var closure = function ()
      {
        oldEventAction();
        eventAction();
      }
      element[eventType] = closure;
    }
    else
    {
      element[eventType] = eventAction;
    }
  }
  return element;
}

function stopEventPropagation(event)
{
  if (typeof event.stopPropagation != 'undefined')
    event.stopPropagation;
  else
    event.cancelBubble = true;
  return event;
}

function preventEventDefault(event)
{
  if (typeof event.preventDefault != 'undefined')
    event.preventDefault();
  else
    event.returnValue = false;
  return event;
}

function endEvent(event)
{
  stopEventPropagation(event);
  preventEventDefault(event);
  return event;
}

function eventSource(event)
{
  var source = document; // default to document if no source found
	if (typeof event.target != 'undefined')
    source = event.target;
  else if (typeof event.srcElement != 'undefined')
    source = event.srcElement;
  if (source.nodeType == 3) // if this element is a text node, grab its parent
    source = source.parentNode;
  return source;
}

function eventKey(event)
{
  var key = 0;
  if (typeof event.keyCode != 'undefined' &&
      event.keyCode != 0)
    key = event.keyCode;
  else if (typeof event.charCode != 'undefiend' &&
           event.charCode != 0)
    key = event.charCode;
  else if (typeof event.which != 'undefined' &&
           event.which != 0)
    key = event.which;
  return key;
}

/*************************************************
*  Checks to see if the Array contains the given *
* needle. If needle is undefined, it defaults to *
* equal "". Return true if found, false otherwise*
*************************************************/

Array.prototype.contains = function(needle)
{
  if (needle == undefined)
    needle = "";
  var returnVal = false;
  var i;
  for(i=0; i<this.length && returnVal == false; i++)
  {
    if (this[i] == needle)
      returnVal = true;
  }
  return returnVal;
}

/*************************************************
*  Removes the first occurance of value in the   *
* array. If the value is not found, no action is *
* taken. Returns the new length of the array.    *
*************************************************/

Array.prototype.remove = function(value)
{
  if (value == undefined)
    return this.length;
  var valueDeleted = false;
  var i;
  for(i=0; i<this.length && valueDeleted == false; i++)
  {
    if (this[i] == value)
    {
      this.splice(i,1);
      valueDeleted = true;
    }
  }
  return this.length;
}

/*************************************************
*  Removes all occurances of value in the        *
* array. If the value is not found, no action is *
* taken. Returns the new length of the array.    *
*************************************************/

Array.prototype.removeAll = function(value)
{
  if (value == undefined)
    return this.length;
  var i;
  for (i=0;i<this.length;i++)
  {
    if (this[i] == value)
    {
      this.splice(i,1);
      i--;
    }
  }
  return this.length;
}

/*************************************************
*  Inserts value at the specified index. If value*
* is undefined, no action is taken. If index is  *
* undefined, value is pushed into the array. If  *
* index is greater than the length of the array, *
* value is added at index and the array is filled*
* with undefined's up to index. If index is less *
* than 0, value is placed at the beginning of the*
* array. The array's length is returned.         *
*************************************************/

Array.prototype.insertAt = function(value, index)
{
  if (value == undefined)
    return this.length;
  if (index == undefined)
    return this.push(value);
  if (index > this.length-1)
  {
    this[index] = value;
    return this.length;
  }
  if (index < 0)
    index = 0;
  this.splice(index,0,value);
  return this.length;
}

/*************************************************
*  Removes all undefined values from the array   *
* and returns the new length.                    *
*************************************************/

Array.prototype.compact = function()
{
  var i;
  for(i=0; i<this.length; i++)
  {
    if (this[i] == undefined)
    {
      this.splice(i,1);
      i--;
    }
  }
  return this.length;
}

/*************************************************
*  Returns the last index number of the array,   *
* this is equivilant to array.length-1.          *
*************************************************/

Array.prototype.lastIndex = function()
{
  if (this.length <= 0)
    return false;
  return this.length-1;
}
 
/*************************************************
*  Returns the last element of the Array. This is*
* equivilant to array[array.length-1]. If the    *
* array is empty, returns false.                 *
*************************************************/

Array.prototype.last = function()
{
  if (this.length <= 0)
    return false;
  return this[this.lastIndex()];
}

/*************************************************
*  Creates an Object specifying only the array   *
* index values, for use in for(in) logic. Ex:    *
* for(var val in array.iterator())               *
* {                                              *
*   alert(array[val]);                           *
* }                                              *
* This avoids the array presenting custom methods*
* in a for(in) logic.                            *
*************************************************/

Array.prototype.iterator = function()
{
  var iterator = {}
  for(var i=0; i<this.length; i++)
    iterator[i] = i;
  return iterator;
}

/*************************************************
*  Creates a deep copy of the array and returns  *
* it. Called recursively if needed               *
*************************************************/

Array.prototype.clone = function()
{
  var copy = [];
  var numberOfItems = this.length;
  for (var i=0; i<numberOfItems; i++)
  {
    var item = this[i];
    if (item.clone != undefined)
      copy[i] = item.clone();
    else
      copy[i] = item;
  }
  return copy;
}
