/********************************************************************
//        NAME- ProgressBar
// DESCRIPTION- Constructor for the ProgressBar object
//      PASSED- element (HTMLElement / string): optional argument,
//                  if provided, the main element of the progress bar
//                  will be set to this, otherwise a blank div will
//                  be created to serve as the main element
//     RETURNS- 
//       NOTES- The ProgressBar class controls the appearance and 
//              adjustment of a progress bar on screen
********************************************************************/
function ProgressBar(element)
{
  this.percent = 0;
  
  if (element == undefined)
    this.element = newElement('div');
  else if (isElement(element))
    this.element = element;
  else if (typeof element == 'string')
    this.element = getElement(element);
  else
    throw('ProgressBar: invalid element - '+element);

  this.stretcher = this.element.appendChild(newElement('div', {'class':'stretcher'}));
  this.indicator = this.stretcher.appendChild(newElement('span', {'class':'indicator', 'text':'0%'}));

  this.init();
}

/********************************************************************
//        NAME- ProgressBar.prototype.init
// DESCRIPTION- Initializes the element of the progress bar
//      PASSED- 
//     RETURNS- 
//       NOTES- 
********************************************************************/
ProgressBar.prototype.init = function init()
{
  addClass(this.element, 'progressBar');
  this.element.controller = this;
  this.element.setPercent = function (percent)
  {
    return this.controller.setPercent(percent)
  };
}

/********************************************************************
//        NAME- ProgressBar.prototype.setPercent
// DESCRIPTION- Sets the percentage of the progress bar
//      PASSED- percent (number): the percentage to set the progress
//                  bar to. Should be an integer between 0 and 100
//     RETURNS- 
//       NOTES- percent will be trimmed to an int between 0 and 100
//              if necessary. Upon 100%, progress bar adds
//              'ui_progressBarFinished' to the stretcher's className
********************************************************************/
ProgressBar.prototype.setPercent = function setPercent(percent, className)
{
  if (percent != Number(percent))
    throw('ProgressBar: invalid percent in ProgressBar.setPercent() - '+percent);
  if (percent == undefined || percent == '' || percent < 0)
    percent = 0;
  if (percent > 100)
    percent = 100;
  className = className || 'finished';
  this.percent = Math.round(percent);
  clearChildren(this.indicator);
  this.indicator.appendChild(newText(this.percent+'%'));
  this.stretcher.style.width = this.percent+'%';
  this.stretcher.className = 'stretcher'; // reset className of stretcher
  if (percent == 100)
    addClass(this.stretcher, className);
  else
    removeClass(this.stretcher, className);
  this.stretcher.cloak();
  this.stretcher.uncloak();
  return this;
}

/********************************************************************
//        NAME- ProgressBar.prototype.draw
// DESCRIPTION- attaches the element to another element
//      PASSED- canvas (HTMLElement): the element to attach the
//                  progress bar to
//     RETURNS- 
//       NOTES- using draw() is only necessary when no element was
//              specified in the constructor and should only be used
//              once, when the bar is not yet attached to the DOM
********************************************************************/
ProgressBar.prototype.draw = function draw(canvas)
{
  canvas.appendChild(this.element);
  return this;
}

ProgressBar.prototype.setClass = function setClass(className)
{
  this.stretcher.className = 'stretcher';
  addClass(this.stretcher, className);
}
