function StatusMonitor()
{
  this.testStatusURL = "views/testStatus.php";
  this.diagnosisResultURL = "views/diagnoseResult.php";
  
  this.state = StatusMonitor.States.INIT;
  
  this.progressBar = new ProgressBar("statusViewProgressBar");
  this.buttonDialog = new TestPromptButtonDialog(this);
  this.textDialog = new TestPromptTextDialog(this);
  
  this.viewController = new BufferedViewController("statusFrameA", "statusFrameB");
  
  this.captionElement = extendElement(getElement("statusCaption"));
  this.messageElement = extendElement(getElement("statusMessage"));
  this.detailsElement = extendElement(getElement("statusInfo"));
  this.buttonElement = extendElement(getElement("cancelRetestButton"));
  this.buttonElement.controller = this;
  this.buttonElement.cloak();
  
  this.resultRequest = new ServerRequest("commands/getProcessResult.php");
  this.cancelRequest = new ServerRequest("commands/cancelProcess.php");

  this.pollingTime = 2000;  
  this.diagnosisMode = false;
  this.lastCaller = null;
  this.currentPID = 0;
  this.queuedTest = null;
  
  this.runningStates = ["running", "canceling"];
  
  this.debugging = false;
  
  this.init();
}

StatusMonitor.prototype.init = function init()
{
  var that = this;
  var loaded = function (event)
  {
    that.frameLoaded(this);
  }
  addEvent(this.viewController.frameA, "load", loaded);
  addEvent(this.viewController.frameB, "load", loaded);
  
//  this.viewController.clearParameters();
//  this.viewController.setURL(this.testStatusURL);
  this.resize();
  this.setState(StatusMonitor.States.IDLE);
}

StatusMonitor.prototype.stateString = function stateString()
{
  for (var name in StatusMonitor.States)
  {
    if (this.state == StatusMonitor.States[name])
      return name;
  }
  return "";
}

StatusMonitor.prototype.resize = function resize(frame)
{
  frame = frame || this.viewController.currentFrame;
  Body.adjustToViewport(frame);
}

StatusMonitor.prototype.frameLoaded = function frameLoaded(frame)
{
  this.log("frameLoaded: " + frame.id + " " + this.stateString());
  var States = StatusMonitor.States;
  switch (this.state)
  {
    case States.INIT:
      this.log("setting state to IDLE");
      this.setState(StatusMonitor.States.IDLE);
//      this.viewController.switchFrames();
      this.resize();
      if (this.queuedTest != null) // we had a startMonitoring request before we got out of INIT
      {
        var test = this.queuedTest;
        this.startMonitoring(test.pid, test.caller, test.diagnosis);
        this.queuedTest = null;
      }
    break;
    
    case States.STARTING:
    case States.RESTARTING:
      var info = frame.contentWindow.TEST_STATUS;
      if (info == undefined)
      {
        this.log("no info found, error");
        this.setState(StatusMonitor.States.IDLE);
        this.error(translate("MSG_COULD_NOT_START_TESTS"));
      }
      else
      {
        this.updateInfo(info);
        if (info.testCount <= 0)
        {
          if (this.runningStates.contains(info.state))
          {
            this.log("running, no tests found, probing");
            this.viewController.switchFrames();
            this.resize();
            this.probe();
          }
          else
          {
            this.log("not running, error");
            this.setState(StatusMonitor.States.IDLE);
            this.error(translate("MSG_COULD_NOT_START_TESTS"));
          }
        }
        else
        {
          this.log("running, tests found, beginning monitor loop");
          this.viewController.switchFrames();
          this.resize();
          this.beginMonitorLoop();
        }
      }
    break;
    
    case States.MONITORING:
    case States.CANCELING:
      var info = frame.contentWindow.TEST_STATUS;
      if (info.testCount <= 0)
      {
        this.log("no tests found, error");
        this.setState(StatusMonitor.States.IDLE);
        this.error(translate("MSG_LOST_CONTACT_WITH_TESTING_ENGINE"));
      }
      else
      {
        this.log("tests found");
        this.updateInfo(info);
        this.viewController.switchFrames();
        this.resize();
        if (this.runningStates.contains(info.state))
        {
          this.log("running");
          if (info["prompt"])
          {
            this.log("prompt found, prompting");
            this.promptUser(info["prompt"]);
          }
          else
          {
            this.log("resuming monitor loop");
            this.monitorLoop();
          }
        }
        else
        {
          this.log("no longer running, ending monitor loop");
          this.stopMonitoring(info);
        }
      }
    break;
    
    case States.FETCHING_DIAGNOSIS_RESULTS:
      this.viewController.switchFrames();
      this.resize();
      /* IE FIX */
      this.viewController.currentFrame.hide();
      this.viewController.currentFrame.show();
      /* END IE FIX */
      var info = frame.contentWindow.DIAGNOSE_STATUS || {
        failureCount : 0,
        warningCount : 0
      };
      if (info.failureCount > 0)
        this.progressBar.setClass("failed");
      else if (info.warningCount > 0)
        this.progressBar.setClass("warning");
      else
        this.progressBar.setClass("finished");
      this.setState(StatusMonitor.States.IDLE);
    break;
  }
}

StatusMonitor.prototype.startMonitoring = function startMonitoring(processID, caller, diagnosis)
{
  this.log("startMonitoring(" + processID + ", " + caller + ", " + diagnosis);
  this.log("current state: " + this.stateString());
  caller = caller || null;
  diagnosis = diagnosis || false;
  if (this.state == StatusMonitor.States.INIT)
  {
    this.queuedTest = {
      caller : caller,
      pid : processID,
      diagnosis : diagnosis
    };
    this.log("in INIT state, queuing test: " + this.queuedTest.pid);
  }
  else if (this.state == StatusMonitor.States.IDLE ||
           this.state == StatusMonitor.States.RESTARTING)
  {
    this.log("ready, starting test");
    this.diagnosisMode = diagnosis;
    this.lastCaller = caller;
    this.currentPID = processID;
    this.setState(StatusMonitor.States.STARTING);
    Frontend.tabs.selectTab("status");
    this.probe();
  }
}

StatusMonitor.prototype.stopMonitoring = function stopMonitoring(info)
{
  if (this.diagnosisMode)
  {
    this.setState(StatusMonitor.States.FETCHING_DIAGNOSIS_RESULTS);
    this.viewController.clearParameters();
    this.viewController.setParameter("processID", this.currentPID);
    this.viewController.setURL(this.diagnosisResultURL);
  }
  else
  {
    this.setState(StatusMonitor.States.IDLE);
    this.resultRequest.clearVariables();
    this.resultRequest.addVariable("processID", this.currentPID);
    if (CONFIG.SCRIPT_TESTING_PID != undefined &&
        (CONFIG.SCRIPT_TESTING_AUTO_CLOSE || false) === true)
    {
      this.resultRequest.callback = function ()
      {
        Frontend.closeWindow();
      }
    }
    else if (CONFIG.DISPLAY_ERROR_SLOTS == true)
    {
      this.resultRequest.callback = function ()
      {
        // options copied from old frontend implementation
        var windowOptions = {
          height      : Viewport.getHeight(),
          width       : Viewport.getWidth(),
          top         : 0,
          left        : 0,
          directories : "no", // no bookmarks / links bar
          hotkeys     : "no", // ?
          location    : "no", // no address bar
          menubar     : "no", // no menu bar
          resizable   : "no", // no resizing of new window
          status      : "no", // no status bar
          toolbar     : "no", // no navigation toolbar (back, reload, home, etc)
          scrollbars  : "yes" // allowing scrolling if needed
        }
        var optionsArray = []
        for (var name in windowOptions)
        {
          var value = windowOptions[name];
          optionsArray.push(name + "=" + value);
        }
        var optionsString = optionsArray.join(",");
        window.open(StatusMonitor.ErrorSlotScreenURL,
                    "slotdisplayWindow",
                    optionsString);
        Frontend.closeWindow();
      }
    }
    else
    {
      this.resultRequest.callback = null;
    }
    this.resultRequest.send();
  }
  if (info.failedCount > 0)
    this.progressBar.setClass("failed");
  this.diagnosisMode = false;
  this.currentPID = 0;
}

StatusMonitor.prototype.probe = function probe(extraParameters)
{
  this.log("probe");
  extraParameters = extraParameters || {};
  this.viewController.clearParameters();
  for (var key in extraParameters)
    this.viewController.setParameter(key, extraParameters[key]);
  this.viewController.setParameter("processID", this.currentPID);
  if (this.diagnosisMode)
    this.viewController.setParameter("diagnosisTests", "yes");
  if (CONFIG.SCRIPT_TESTING_PID != undefined)
    this.viewController.setParameter("scriptTesting", "yes");
  this.viewController.setURL(this.testStatusURL);
}

StatusMonitor.prototype.beginMonitorLoop = function beginMonitorLoop()
{
  if (this.state == StatusMonitor.States.STARTING)
  {
    this.setState(StatusMonitor.States.MONITORING);
    this.monitorLoop();
  }
}

StatusMonitor.prototype.monitorLoop = function monitorLoop()
{
  if (this.state == StatusMonitor.States.MONITORING ||
      this.state == StatusMonitor.States.CANCELING)
  {
    var that = this;
    setTimeout(function ()
    {
      that.probe();
    }, this.pollingTime);
  }
}

StatusMonitor.prototype.restart = function restart()
{
  if (this.state == StatusMonitor.States.IDLE)
  {
    this.setState(StatusMonitor.States.RESTARTING);
    this.lastCaller.request.send();
  }
}

StatusMonitor.prototype.cancel = function cancel()
{
  if (this.state.active == true &&
      this.state != StatusMonitor.States.CANCELING &&
      this.currentPID > 0)
  {
    this.cancelRequest.clearVariables();
    this.cancelRequest.addVariable("processID", this.currentPID);
    this.cancelRequest.send();
    this.setState(StatusMonitor.States.CANCELING);    
  }
}

StatusMonitor.prototype.updateInfo = function updateInfo(info)
{
  this.captionElement.innerHTML = info.caption;
  this.detailsElement.innerHTML = info.details;
  this.messageElement.innerHTML = info.message;
  this.progressBar.setPercent(info.percentComplete);
}

StatusMonitor.prototype.promptUser = function promptUser(data)
{
  this.setState(StatusMonitor.States.PROMPTING);
  var ID = data.ID;
  var type = data.type;
  var caption = data.caption;
  var content = data.content;
  var options = data.options;
  switch (type)
  {
    case "button":
      var dialog = this.buttonDialog;
      dialog.promptID = ID;
      dialog.setHeader(caption);
      dialog.setContent(content);
      dialog.setOptions(options);
    break;
    
    case "text":
      var dialog = this.textDialog;
      dialog.promptID = ID;
      dialog.setHeader(caption);
      dialog.setContent(content);
    break;
  }
  dialog.show();
}

StatusMonitor.prototype.replyToPrompt = function replyToPrompt(promptID, value)
{
  this.setState(StatusMonitor.States.MONITORING);
  var parameters = {
    promptID : promptID,
    promptResponse : value
  };
  this.probe(parameters);
}

StatusMonitor.prototype.setState = function setState(state)
{
  var states = StatusMonitor.States;
  this.log("setState: " + this.state.name + " => " + state.name);
  switch (state)
  {
    case states.IDLE:
//      var lastState = this.state;
      this.state = state;
      this.enableTabs();
      if (this.lastCaller == null)
      {
        this.hideButton();
      }
      else
      {
        this.retestButton();
      }
      /*
      if (lastState == states.INIT) // just switched out of init for the first time
      {
        if (this.queuedTest != null) // we had a startMonitoring request before we got out of INIT
        {
          var test = this.queuedTest;
          this.startMonitoring(test.pid, test.caller, test.diagnosis);
          this.queuedTest = null;
        }
      }
      */
    break;
    
    case states.STARTING:
      this.state = state;
      this.disableTabs();
      this.cancelButton();
    break;
    
    case states.MONITORING:
      this.state = state;
      this.disableTabs();
      this.cancelButton();
    break;
    
    case states.PROMPTING:
      this.state = state;
      this.disableTabs();
      this.disableButton();
    break;
    
    case states.CANCELING:
      this.state = state;
      this.disableTabs();
      this.cancelingButton();
    break;
    
    case states.RESTARTING:
      this.state = state;
      this.disableTabs();
      this.retestingButton();
    break;
    
    case states.FETCHING_DIAGNOSIS_RESULTS:
      this.state = state;
      this.disableTabs();
      this.retestButton();
      this.disableButton();
    break;
  }
}

StatusMonitor.prototype.disableTabs = function disableTabs()
{
  Frontend.tabs.disableTab("diagnose");
  Frontend.tabs.disableTab("test");
  Frontend.tabs.helpTabs.disableTab("testComponent");
}

StatusMonitor.prototype.enableTabs = function enableTabs()
{
  Frontend.tabs.enableTab("diagnose");
  Frontend.tabs.enableTab("test");
  Frontend.tabs.helpTabs.enableTab("testComponent");
}

StatusMonitor.prototype.disableButton = function disableButton()
{
  this.buttonElement.disable();
}

StatusMonitor.prototype.enableButton = function enableButton()
{
  this.buttonElement.enable();
}

StatusMonitor.prototype.cancelButton = function cancelButton()
{
  this.buttonElement.innerHTML = translate("MSG_CANCEL");
  this.buttonElement.action = function ()
  {
    this.controller.cancel();
  }
  this.enableButton();
  this.showButton();
}

StatusMonitor.prototype.cancelingButton = function cancelingButton()
{
  this.cancelButton();
  this.disableButton();
}

StatusMonitor.prototype.retestButton = function retestButton()
{
  this.buttonElement.innerHTML = translate("MSG_RETEST");
  this.buttonElement.action = function ()
  {
    this.controller.restart();
  }
  this.enableButton();
  this.showButton();
}

StatusMonitor.prototype.retestingButton = function retestingButton()
{
  this.retestButton();
  this.disableButton();
}

StatusMonitor.prototype.hideButton = function hideButton()
{
  this.buttonElement.cloak();
}

StatusMonitor.prototype.showButton = function showButton()
{
  this.buttonElement.uncloak();
}

StatusMonitor.prototype.error = function error(msg)
{
  Frontend.inform(msg, translate("MSG_HP_INSIGHT_DIAGNOSTICS"));
}

StatusMonitor.prototype.log = function log()
{
  if (this.debugging == true)
  {
    if ((console || {}).log)
      console.log.apply(console, arguments);
    else if ((Debugger || {}).log)
      Debugger.log.apply(Debugger, arguments);
  }
}

StatusMonitor.prototype.isIdle = function isIdle()
{
  var state = this.state;
  var states = StatusMonitor.States;
  return ( state == states.INIT || 
           state == states.IDLE );
}

StatusMonitor.ErrorSlotScreenURL = "slotdisplay.php";

StatusMonitor.States = {
  INIT : {
    name : "INIT",
    active : false
  },
  IDLE : {
    name : "IDLE",
    active : false
  },
  STARTING : {
    name : "STARTING",
    active : true
  },
  MONITORING : {
    name : "MONITORING",
    active : true
  },
  PROMPTING : {
    name : "PROMPTING",
    active : true
  },
  CANCELING : {
    name : "CANCELING",
    active : true
  },
  RESTARTING : {
    name : "RESTARTING",
    active : true
  },
  FETCHING_DIAGNOSIS_RESULTS : {
    name : "FETCHING_DIAGNOSIS_RESULTS",
    active : false
  }
}