#!/bin/sh
#
###############################################################################
##  (C) Copyright 2005,2009 Hewlett-Packard Development Company, L.P.
###############################################################################
#
HOMEDIR="/opt/hp/hp-OpenIPMI"
KVER="$(uname -r)"
KERNEL_MAJOR_VERSION=`echo $KVER | cut -b1,2,3`
DISTROID="$HOMEDIR/distro-id.sh"

if [ `expr ${KERNEL_MAJOR_VERSION} \>= 2.6` -eq 1 ];then
    MOD_EXT="ko"
else
    MOD_EXT="o"
fi

is_installed() {
    local pkg="$1"

    if rpm -q "$pkg" > /dev/null 2>&1; then
	echo "The package $pkg has been found"
	return 0
    else
	echo "The package $pkg is missing"
	return 1
    fi
}

###############################################################################
## ValidateKernel()
## This routine will check to see if the appropriate kernel environment is in
## place if we need to build.
###############################################################################
ValidateKernel() {
    # we need to do a test load because RHEL 3 has several
    # different kernels with the same uname -r and these
    # modules will not load on all of them
    if [ -d "$HOMEDIR/bin/$KVER" ]; then
        tmpfile="$(mktemp -t hp-OpenIPMI.XXXXXXXXXX)" || exit 1
        if ! insmod "$HOMEDIR/bin/$KVER/ipmi_msghandler.${MOD_EXT}" 2>"$tmpfile"; then
            if [ "`sed -e 's/.*\(Invalid module format\)/\1/' $tmpfile`" != \
		"Invalid module format" ]; then
                echo "There is an issue loading:  $HOMEDIR/bin/$KVER/ipmi_msghandler.${MOD_EXT} "
                echo "Attempting a rebuild of the hp-OpenIPMI driver . . ."
                rm -rf "$HOMEDIR/bin/$KVER"
            fi
        else
            rmmod ipmi_msghandler
        fi
        rm -f "$tmpfile"
    fi

    if [ ! -d "$HOMEDIR/bin/$KVER" ]; then
	echo "This kernel requires a rebuild.  Seeking source files"
	if [ -d "/lib/modules/$KVER/source" -a \
            ! -f "/lib/modules/$KVER/source/include/linux/ipmi.h" ]; then
	    echo "The file:  /lib/modules/$KVER/source/include/linux/ipmi.h"
	    echo "does not exist.  This is a strong indication that the"
	    echo "necessary Linux Kernel Development packages are not"
	    echo "installed."
	    return 2
	elif [ ! -d "/lib/modules/$KVER/build" ]; then
	    echo "The file:  /lib/modules/$KVER/build"
	    echo "does not exist.  This is a strong indication that the"
	    echo "necessary Linux Kernel Development packages are not"
	    echo "installed."
	    return 2
	else
	    echo "Basic source files appear to be installed.  Investigating further."
	    if [ ! -f "/lib/modules/$KVER/build/.config" ]; then
		echo "The file: /lib/modules/$KVER/build/.config"
		echo "does NOT exist.  There must be a Linux kernel"
		echo "'.config' file with IPMI enabled to build this"
		echo "package."
		return 2
	    fi
	    return 1
	fi
    fi

    return 0
}

#
##############################################################################
#                           ###### M A I N ######
##############################################################################
#

ValidateKernel

case $? in
    0)
	exit 0
	;;
    1)
	src_found=Y
	;;
    2)
	src_found=N
	;;
    *)
	src_found=N
	echo "Unknown return code from ValidateKernel"
	exit 1
	;;
esac

#
# If we get this far, then we need to perform a deeper search for
# the Linux kernel source files.  This is very dependent on the
# particular flavor of Linux loaded.
#

distro_name="$($DISTROID | grep '^name: ' | cut -d' ' -f2-)"
distro_major="$($DISTROID | grep '^major: ' | cut -d' ' -f2-)"

if [ "$distro_name" = "Red Hat" ]; then
    case "$distro_major" in
	"3")
	    echo "Checking Sources for Red Hat Enterprise Linux 3"
	    LINUX_BASE=`echo "$KVER" | sed -e 's/smp//g' -e 's/hugemem//g'`
	    RetCode=0
	    src_found=N
	    is_installed kernel-utils || exit 1
	    is_installed glibc-kernheaders || exit 1
	    is_installed "kernel-source-${LINUX_BASE}" || exit 1

	    src_found="Y"
	    ;;
	"4")
	    echo "Checking Sources for Red Hat Enterprise Linux 4"
	    LINUX_BASE=`echo "$KVER" | sed -e 's/smp//g' -e 's/hugemem//g'`
	    RetCode=0
	    src_found=N
	    is_installed kernel-utils || exit 1
	    is_installed glibc-kernheaders || exit 1
	    if echo "$KVER" | grep -q smp; then
		is_installed "kernel-smp-${LINUX_BASE}" || exit 1
		is_installed "kernel-smp-devel-${LINUX_BASE}" || exit 1
	    else
		RetCode=1
	    fi

	    if [ $RetCode -eq 1 ]; then
		RetCode=0
		if echo "$KVER" | grep -q hugemem; then
		    is_installed "kernel-hugemem-${LINUX_BASE}" || exit 1
		    is_installed "kernel-hugemem-devel-${LINUX_BASE}" || exit 1
		else
		    RetCode=1
		fi
	    fi

	    if [ $RetCode -eq 1 ]; then
		RetCode=0
		is_installed "kernel-devel-${LINUX_BASE}" || exit 1
	    fi
	    
	    src_found="Y"
	    ;;
    esac
elif [ "$distro_name" = "SuSE Linux" ]; then
    case "$distro_major" in
	"9")
	    echo "Checking Sources for SUSE LINUX Enterprise Server 9"
	    LINUX_BASE=`echo "$KVER" | sed -e 's/-smp//g' -e 's/-default//g' -e 's/-bigsmp//g'`
	    RetCode=0
	    src_found=N
	    is_installed "kernel-syms-${LINUX_BASE}" || exit 1
	    is_installed "kernel-source-${LINUX_BASE}" || exit 1
	
	    src_found="Y"
	    ;;
	"10"|"11")
	    echo "Checking Sources for SUSE Linux Enterprise Server $distro_major"
	    LINUX_BASE=`echo "$KVER" | sed -e 's/-smp//g' -e 's/-default//g' -e 's/-bigsmp//g' -e 's/-xenpae//g' -e 's/-xen//g'`
	    RetCode=0
	    src_found=N

	    if [ -e /lib/modules/$KVER/source ]; then
		src_found="Y"
	    else
		echo "Warning: kernel source not found." >&2
		echo "You may need to install the kernel-source-${LINUX_BASE} RPM." >&2
	    fi
	    ;;
    esac
elif [ "$distro_name" = "VMware ESX" ] && [ "$distro_major" = "3" ]; then
	echo "Checking Sources for VMware ESX 3.x"
	RetCode=1
	src_found=N
	if [ ! -f "/lib/modules/$KVER/build/Makefile" ]; then
	    echo "There does not appear to be source files availableto rebuild"
	elif [ ! -f "/lib/modules/$KVER/build/.config" ]; then
	    echo "No '.config' file available. Seeking vmnix.config"
	    if [ -f "/lib/modules/$KVER/build/configs/vmnix.config" ]; then
		echo " /lib/modules/$KVER/build/configs/vmnix.config found."
		echo "Using as the default kernel configuration file."
		cp -p "/lib/modules/$KVER/build/configs/vmnix.config" \
		    "/lib/modules/$KVER/build/.config"
		RetCode=0
		src_found="Y"
	    fi
	else
	    RetCode=0
	    src_found="Y"
	fi

	if [ $RetCode -eq 1 ]; then
	    exit 1;
	fi
else
	if [ "$src_found" = "N" ]; then
	    echo "Unknown Linux Distribution - No Source to build"
	    exit 1
	else
	    echo "Unknown Linux Distribution - will attempt to build"
	    echo "The hp-OpenIPMI(4) and hpasm(4) man pages have additional"
            echo "information to resolve this issue. The HP Advanced System"
            echo "Management software is currently disabled on this system."
	fi
fi

if [ "$src_found" != "N" ]; then
    cd "$HOMEDIR"
    make clean
    make
    if [ $? != 0 ]; then
	echo ""
	echo "WARNING:  Build did not complete successfully!"
	echo ""
	exit 1
    fi
fi
