#!/var/python/bin/python
"""
Copyright 2000-2022 Citrix Systems, Inc. All rights reserved.
This software and documentation contain valuable trade
secrets and proprietary property belonging to Citrix Systems, Inc.
None of this software and documentation may be copied,
duplicated or disclosed without the express
written permission of Citrix Systems, Inc.
"""

from abc import ABCMeta, abstractmethod
import base64
import os

import libnitrocli

import adal
from msrestazure.azure_cloud import AZURE_PUBLIC_CLOUD
from msrestazure.azure_active_directory import AdalAuthentication

from util import info, error, fail, debug, get_net_mask

AZURE_APP_CONFIG_TARGET = '/nsconfig/.AZURE/cred'


def decorate_configure(func):
    def configure_full(self):
        if not self.sub_config:
            debug("%s not present." % self.__CONFIG_DESC__)
            return
        info("configuring %s" % self.__CONFIG_DESC__)
        try:
            func(self)
        except Exception as err:
            error("exception occurred: %s" % str(err))
        info("%s configuration finished" % self.__CONFIG_DESC__)
    return configure_full


class _Configurator(object, metaclass=ABCMeta):

    __CONFIG_KEY__ = ""
    __CONFIG_DESC__ = ""

    def __init__(self, config_dict):
        self.nitrocli = libnitrocli.nitro_cli()
        self.parser = libnitrocli.nitro_cli_output_parser()
        self.sub_config = config_dict.get(self.__CONFIG_KEY__, {})

    def _save_config(self):
        self.nitrocli.save_nsconfig()

    @abstractmethod
    def configure(self):
        pass


class HAConfigurator(_Configurator):

    __CONFIG_KEY__ = "ha_config"
    __CONFIG_DESC__ = "HA Pair"

    def _configure_peer_node(self):
        if "peer_node" in self.sub_config:
            debug(
                "configuring peer node NSIP -> '%s'" %
                self.sub_config["peer_node"])
            out = self.nitrocli.add_hanode(
                self.sub_config["peer_node"], inc="enabled")
            if not self.parser.success(out):
                fail("ha node configuration failed with output: %s" % out)

    @decorate_configure
    def configure(self):
        self._configure_peer_node()
        self._save_config()


class AzureAppConfigurator(_Configurator):

    __CONFIG_KEY__ = "azure_app_config"
    __CONFIG_DESC__ = "Azure App Credential"

    def _adal_authentication(self):
        login_endpoint = AZURE_PUBLIC_CLOUD.endpoints.active_directory
        resource = AZURE_PUBLIC_CLOUD.endpoints.active_directory_resource_id
        auth_uri = login_endpoint + '/' + self.sub_config["tenant_id"]
        try:
            context = adal.AuthenticationContext(auth_uri)
            context.acquire_token_with_client_credentials(resource,
                                                          self.sub_config["app_id"],
                                                          self.sub_config["secret_key"])
            return True
        except adal.adal_error.AdalError:
            debug_out = self.nitrocli.get_service()
            debug(
                "DNS service info (show service): %s" %
                debug_out.get(
                    "service", []))
            fail('Azure app authentication failed. Ignoring Keys')
        return False

    def _write_app_config(self):
        if os.path.isfile(AZURE_APP_CONFIG_TARGET):
            fail("cred file already exists. Skipped overwrite.")
        data = [base64.b64encode(self.sub_config["tenant_id"].encode()),
                base64.b64encode(self.sub_config["app_id"].encode()),
                base64.b64encode(self.sub_config["secret_key"].encode())]
        with open(AZURE_APP_CONFIG_TARGET, 'wb') as f_b:
            f_b.write(b"\n".join(data))

    @decorate_configure
    def configure(self):
        if self. _adal_authentication():
            self._write_app_config()


class VpxConfigurator(_Configurator):

    __CONFIG_KEY__ = "vpx_config"
    __CONFIG_DESC__ = "VPX generic"

    def _configure_ip(self, pvt_ip, subnet, ip_type=None):
        cidr = subnet.split('/')[-1]
        mask = get_net_mask(cidr)
        debug("configuring %s -> '%s/%s'" %
              (ip_type if ip_type else "snip", pvt_ip, mask))
        if ip_type:
            out = self.nitrocli.add_nsip(pvt_ip, mask, ip_type)
        else:
            out = self.nitrocli.add_nsip(pvt_ip, mask)
        if not self.parser.success(out):
            fail("ip configuration failed with output: %s" % out)

    def _configure_feature(self):
        if "ns_features" in self.sub_config:
            debug(
                "configuring features -> '%s'" %
                self.sub_config["ns_features"])
            out = self.nitrocli.enable_nsfeature(
                self.sub_config["ns_features"])
            if not self.parser.success(out):
                fail("feature configuration failed with output: %s" % out)

    def _configure_mode(self):
        if "ns_modes" in self.sub_config:
            debug(
                "configuring modes -> '%s'" %
                self.sub_config["ns_modes"])
            try:
                for mode,val in list(self.sub_config["ns_modes"].items()):
                    if "True" in val:
                        out = self.nitrocli.enable_nsmode(mode)
                    elif "False" in val:
                        out = self.nitrocli.disable_nsmode(mode)
                    if not self.parser.success(out):
                        fail("feature configuration failed for mode: %s" % mode)
            except Exception as err:
                error("exception occurred in _configure_mode: %s" % str(err))

    def _configure_snip(self, key, subnet, ip_type=None):
        if key in self.sub_config and subnet in self.sub_config:
            self._configure_ip(
                self.sub_config[key],
                self.sub_config[subnet],
                ip_type)

    def _configure_snips(self):
        # Default expected keys (To be depecreted in future)
        self._configure_snip("pvt_ip_11", "subnet_11", "vip")
        self._configure_snip("snip_11", "subnet_11")
        self._configure_snip("pvt_ip_12", "subnet_12")

        # List of IPs from 'snips' List
        if "snips" in self.sub_config:
            for snip in self.sub_config["snips"]:
                if "ip" in snip and "subnet" in snip:
                    snip_ip = snip["ip"]
                    snip_subnet = snip["subnet"]
                    snip_type = None
                    if "type" in snip:
                        snip_type = snip["type"]
                    self._configure_ip(snip_ip, snip_subnet, snip_type)

    @decorate_configure
    def configure(self):
        self._configure_snips()
        self._configure_feature()
        self._configure_mode()
        self._save_config()
