#!/usr/bin/python
###############################################################################
#
#  Copyright (c) 2019-2024 Citrix Systems, Inc.
#  All rights reserved.
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions are met:
#      * Redistributions of source code must retain the above copyright
#        notice, this list of conditions and the following disclaimer.
#      * Redistributions in binary form must reproduce the above copyright
#        notice, this list of conditions and the following disclaimer in the
#        documentation and/or other materials provided with the distribution.
#      * Neither the name of the Citrix Systems, Inc. nor the
#        names of its contributors may be used to endorse or promote products
#        derived from this software without specific prior written permission.
#
#  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
#  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
#  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
#  ARE DISCLAIMED. IN NO EVENT SHALL CITRIX SYSTEMS, INC. BE LIABLE FOR ANY
#  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
#  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
#  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
#  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
#  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
#  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
################################################################################

"""
Utility functions for AWS
"""

import enum
import json
import logging
import os
import os.path
import signal
import time
import sys

import boto3
import requests
import subprocess
import shlex
from botocore.client import Config
from libnitrocli import nitro_cli, nitro_cli_output_parser
from netaddr import IPNetwork, IPAddress
from src.ha_state import HaStatus
from src.ns_ha import NS

SHARED_VPC_IAM_ROLE = "/flash/nsconfig/.AWS/shared_vpc_role_arn"
IAM_NOT_OK_FILE = '/flash/nsconfig/.AWS/iam_prev_not_ok'
C2S_EC2_ENDPOINT = 'https://ec2.us-iso-east-1.c2s.ic.gov'
CLOUDADAPTER_PID_FILE = "/nsconfig/.AWS/cloudadapter.pid"
RAIN_SCALE_PID_FILE = "/nsconfig/.AWS/rain_scale.pid"
RAIN_STAT_PID_FILE = "/nsconfig/.AWS/rain_stats.pid"
CLOUDADAPTER_BIN = sys.executable.rsplit('/', 1)[0] + "/cloudadapter"
RAIN_SCALE_BIN = sys.executable.rsplit('/', 1)[0] + "/rain_scale"
RAIN_STAT_BIN = sys.executable.rsplit('/', 1)[0] + "/rain_stats"
SC2S_EC2_ENDPOINT = 'https://ec2.us-isob-east-1.sc2s.sgov.gov/'

""" Defining metadata server ip common part """
METADATA_URL = "http://169.254.169.254/latest"


class NetworkModeType(enum.Enum):
    """ Defines various HA Modes """
    AWS_HA_MODE_UNKNOWN = -1
    AWS_HA_MODE_STANDALONE = 0
    AWS_HA_MODE_ENI_TRANSFER = 1
    AWS_HA_MODE_PIP_TRANSFER = 2
    AWS_HA_MODE_EIP_TRANSFER = 3
    AWS_HA_MODE_CLUSTER_TRANSFER = 4
    AWS_HA_MODE_MZPIP_TRANSFER = 5


IAM_PRIVILEGES = {
    NetworkModeType.AWS_HA_MODE_ENI_TRANSFER:
        [
            "ec2:DescribeInstances",
            "ec2:DescribeNetworkInterfaces",
            "ec2:AttachNetworkInterface",
            "ec2:DetachNetworkInterface",
            "ec2:StopInstances",
            "ec2:StartInstances",
            "ec2:RebootInstances"
        ],
    NetworkModeType.AWS_HA_MODE_PIP_TRANSFER:
        [
            "ec2:DescribeInstances",
            "ec2:AssignPrivateIpAddresses",
        ],
    NetworkModeType.AWS_HA_MODE_CLUSTER_TRANSFER:
        [
            "ec2:AssignPrivateIpAddresses",
        ],
    NetworkModeType.AWS_HA_MODE_EIP_TRANSFER:
        [
            "ec2:DescribeInstances",
            "ec2:DescribeAddresses",
            "ec2:AssociateAddress",
            "ec2:DisassociateAddress"
        ],
    NetworkModeType.AWS_HA_MODE_MZPIP_TRANSFER:
        [
            "ec2:DescribeInstances",
            "ec2:DescribeRouteTables",
            "ec2:DeleteRoute",
            "ec2:CreateRoute",
            "ec2:ModifyNetworkInterfaceAttribute"
        ]
}


class Virtualserver:
    """ class to store vserver details """

    def __init__(self, own_ip, peer_ip, eni_id, eip_alloc_id, name=""):
        self.own_ip = own_ip
        self.peer_ip = peer_ip
        self.eni_id = eni_id
        self.eip_alloc_id = eip_alloc_id
        self.name = name


class AwsInstanceInfo:
    """ Contains Clould specific API. Must implement
        update_ha_info ,handle_failover and __init__"""

    def __init__(self, ns_obj):
        self.peer_ip = ns_obj.peer_ip
        self.inc_mode = ns_obj.inc_mode
        self.state = ns_obj.state
        self.powerOnPeer = False
        self.ipsetbinding = None
        self.eips = None
        self.interfacemetadata = None
        self.peer_interfaces = []
        self.my_interfaces = []
        self.peer_instance_id = None
        self.iam_role = self.__get_iam_info()
        self.retry_config = None
        self.vpc_id = self.__get_vpc_id()
        self.region = self.__get_region()[:-1]
        self.instance_id = self.__get_instance_id()
        self.__set_api_retry_setting()
        self.prev_interfaces = None
        self.instance_details_acquired_once = False
        if self.region.startswith("us-isob-"):
            logging.debug('Initializing the Secret Region Boto3 in  %s', self.region)
            self.client = boto3.client('ec2', region_name=self.region,
                                       endpoint_url=SC2S_EC2_ENDPOINT,
                                       verify=False,
                                       config=self.retry_config)
        elif self.region.startswith("us-iso-"):
            logging.debug('Initializing the Top Secret Region Boto3 in  %s', self.region)
            self.client = boto3.client('ec2', region_name=self.region,
                                       endpoint_url=C2S_EC2_ENDPOINT,
                                       verify=False,
                                       config=self.retry_config)
        else:
            logging.debug('Initializing the region boto3 %s', self.region)
            self.client = boto3.client('ec2', region_name=self.region,
                                       config=self.retry_config)

        if self.state not in [HaStatus.NON_CCO]:
            self.__get_my_info()

        self.network_mode = self.__get_network_mode_type()
        logging.debug('Network mode identified as : %s',self.network_mode)
        try:
            self.__nitro = nitro_cli()
            self.__parser = nitro_cli_output_parser()
        except:
            self.__nitro = None
            self.__parser = None
            logging.error('Nitro cli config failed')

    def __get_token(self):
        """ This function generate a token for metadata call """
        headers = {'X-aws-ec2-metadata-token-ttl-seconds': '600',}
        req_ip = METADATA_URL + '/api/token'
        response = requests.put(req_ip , headers=headers)
        return response.text

    def __get_meta_data_v2(self, fields):
        """ This function gives metadata v2 output
            given proper input field """
        url = METADATA_URL + '/meta-data' + fields
        token = self.__get_token()
        resp = requests.get(url, headers={"X-aws-ec2-metadata-token":token}, timeout=10)
        if resp.status_code != 200:
            logging.error("Meta-data server 169.254.169.254 returned error code : %s", resp.status_code)
            return 0
        return resp.text

    def __get_vpc_id(self):
        """ This function gives the vpc_id of
            managment interface as output """
        macs = self.__get_meta_data_v2('/network/interfaces/macs')
        mac = macs.split('\n')
        fields = '/network/interfaces/macs/' +mac[0] +'vpc-id'
        vpc_id = self.__get_meta_data_v2(fields)
        return vpc_id

    def __init_client_for_shared_iam(self):
        try:
            if os.path.exists(SHARED_VPC_IAM_ROLE):
                with open(SHARED_VPC_IAM_ROLE, 'r') as arn_file:
                    role_arn = arn_file.read().strip()
                    temp_client = boto3.client("sts",
                                               region_name=self.region,
                                               config=self.retry_config)
                    credentials = temp_client.assume_role(
                        RoleArn=role_arn,
                        RoleSessionName="testSession")['Credentials']
                    self.client_routes = boto3.client("ec2",
                                                      aws_access_key_id=credentials['AccessKeyId'],
                                                      aws_secret_access_key=credentials['SecretAccessKey'],
                                                      aws_session_token=credentials['SessionToken'],
                                                      region_name=self.region, config=self.retry_config)
            else:
                self.client_routes = self.client;
        except Exception as e:
            logging.error("Shared IAM role failed to be assumed error: %s ", str(e))
            self.client_routes = self.client;

    def __get_region(self):
        """ This function gives the region of the instance """
        region = self.__get_meta_data_v2('/placement/availability-zone')
        return region

    def __get_instance_id(self):
        """ This function gives instance id as output """
        instance_id = self.__get_meta_data_v2('/instance-id')
        return instance_id

    def __get_iam_info(self):
        """ This function gives iam info """
        iam = self.__get_meta_data_v2('/iam/security-credentials')
        return iam

    def __set_api_retry_setting(self):
        """ retry configuration for API calls """
        self.retry_config = Config(connect_timeout=20,
                                   retries={'max_attempts': 3})


    def __check_privileges(self, feature):
        """ Verifies IAM previleges """
        if self.region.startswith("us-iso-") or self.region.startswith("us-isob-"):
            logging.info("IAM Check is not supported in region : %s",self.region)
            return 0

        if feature in IAM_PRIVILEGES:
            return self.__check_iam(IAM_PRIVILEGES[feature])
        else:
            return ""


    def __check_iam(self, actions):
        """ Checks IAM for the instance """
        client_iam = boto3.client("iam", region_name=self.region, config=self.retry_config)
        role_name = client_iam.get_role(RoleName=self.iam_role)["Role"]["Arn"]
        result = client_iam.simulate_principal_policy(PolicySourceArn=role_name,
                                                      ActionNames=actions)
        if result["ResponseMetadata"]["HTTPStatusCode"] != 200:
            logging.error("Call to IAM simulate_principal_policy failed ")
            return actions
        iam_missing = ""
        for i in result["EvaluationResults"]:
            if i["EvalDecision"] != "allowed":
                iam_missing += (i['EvalActionName'] + ", ")
                logging.error("IAM permission not configured : %s",i['EvalActionName'])
        return iam_missing

    def __get_my_info(self):
        """ Get the describe instance output for  self instance"""
        try:
            response = self.client.describe_instances(InstanceIds=[self.instance_id])
            self.my_interfaces = response['Reservations'][0]['Instances'][0]['NetworkInterfaces']
        except Exception as e:
            logging.error("Unable to get my instance details, \
                Check Internet connectvity !!!")
            logging.info("Boto3 API failed with exception %s", str(e))
        return

    def __get_peer_info(self, peer_ip):
        """ Get the primary Private IPs for peer instance"""
        try:
            response = self.client.describe_instances(Filters=[
                {'Name': 'private-ip-address', 'Values': [peer_ip]},
                {'Name': 'vpc-id', 'Values': [self.vpc_id]}])
            self.peer_interfaces = response['Reservations'][0] \
                ['Instances'][0]['NetworkInterfaces']
            self.peer_instance_id = response["Reservations"][0] \
                ["Instances"][0]["InstanceId"]
        except Exception as e:
            logging.error("Unable to get peer instance details, \
                Check Internet connectvity !!!")
            logging.info("Boto3 API failed with exception %s", str(e))
        return


    def get_intf_privateip_map(self):
        """ return dictionary where key is interface id and value is
            associated privateip list """
        privateips = {}
        interface_ip_map = {}

        for interface in self.peer_interfaces:
            pipdictlist = interface['PrivateIpAddresses'][1:]
            if pipdictlist:
                piplist = [sub['PrivateIpAddress'] for sub in pipdictlist]
                if interface['SubnetId'] not in privateips:
                    privateips[interface['SubnetId']] = []
                privateips[interface['SubnetId']].append(piplist)
        logging.debug("Subnet IPs Map %s",privateips)

        for interface in self.my_interfaces:
            for subnetid in privateips:
                if subnetid == interface['SubnetId'] and privateips[subnetid]:
                    interface_ip_map[interface['NetworkInterfaceId']] = \
                        privateips[subnetid].pop(0)
        logging.debug('Interface ip map is :  %s',interface_ip_map)
        return interface_ip_map


    def get_intf_clip_map(self):
        """ Gets the interface to CLIP mapping in Cluster """
        clips = []
        interface_clip_map = {}
        privateipdict = self.__nitro.get_nsip()
        for ip in privateipdict['nsip']:
            if ip['type'] == 'CLIP':
                clips.append(ip['ipaddress'])

        for interface in self.my_interfaces:
            interface_clip_map[interface['NetworkInterfaceId']] = []
            mac = interface['MacAddress']
            fields = '/network/interfaces/macs/' + mac + '/subnet-ipv4-cidr-block'
            cidr_ipv4 = self.__get_meta_data_v2(fields)
            for clip in clips:
                if IPAddress(clip) in \
                        IPNetwork(cidr_ipv4):
                    interface_clip_map[interface['NetworkInterfaceId']].append(clip)
        logging.info('Interface clip map is : %s',interface_clip_map)
        return interface_clip_map


    def migrate_pip_from_peer(self, interface_pip_map):
        """ Migrate secondary private ips from new secondary to new primary """
        logging.info("Moving private ips from secondary to primary")
        logging.debug("interface pip map obtained is : %s",interface_pip_map)
        if self.region.startswith("us-isob-"):
            client_res = boto3.resource('ec2', region_name=self.region,
                                        endpoint_url=SC2S_EC2_ENDPOINT,
                                        verify=False,
                                        config=self.retry_config)
        elif self.region.startswith("us-iso-"):
            client_res = boto3.resource('ec2', region_name=self.region,
                                        endpoint_url=C2S_EC2_ENDPOINT,
                                        verify=False,
                                        config=self.retry_config)
        else:
            client_res = boto3.resource('ec2', region_name=self.region, config=self.retry_config)

        for interfaceid, iplist in iter(interface_pip_map.items()):
            if not iplist:
                continue
            network_interface = client_res.NetworkInterface(interfaceid)
            if self.state == HaStatus.PRIMARY:
                type = "HA"
            elif self.state == HaStatus.CCO:
                type = "CLIP"
            try:
                network_interface.assign_private_ip_addresses(AllowReassignment \
                                                                  =True, PrivateIpAddresses=iplist)
                logging.info("Private IP movement is successfull for interface : %s", interfaceid)
                ipstr = iplist[0]
                log_str = "Private IP movement is successfull for interface " + interfaceid + " and ip mapping is " + str(ipstr)
                out = self.__nitro.get_cloud_apistatus("AWSCLOUD", type)
                if self.__parser.success(out):
                    status_str = out[u'cloudapistatus'][0][u'status']
                    logging.info("Previous status of the CLI is " + status_str)
                    self.__nitro.set_cloud_apistatus("AWSCLOUD", type, "SUCCESS", log_str)
                    self.__nitro.rm_cloud_apistatus("AWSCLOUD", type)
                    logging.info("Changing status of the CLI to SUCCESS")
                else:
                    self.__nitro.add_cloud_apistatus("AWSCLOUD", type, "SUCCESS", log_str)
                    self.__nitro.rm_cloud_apistatus("AWSCLOUD", type)
                    logging.info("Adding status of the CLI to SUCCESS")
            except Exception as e:
                logging.error("Private IP movement failed for the interface\
                        %s", interfaceid)
                logging.info("Boto3 API failed with exception %s", str(e))
                ipstr = iplist[0]
                log_str = "Private IP movement failed for interface " + interfaceid + " and ip mapping is " + str(ipstr)
                out = self.__nitro.get_cloud_apistatus("AWSCLOUD", type)
                if self.__parser.success(out):
                    status_str = out[u'cloudapistatus'][0][u'status']
                    logging.info("Previous status of the CLI is " + status_str)
                    self.__nitro.set_cloud_apistatus("AWSCLOUD", type, "FAILURE", log_str)
                    logging.info("Changing status of the CLI to FAILURE")
                else:
                    self.__nitro.add_cloud_apistatus("AWSCLOUD", type, "FAILURE", log_str)
                    logging.info("Adding status of the CLI to FAILURE")

    def get_vserver_list(self):
        """ Fetches list of vservers with ipset configured """
        vserverlist = []
        vserveripdict = []
        if self.__nitro is None:
            self.__nitro = nitro_cli()
        try:
            vserveripdict = self.__nitro.get_lbvserver()['lbvserver']
            logging.debug("List of LB vserver identified are : %s", vserveripdict)
        except Exception as e:
            logging.info("No lbvserver configuration present, %s", str(e))
        try:
            csvserveripdict = self.__nitro.get_csvserver()['csvserver']
            logging.debug("List of CS vserver identified are : %s",csvserveripdict)
            vserveripdict = vserveripdict + csvserveripdict
        except Exception as e:
            logging.info("No csvserver configuration present, %s", str(e))
        try:
            vpnvserveripdict = self.__nitro.get_vpnvserver()['vpnvserver']
            logging.debug("List of SSL VPN vserver identified are : %s",vpnvserveripdict)
            vserveripdict = vserveripdict + vpnvserveripdict
        except Exception as e:
            logging.info("No vpnvserver configuration present, %s", str(e))
        logging.debug("Final ip dict after merging all is : %s ",vserveripdict)
        for vserver in vserveripdict:
            if 'ipset' in vserver.keys():
                ipsetip = self.get_ipset_ip(vserver['ipset'])
                if ipsetip is None:
                    logging.warning("ipsetip is none for %s", vserver['ipset'])
                    continue
                logging.info("ipset ip is : %s ", ipsetip)
                eni_id = self.get_eniid_for_ip(vserver['ipv46'])
                logging.debug("eni id is : %s ", eni_id)
                if eni_id:
                    eip_alloc_id = self.get_eipallocid_for_vip(ipsetip)
                    logging.debug("eip allocation id is : %s ", eip_alloc_id)
                    if eip_alloc_id:
                        vserverobj = Virtualserver(vserver['ipv46'], ipsetip,
                                                   eni_id, eip_alloc_id, vserver['name'])
                        vserverlist.append(vserverobj)
                else:
                    eni_id = self.get_eniid_for_ip(ipsetip)
                    if eni_id:
                        eip_alloc_id = self.get_eipallocid_for_vip(vserver['ipv46'])
                        logging.debug("eip allocation id in ipset loop is : %s ", eip_alloc_id)
                        if eip_alloc_id:
                            vserverobj = Virtualserver(ipsetip, vserver['ipv46'], eni_id,
                                                       eip_alloc_id, vserver['name'])
                            vserverlist.append(vserverobj)
        return vserverlist


    def get_eips(self):
        """ Fetches the Elastic IPs from AWS """
        instance_eips = self.client.describe_addresses(Filters=[\
                {'Name': 'instance-id', 'Values': [self.peer_instance_id]}])
        self.eips = instance_eips['Addresses']
        logging.debug("EIP list is : %s ", self.eips)


    def get_eipallocid_for_vip(self, private_ip):
        """ returns AllocationID for a private IP """
        for eip in self.eips:
            if 'PrivateIpAddress' in eip.keys() and eip['PrivateIpAddress'] == private_ip:
                return eip['AllocationId']
        return None


    def get_eniid_for_ip(self, ipaddr):
        """ checks and return eni-id if the given IP is own IP """
        for interface in self.my_interfaces:
            for ipinfo  in interface['PrivateIpAddresses']:
                if ipinfo['PrivateIpAddress'] == ipaddr:
                    return interface['NetworkInterfaceId']
        return None

    def get_ipsetbindings(self):
        """ gets the ipset binding configuration """
        if self.__nitro is None:
            self.__nitro = nitro_cli()
        ipsetconfig = self.__nitro.get_all_ipset_binding()
        logging.debug("All ipset binding : %s ", ipsetconfig)
        if 'ipset_binding' in ipsetconfig.keys():
            self.ipsetbinding = ipsetconfig['ipset_binding']
            logging.debug("Each ipset binding is  :%s ", self.ipsetbinding)
        else:
            self.ipsetbinding = None


    def get_ipset_ip(self, ipsetname):
        """ gets the list of IPs in an ipset """
        for ipset in self.ipsetbinding:
            if 'ipset_nsip_binding' in ipset and ipset['ipset_nsip_binding'][0]['name'] == ipsetname:
                logging.debug("List of ip in ipset : %s ", ipset['ipset_nsip_binding'][0]['ipaddress'])
                return ipset['ipset_nsip_binding'][0]['ipaddress']
        return None


    def migrate_eip_from_peer(self):
        """ Migrate all EIPs from new secondary to new primary """
        logging.info("Handling EIP HA Failover. Moving EIPs  from secondary to primary")
        self.get_eips()
        if not self.eips:
            logging.info("No EIPs to move")
            return
        self.get_ipsetbindings()
        vserverlist = self.get_vserver_list()
        logging.debug("vserver list is :%s ", vserverlist)
        for vserver in vserverlist:
            logging.info("Moving EIP for the vserver :%s", vserver.name)
            logging.debug("vserver EIP alloc id is :%s", vserver.eip_alloc_id)
            logging.debug("vserver ENI id is  :%s", vserver.eni_id)
            logging.debug("vserver ip address is  :%s", vserver.own_ip)
            try:
                self.client.associate_address(AllocationId=vserver.eip_alloc_id, NetworkInterfaceId=
                vserver.eni_id, PrivateIpAddress=vserver.own_ip, AllowReassociation=True)
                logging.info("Moving EIP for %s is successfull", vserver.own_ip)
            except Exception as e:
                logging.error("Moving EIP for %s is failed with exception %s",
                               vserver.own_ip, str(e))


    def do_sleep(self):
        time.sleep(600)

    def update_ha_info(self, ns_obj):
        """ Updates HA info """
        self.state = ns_obj.state
        self.peer_ip = ns_obj.peer_ip
        self.inc_mode = ns_obj.inc_mode
        self.iam_role = self.__get_iam_info()

        if self.state in [HaStatus.STANDALONE, HaStatus.PRIMARY, HaStatus.CCO]:
            self.__launch_aws_daemons()

        if self.state in [HaStatus.SECONDARY, HaStatus.NON_CCO]:
            self.__send_signal_to_aws_daemon()

        if self.state not in [HaStatus.NON_CCO]:
            self.__get_my_info()

        self.network_mode = self.__get_network_mode_type()
        self.sync_ha_running_legacy_flag()
        return

    def retryPowerOn(self):
        count = 0
        while (count < 12):
            if os.path.isfile('/flash/nsconfig/.AWS/poweron_peer') is True:
                time.sleep(3)
                self.handle_poweron_peer()
                count = count + 1
                continue
            else:
                self.powerOnPeer = False
                self.instance_details_acquired_once = False
                logging.error("Retried peer power on many times.. HA peer is down, now giving up...")
                os.system("rm /flash/nsconfig/.AWS/poweron_peer")
                break

    def sync_ha_running_legacy_flag(self):
        if self.network_mode is NetworkModeType.AWS_HA_MODE_ENI_TRANSFER:
            os.system("touch /flash/nsconfig/.AWS/.ha_running_legacy")
        else:
            os.system("rm -f /flash/nsconfig/.AWS/.ha_running_legacy")

    def get_eni_peer_interfaces_info(self):
        """ Fetches peer Non management interface info """
        result = {}
        for item in self.peer_interfaces:
            # ignore management interface
            if item["Attachment"]["DeviceIndex"] == 0:
                continue
            result[item["Attachment"]["DeviceIndex"]] = {"AttachmentId": item["Attachment"] \
                ["AttachmentId"], "NetworkInterfaceId": item["NetworkInterfaceId"]}
        return result

# HA SameZone IPV6 code starts #
    def __move_Ipv6(self,remote_interface,my_interface,ipv6_dict):
        logging.info("Moving IPv6 ip's to Primary")
        ipv6_list = [ipv6['Ipv6Address'] for ipv6 in ipv6_dict]
        try:
            client_res6 = boto3.client('ec2', self.region, config=self.retry_config)
            client_res6.unassign_ipv6_addresses(Ipv6Addresses=ipv6_list, \
                                                              NetworkInterfaceId=remote_interface)
            logging.info("IPv6 "+str(ipv6_list)+" is removed from Secondary's interface : %s", remote_interface)
            client_res6.assign_ipv6_addresses(Ipv6Addresses=ipv6_list, \
                                                              NetworkInterfaceId=my_interface)
            logging.info("IPv6 is moved to Primary's interface : %s", my_interface)
        except Exception as e:
            logging.error("IP6 movement failed for Primary's interface %s", my_interface)
            logging.info("Boto3 API failed with exception %s", str(e))


    def get_ipv6_topology(self):
        [self.__move_Ipv6(peer_interface['NetworkInterfaceId'],
         self_interface['NetworkInterfaceId'], peer_interface['Ipv6Addresses'])
         for peer_interface in self.peer_interfaces
         for self_interface in self.my_interfaces
         if peer_interface['Ipv6Addresses'] and
         peer_interface['SubnetId'] == self_interface['SubnetId']]

# HA Multizone PIP code starts #
    def __get_peer_eni_map(self):
        peer_interface_map = {}
        for interface in self.peer_interfaces:
            peer_interface_map[interface['Attachment']['DeviceIndex']] = interface['NetworkInterfaceId'] #taking devindex so that interface order is maintained.
        logging.debug("HAPIP Interface map of peer is :%s",peer_interface_map)
        return peer_interface_map


    def __get_self_eni_map(self):
        eniaddress = {}

        for eni in self.my_interfaces:
            eniaddress[int(eni['Attachment']['DeviceIndex'])]=eni['NetworkInterfaceId'] #dev index to make sure interface order is maintained
            logging.debug("HAPIP Self eni mapping is :%s", eniaddress)
        return eniaddress


    def __merge_peer_self_eni_dict(self): # creating a dict with key,value as [eth1-remote: eth1-self, eth2-remote:eth2-self]
        peer_intf_map = self.__get_peer_eni_map()
        self_intf_map = self.__get_self_eni_map()
        interface_map = {}
        for key in peer_intf_map.keys(): 
            if key in self_intf_map.keys():
                interface_map[peer_intf_map[key]] = self_intf_map[key]  # matching peer-eni:self-eni if indices exist in both
            else:
                interface_map[peer_intf_map[key]] = self_intf_map[0]    # transfering all remaining peer-eni routes to self-eni-0 (management)
        logging.debug("HAPIP Merged interface map is :%s",interface_map)
        return interface_map


    def update_route_table(self):
        try:
            self.__init_client_for_shared_iam()
            resp_rt_table = self.client_routes.describe_route_tables(Filters=[{'Name': 'vpc-id','Values': [self.vpc_id,]},])['RouteTables']
        except Exception as e:
            logging.critical("Boto3 API failed with exception %s", str(e))
        interface_mapping=self.__merge_peer_self_eni_dict()    # getting the remote to self interface map
        for key in interface_mapping.keys():                 # for each remote interface route
            route_exist= False                               # Flag to be used in source/dst check disabling only 1 time in loop
            for iter in resp_rt_table:
                for each_route in  iter['Routes']:
                    try:
                        if each_route['NetworkInterfaceId'] == key:
                            logging.info("Found route in VPC for "+str(each_route['DestinationCidrBlock'])+" for ENI %s", key)
                        #Here key is remote HA interface id
                        #and interface_mapping[key] is self interface
                        #If route target matches remote eniid then delete that route and create a new route with self interface as target
                            try:
                                self.client_routes.delete_route(DestinationCidrBlock=each_route['DestinationCidrBlock'],RouteTableId=iter['RouteTableId'])
                                logging.info("Route Deleted for "+str(each_route['DestinationCidrBlock'])+" with eniid "+str(key))
                                self.client_routes.create_route(DestinationCidrBlock=each_route['DestinationCidrBlock'],NetworkInterfaceId=interface_mapping[key],RouteTableId=iter['RouteTableId'])
                                logging.info("Route created for  "+str(each_route['DestinationCidrBlock'])+" with eniid "+str(interface_mapping[key]))
                                route_exist=True
                            except Exception as e:
                                logging.warning("Route modification Failed for "+str(each_route['DestinationCidrBlock']))
                                logging.info("Route modification failed with error %s", str(e))
                    except :
                        continue
            if route_exist == True:                         # This is to disable src/dst check only once in a loop
                try:
                     self.client.modify_network_interface_attribute(NetworkInterfaceId=interface_mapping[key], SourceDestCheck={'Value': False})
                     logging.debug("Disabled the Src/Dst check for interface:"+str(interface_mapping[key]))
                except Exception as e:
                     logging.critical("Unable to disable src/dst check, verify IAM Permission !!!")
                     logging.warning("Src/Dst Check modification failed with exception %s", str(e))
        #Checking IAM permissions for HA Multizone PIP
        try:
            iam_role_check = self.__check_privileges(NetworkModeType.AWS_HA_MODE_MZPIP_TRANSFER)
            if iam_role_check:
                logging.info("If MultiZone PrivateIp HA is configured then : %s IAM Permission missing on ADC" % iam_role_check)
            else:
                logging.info("MZ-PIP-HA IAM Permissions are proper")
        except Exception as e:
            logging.info("Boto3 API failed while chekcing IAM permissions %s", str(e))
# HA Multizone PIP code ends #


    def exec_cmd(self, cmd_list):
        """ executes a sysctl command and returns the output """
        try:
            process = subprocess.run(cmd_list, stdout=subprocess.PIPE, universal_newlines=True)
            return shlex.split(process.stdout)
        except Exception as err:
            logging.error('Command execution failed %s' %err)
            raise


    def is_primary_node(self):
        """ Checks the node is primary """
        return self.state == HaStatus.PRIMARY


    def get_eni_local_interfaces_info(self):
        """ Fetches local data interfaces """
        try:
            response = self.client.describe_instances(InstanceIds=[self.instance_id, ], )
        except Exception as e:
            logging.error("Unable to get peer instance details, Check Internet connectvity !!!")
            logging.info("Boto3 API failed with exception %s", str(e))
            raise
        result = {}
        for item in response["Reservations"][0]["Instances"][0]["NetworkInterfaces"]:
            # ignore management interface
            if item["Attachment"]["DeviceIndex"] == 0:
                continue
            result[item["Attachment"]["DeviceIndex"]] = {"AttachmentId": item["Attachment"] \
                ["AttachmentId"], "NetworkInterfaceId": item["NetworkInterfaceId"]}
        return result


    def __is_mode_eni(self, peer_interface_info, interface_info):
        """ Checking possibility for HA using eni mode """
        ret = True
        if (len(peer_interface_info) == interface_info) or ((len(peer_interface_info) != 0)
                                                            and (interface_info != 0)) is True:
            ret = False
        return ret

    def __is_mode_pip(self, peer_interface_info, interface_info):
        """ Checking possibility for HA using PIP mode """
        ret = True
        if len(peer_interface_info) != interface_info is True:
            ret = False
        return ret


    def __get_network_mode_type(self):
        """ gets the HA Mode type """
        if self.state == HaStatus.STANDALONE:
            return NetworkModeType.AWS_HA_MODE_STANDALONE

        if self.state in [HaStatus.CCO, HaStatus.NON_CCO]:
            return NetworkModeType.AWS_HA_MODE_CLUSTER_TRANSFER

        self.__get_peer_info(self.peer_ip)

        if self.inc_mode != "DISABLED":
            return NetworkModeType.AWS_HA_MODE_EIP_TRANSFER

        peer_interface_info = self.get_eni_peer_interfaces_info()
        # ignore management interface
        interface_info = len(self.my_interfaces) - 1


        if self.__is_mode_eni(peer_interface_info, interface_info) is True:
            logging.info("AWS HA type is ENI")
            return NetworkModeType.AWS_HA_MODE_ENI_TRANSFER
        elif self.__is_mode_pip(peer_interface_info, interface_info) is True:
            logging.info("AWS HA type is PIP")
            return NetworkModeType.AWS_HA_MODE_PIP_TRANSFER

        logging.error("AWS HA type is UNKNOWN")
        return NetworkModeType.AWS_HA_MODE_UNKNOWN


    def __send_signal_to_aws_daemon(self):
        """ Send singnal to all AWS deamon to shut down in secondary"""
        NS.send_signal_to_process(signal.SIGUSR1, RAIN_SCALE_PID_FILE)
        NS.send_signal_to_process(signal.SIGUSR1, RAIN_STAT_PID_FILE)
        NS.send_signal_to_process(signal.SIGUSR1, CLOUDADAPTER_PID_FILE)


    def __launch_aws_daemons(self):
        """ Start aws daemons """

        NS.launch_process_if_not_running(RAIN_SCALE_BIN, RAIN_SCALE_PID_FILE)
        NS.launch_process_if_not_running(RAIN_STAT_BIN, RAIN_STAT_PID_FILE)
        NS.launch_process_if_not_running(CLOUDADAPTER_BIN, CLOUDADAPTER_PID_FILE)

    """ Delete this when deprecating ENI START """
    def ensure_eni_status(self, status, interfaceInfo):
        """ Ensures attachment status is true for an ENI """
        for deviceIndex, ID in  interfaceInfo.items():
            while True:
                if self.get_attachment_status(ID['NetworkInterfaceId'], status) is False:
                    logging.info("Status ENI interface in progress, match again for validation\
                      of: %s status for: %s", status, ID['NetworkInterfaceId'])
                    time.sleep(1)
                    continue
                else:
                    break

        logging.info("ENI interfaces status is %s", status)
        return True

    def get_attachment_status(self, interfaceId, state):
        """ Fetches the attachment status for an ENI ID """
        try:
            response = self.client.describe_network_interfaces(NetworkInterfaceIds=[interfaceId,],)
            status = response['NetworkInterfaces'][0]['Status']
            logging.info("Status of ENI interface %s  and status %s", interfaceId, status)
        except Exception as e:
            logging.error("Boto3 API failed with exception %s"%(str(e)))
        return status == state

    def check_pe_for_hotplug(self):
        """ Checks the hotplugged interfaces """
        cmd_list = ['sysctl','net.link.generic.system.ifcount']
        cur_interfaces = int(self.exec_cmd(cmd_list)[-1]) 
        logging.info("hotplug_check: got interface count: %d", cur_interfaces)

        if self.prev_interfaces != cur_interfaces:
            logging.info("Hotplug interface ...")
            time.sleep(20)
            os.system("/netscaler/awsconfig -p")
            #self.hotplug_pe()
            self.prev_interfaces = cur_interfaces

    def handle_poweroff_peer(self):
        """ power of the peer instance """
        try:
            self.client.stop_instances(InstanceIds=[self.peer_instance_id], Force=True)
            logging.info("Powering off VM with Instance Id %s", self.peer_instance_id)
        except Exception as e:
            logging.error("Boto3 API failed with exception %s"%(str(e)))
        return

    def handle_poweron_peer(self):
        """ power on peer instance """
        if self.instance_details_acquired_once is True:
            logging.info("Powering on peer VM %s", self.peer_ip)
            try:
                response = self.client.start_instances(InstanceIds=[self.peer_instance_id])
            except Exception as e:
                logging.error("Boto3 API failed with exception %s"%(str(e)))
                self.powerOnPeer = True
                return
            logging.info("Powering on peer VM with Instance Id %s", self.peer_instance_id)
            currentState = response['StartingInstances'][0]['CurrentState']['Name']
            pervState = response['StartingInstances'][0]['PreviousState']['Name']
            logging.info("Peer Current state: %s , previous state: %s", currentState, pervState)
            logging.info("Peer VM %s powered on successfully.", self.peer_instance_id)
            os.system("rm /flash/nsconfig/.AWS/poweron_peer")
            return True
        else:
            logging.error("Instance details not acquired. peer power on skipped for now.")
            return False

    def handle_powercycle_peer(self):
        """ power cycle the peer instance """
        if self.is_primary_node() == False:
            logging.info("Secondary node is not supposed to perform powercycle, skipping.")
            return
        logging.debug("Initiate power cycle of peer instance ip %s", self.peer_ip)
        self.handle_poweroff_peer()
        os.system("touch /flash/nsconfig/.AWS/poweron_peer")

    def migrate_eni_from_peer(self):
        """ Migrate data ENIs from peer """
        logging.info("Moving ENI ips from secondary to primary")
        peerInterfaceInfo   = self.get_eni_peer_interfaces_info()
        #ignore management interface
        interfaceInfo       = self.get_eni_local_interfaces_info()

        total_int_count = len(peerInterfaceInfo) + len(interfaceInfo) + 1
        self.instance_details_acquired_once = True

        logging.info("Total interfaces expected on primary: %d, interfaces present on primary: \
        %d, secondary: %d.", total_int_count, len(peerInterfaceInfo) + 1, len(interfaceInfo) + 1)

        if total_int_count == len(interfaceInfo) + 1:
            if len(peerInterfaceInfo) > 0:
                logging.error("Expected only mgmt interface on secondary, found more than 1")

            logging.info("all ENIs already attached to primary, skipping move ENI.")
            return

        logging.info("Detach All ENIs except Mgmt ENI from failing instance.")
        for deviceIndex, ID in peerInterfaceInfo.items():
            logging.info("Detaching Id %d attachId %s", deviceIndex, ID["AttachmentId"])
            try:
                self.client.detach_network_interface(AttachmentId = ID["AttachmentId"], Force=True)
            except Exception as e:
                logging.error("Boto3 API failed with exception %s"%(str(e)))

        if self.ensure_eni_status('available', peerInterfaceInfo) is False:
            return

        logging.info("Attach All ENIs except Mgmt ENI from failing instance.")
        for devIndex, ID in peerInterfaceInfo.items():
            logging.info("Attaching Id %s attachId %s", devIndex, ID['NetworkInterfaceId'])
            try:
                self.client.attach_network_interface(DeviceIndex=devIndex, \
                InstanceId=self.instance_id, NetworkInterfaceId=ID['NetworkInterfaceId'])
            except Exception as e:
                logging.error("Boto3 API failed with exception %s"%(str(e)))

        if self.ensure_eni_status('in-use', interfaceInfo) is False:
            return

        self.check_pe_for_hotplug()
        logging.info("Power cycle peer node after ENI migration.")
        self.handle_powercycle_peer()
        time.sleep(120)
        self.handle_poweron_peer()
        return
        """ Delete this when deprecating ENI END """

    def handle_failover(self):
        """ Handles Failover """
        if self.state == HaStatus.PRIMARY:

            if self.inc_mode == "DISABLED":
                if self.network_mode is NetworkModeType.AWS_HA_MODE_ENI_TRANSFER:
                    logging.error("HA ENI mode will be deprecated soon "
                                  "Check citrix 13.0 doc for HA PIP mode")
                    self.migrate_eni_from_peer()
                    if self.powerOnPeer is True:
                        self.retryPowerOn()
                elif self.network_mode is NetworkModeType.AWS_HA_MODE_PIP_TRANSFER:
                    intf_privateip_map = self.get_intf_privateip_map()
                    if intf_privateip_map:
                        logging.info("Handling SZHA failover here and peer ip is " + self.peer_ip)
                        self.migrate_pip_from_peer(intf_privateip_map)
                    else:
                        logging.info("No Secondary IPs to be moved from peer " + self.peer_ip)
                    self.get_ipv6_topology()
            else:
                logging.info("Handling MZHA failover here and peer ip is " + self.peer_ip)
                try:
                    self.migrate_eip_from_peer()     # This is EIP Multizone code trigger
                except Exception as e:
                    logging.error("Failed with exception %s", str(e))
                self.update_route_table()        # This is PIP Multizone code trigger.
        elif self.state == HaStatus.CCO:
            intf_clip_map = self.get_intf_clip_map()
            self.migrate_pip_from_peer(intf_clip_map)


    def do_periodic(self):
        """ Do periodic cloud specific tasks
                1. IAM check
                2. Check for status of rain_scale and rain_stat process """

        if self.state == HaStatus.STANDALONE:
            logging.debug("Standalone System skipping Periodic Tasks")
            if os.path.exists(IAM_NOT_OK_FILE):
                os.remove(IAM_NOT_OK_FILE)
            return

        missing_instance_iam = self.__check_privileges(self.network_mode)

        if missing_instance_iam:
            with open(IAM_NOT_OK_FILE, 'w') as iam_file:
                iam_file.write(json.dumps({"instance_iam": list(missing_instance_iam)}))
        else:
            logging.debug("Periodic IAM Check Passed")
            if os.path.exists(IAM_NOT_OK_FILE):
                os.remove(IAM_NOT_OK_FILE)
        return
