#!/usr/bin/python
###############################################################################
#
#  Copyright (c) 2019-2024 Citrix Systems, Inc.
#  All rights reserved.
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions are met:
#      * Redistributions of source code must retain the above copyright
#        notice, this list of conditions and the following disclaimer.
#      * Redistributions in binary form must reproduce the above copyright
#        notice, this list of conditions and the following disclaimer in the
#        documentation and/or other materials provided with the distribution.
#      * Neither the name of the Citrix Systems, Inc. nor the
#        names of its contributors may be used to endorse or promote products
#        derived from this software without specific prior written permission.
#
#  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
#  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
#  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
#  ARE DISCLAIMED. IN NO EVENT SHALL CITRIX SYSTEMS, INC. BE LIABLE FOR ANY
#  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
#  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
#  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
#  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
#  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
###############################################################################

import logging
from libnitrocli import nitro_cli
from src.ha_state import HaStatus
import subprocess
import shlex
import os
import psutil

class NS:
    """ Generic API class for HA.Contains netscaler specific APIs functions. """
    def __init__(self):
        self.nitro = None
        self.peer_ip = None
        self.inc_mode = None
        self.cloudplatform = 0
        self.state = HaStatus.STANDALONE
        self.set_cloud_platform()
        self.init_nitro()
        self.update_ha_info()


    def set_cloud_platform(self):
        """ Initialize the platform ID """
        #If VPX on linux read cloud platform from environ file, else use sysctl
        if "NS_APPLIANCE" in os.environ:
            if "VPX_ON_CLOUD" in os.environ:
                self.cloudplatform = int(os.environ.get("VPX_ON_CLOUD"))
        else:
            cmd_list = ['sysctl', 'netscaler.vpx_on_cloud']
            self.cloudplatform = int(self.exec_cmd(cmd_list)[-1])

    def exec_cmd(self, cmd_list):
        try:
            process = subprocess.run(cmd_list, stdout=subprocess.PIPE, universal_newlines=True)
            return shlex.split(process.stdout)
        except Exception as err:
            logging.error('Command execution failed %s' %err)
            raise

    def init_nitro(self):
        """ Initialize nitrocli object """
        try:
            self.nitro = nitro_cli()
        except:
            self.nitro = None
            logging.error('Nitro cli config failed')

    def update_ha_info(self):
        """ Gets HA /  CLUSTER information"""
        self.state = self.get_node_state()
        if self.state in [HaStatus.PRIMARY, HaStatus.SECONDARY]:
            self.update_peer_ip()
        else :
            self.peer_ip = None
            self.inc_mode = None

    def update_peer_ip(self):
        """ Updates peer IP and INC mode """
        hanode_info = self.nitro.get_hanode()
        self.peer_ip = hanode_info['hanode'][1]['ipaddress']
        self.inc_mode = hanode_info['hanode'][0]['inc']
        logging.info("Peer IP is %s INC Mode is %s"%(self.peer_ip, \
		self.inc_mode))

    def get_node_state(self):
        """ Get the nsconfig and the system type """
        cco_node = 0x00000040
        master_node = 0x00000004
        nsconfig_info = self.nitro.get_nsconfig()
        node_cco = str(nsconfig_info[u'nsconfig'])
        if node_cco == 'NON-CCO':
            return HaStatus.NON_CCO
        ns_systemtype = str(nsconfig_info[u'nsconfig'][u'systemtype'])
        nc_flags = int(nsconfig_info[u'nsconfig'][u'flags'])
        if ns_systemtype == 'Cluster':
            return (HaStatus.CCO if (nc_flags & cco_node) == cco_node \
			else HaStatus.NON_CCO)
        elif ns_systemtype == 'HA':
            return (HaStatus.PRIMARY if (nc_flags & master_node) == \
			master_node else HaStatus.SECONDARY)
        return HaStatus.STANDALONE

    @staticmethod
    def launch_process_if_not_running(cmd, daemon_pid_file):
        """Check if daemon is already running Else create the process """
        if os.path.exists(daemon_pid_file) and \
			os.stat(daemon_pid_file).st_size != 0:
            with open(daemon_pid_file,'r') as pid:
                daemon_pid = int(pid.read())
                if psutil.pid_exists(daemon_pid):
                    daemon_proc = psutil.Process(daemon_pid)
                    if daemon_proc.status() in { psutil.STATUS_DEAD, psutil.STATUS_ZOMBIE }:
                        logging.info("Daemon not running, process status - ", daemon_proc.status())
                    else:
                        logging.debug("Daemon is already running")
                        return
        logging.info("launch_process : cmd : %s, daemon_pid_file: %s"\
		%(cmd, daemon_pid_file))
        subprocess.Popen([cmd])
        #TODO Use shlex.split() in case we need to run complex shell cmds
        return

    @staticmethod
    def send_signal_to_process(signal_num, daemon_pid_file):
        """ sends a given signal to the process """
        logging.debug("send_signal_to_process : signal_num : %s, \
		daemon_pid_file: %s"%(signal_num, daemon_pid_file))
        if os.path.exists(daemon_pid_file) and \
			os.stat(daemon_pid_file).st_size != 0:
            with open(daemon_pid_file,'r') as pid:
                daemon_pid = int(pid.read())
                if psutil.pid_exists(daemon_pid):
                    os.kill(daemon_pid, signal_num)
                    logging.info("Stopped the process %s", daemon_pid_file)
