# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""RSA key generation.

Create new keys using the newkeypair function. It will return a tuple
containing the encoded public and private keys.
"""
from .codec import encode_publickey, encode_privatekey
import rsa


def newkeypair():
    """Generates public and private keys, and returns them as (pub, priv).

    :returns:   a tuple (pub, priv) containing the encoded keys
    """
    (pub, priv) = _newkeys(2048)
    return encode_publickey(pub), encode_privatekey(priv)


def _newkeys(nbits):
    (p, q) = _find_p_q(nbits // 2)
    (e, d) = rsa.key.calculate_keys(p, q)
    n = p * q
    return rsa.PublicKey(n, e), rsa.PrivateKey(n, e, d, p, q)


def _find_p_q(nbits):
    total_bits = nbits * 2

    # IMPORTANT (luisga): p and q must have the same bit length to ensure .NET compatibility.
    pbits = nbits
    qbits = nbits

    # choose the initial primes
    p = rsa.prime.getprime(pbits)
    q = rsa.prime.getprime(qbits)

    def primes_acceptable(p, q):
        if p == q:
            return False

        # ensure we have just the right amount of bits
        found_size = rsa.common.bit_size(p * q)
        return total_bits == found_size

    # keep choosing other primes until they match the requirements.
    change_p = False
    while not primes_acceptable(p, q):
        # change p and q on alternate iterations
        if change_p:
            p = rsa.prime.getprime(pbits)
        else:
            q = rsa.prime.getprime(qbits)

        change_p = not change_p

    # p > q as described in http://www.di-mgt.com.au/rsa_alg.html#crt
    return max(p, q), min(p, q)
