# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""RSA signing operations.

Sign values and verify signatures using the sign/verify functions. They
both accept the keys as objects or encoded.
"""
from .codec import decode_publickey, decode_privatekey
import rsa

_HASH_ALGORITH = 'SHA-256'


def sign(message, priv_key):
    """Signs a message using the given private key using RSA-SHA256.

    :param message:     the message to sign
    :param priv_key:    the private key to use for signing
    :returns:           the RSA-SHA256 signature encoded as base64
    """
    if message is None:
        return None

    key = decode_privatekey(priv_key) if not isinstance(priv_key, rsa.PrivateKey) else priv_key
    sig = rsa.sign(message, key, _HASH_ALGORITH)
    return sig


def verify(message, signature, pub_key):
    """Verifies that the given signature matches the message.

    :param message:     the signed message to verify
    :param signature:   the signature, as returned by :py:func:`cwc.crypto.rsa.sign`
    :param pub_key:     the public key to use to verify the signature
    :returns:           a boolean value indicating whether the signature matches the message
    """
    if message is None or signature is None:
        return None

    key = decode_publickey(pub_key) if not isinstance(pub_key, rsa.PublicKey) else pub_key
    return rsa.verify(message, signature, key)
