# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""Error handling definitions"""


class ErrorCode:
    """Error codes as defined in libccauth"""
    OK = 0
    ERROR = 1
    OUT_OF_MEMORY = 2
    INVALID_ARGUMENT = 3
    MISSING_ROUTE = 4
    MISSING_SERVICE_NAME = 5
    MISSING_SIGN_CALLBACK = 6
    MISSING_REQUEST_CALLBACK = 7
    INVALID_REQUEST = 8
    REQUEST_ABORTED = 9
    REQUEST_FAILED = 10
    INVALID_RESPONSE = 11
    MISSING_VERIFY_CALLBACK = 12
    BROKEN_CIRCUIT = 17

    @staticmethod
    def tostring(code):
        """Converts an error code to its string representation.

        :param code:    the error code to convert
        :returns:       the str representation matching the given error code
        """
        if code == ErrorCode.OK: return 'OK'
        if code == ErrorCode.ERROR: return 'ERROR'
        if code == ErrorCode.OUT_OF_MEMORY: return 'OUT_OF_MEMORY'
        if code == ErrorCode.INVALID_ARGUMENT: return 'INVALID_ARGUMENT'
        if code == ErrorCode.MISSING_ROUTE: return 'MISSING_ROUTE'
        if code == ErrorCode.MISSING_SERVICE_NAME: return 'MISSING_SERVICE_NAME'
        if code == ErrorCode.MISSING_SIGN_CALLBACK: return 'MISSING_SIGN_CALLBACK'
        if code == ErrorCode.MISSING_REQUEST_CALLBACK: return 'MISSING_REQUEST_CALLBACK'
        if code == ErrorCode.INVALID_REQUEST: return 'INVALID_REQUEST'
        if code == ErrorCode.REQUEST_ABORTED: return 'REQUEST_ABORTED'
        if code == ErrorCode.REQUEST_FAILED: return 'REQUEST_FAILED'
        if code == ErrorCode.INVALID_RESPONSE: return 'INVALID_RESPONSE'
        if code == ErrorCode.MISSING_VERIFY_CALLBACK: return 'MISSING_VERIFY_CALLBACK'
        if code == ErrorCode.BROKEN_CIRCUIT: return 'BROKEN_CIRCUIT'

        raise ValueError('code')


class CCAuthException(Exception):
    """Exception raised when libccauth operations fail"""

    def __init__(self, errorcode, message):
        # Call the base class constructor with the parameters it needs
        super(CCAuthException, self).__init__(message)

        self.errorcode = errorcode
        self.message = message

    def __str__(self):
        return '[{0}] {1}'.format(ErrorCode.tostring(self.errorcode), self.message)


def ensure_success_code(errorcode, message):
    """Ensures that a given error code indicates success or raises a py:class:`ccauth.CCAuthException`.

    :param errorcode:   the error code to inspect
    :param message:     the error message to include when code does not indicate success
    """
    if errorcode != ErrorCode.OK:
        raise CCAuthException(errorcode, message)
