# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""Binding operation targeting the current OS + ARCH."""
import os
import platform
import ctypes

# assigning this to a global variable so that it acts like a global cache, even if _load_library() is called multiple
# times during the ccauth library's life-cycle
_libccauth = None


def _get_lib_filename():
    # build '<repo_root>/ccauth/resources/lib' from the current file path
    current_path = os.path.realpath(__file__)
    pattern = '{0}ccauth{0}'.format(os.sep)
    index = current_path.rindex(pattern)
    ccauth_dir = current_path[:index + len(pattern)]
    lib_dir = '{0}resources{1}lib{1}'.format(ccauth_dir, os.sep)

    # ensure that '<repo_root>/ccauth/resources/lib' exists
    if not os.path.isdir(lib_dir):
        raise SystemExit("could not find '{0}'".format(lib_dir))

    # detect OS and ARCH
    system = platform.system().lower()
    architecture = platform.architecture()[0]

    if system not in ['windows', 'darwin', 'linux', 'freebsd'] or architecture not in ['64bit', '32bit']:
        raise SystemExit('{0}/{1} is not supported '.format(system, architecture))

    # build path
    arch = 'amd64' if architecture == '64bit' else ('i386' if architecture == '32bit' else None)
    ext = 'dll' if system == 'windows' else ('dylib' if system == 'darwin' else 'so')
    return '{0}{2}{1}{3}{1}libccauth.{4}'.format(lib_dir, os.sep, system, arch, ext)


# TODO: copy resources to ccauth-python/ccauth/libs/


def _load_library():
    global _libccauth

    if _libccauth is None:
        filename = _get_lib_filename()
        _libccauth = ctypes.cdll.LoadLibrary(filename)

    return _libccauth
