# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

from ctypes import *


class CCAuthRsaParam(Structure):
    """Creates a struct mirroring ccauth_rsa_param"""
    _fields_ = [
        ('data', POINTER(c_ubyte)),
        ('size', c_size_t)
    ]

    def _unmarshal(self):
        if hasattr(int, 'from_bytes'):
            value = bytes(POINTER(c_ubyte * self.size).from_buffer(self.data)[0])
            return int.from_bytes(value, byteorder='big', signed=False)
        else:
            value = string_at(self.data, self.size)
            return int(value.encode('hex'), 16)

    @staticmethod
    def _marshal(param, pack_power_two=True):
        b = _tobytes(param, pack_power_two)
        data = c_char_p(b)

        native = CCAuthRsaParam()
        native.data = cast(data, POINTER(c_ubyte))
        native.size = len(b)
        return native


class CCAuthRsaParameters(Structure):
    """Creates a struct mirroring ccauth_rsa_parameters"""
    _fields_ = [
        ('exponent', CCAuthRsaParam),
        ('modulus', CCAuthRsaParam),
        ('p', CCAuthRsaParam),
        ('q', CCAuthRsaParam),
        ('dp', CCAuthRsaParam),
        ('dq', CCAuthRsaParam),
        ('inverse_q', CCAuthRsaParam),
        ('d', CCAuthRsaParam)
    ]


def _tobytes(number, pack_power_two=True):
    if hasattr(number, 'to_bytes'):
        bitlen = number.bit_length()

        if pack_power_two:
            bytelen = max(1, _next_power_two(bitlen) // 8)
        else:
            bytelen = bitlen // 8 if bitlen % 8 == 0 else bitlen // 8 + 1

        return number.to_bytes(bytelen, byteorder='big', signed=False)
    else:
        s = '{0:x}'.format(number)

        if len(s) % 2 != 0:
            s = '0' + s

        return s.decode('hex')


def _next_power_two(n):
    return 1 << (n - 1).bit_length()
