# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""Default request handler implementation."""
from ccauth.util import asbytes, asstring
from .request import RequestMethod, ResponseMessage
import requests


def default_request_handler(request):
    """Default request handler.

    :param request:     the request message, see :py:class:`ccauth.net.RequestMessage`
    :returns:           an instance of :py:class:`ccauth.net.ResponseMessage`
    """
    return _request_handler_impl(request, None)


def pooled_request_handler(pool_connections=40, pool_maxsize=80):
    """Creates a custom request handler that reuses the underlying TCP connections.

    :param pool_connections:    the number of connections to cache
    :param pool_maxsize:        the maximum number of connections that will be saved
    :returns:                   a function that can be used as request handler
    """
    session = requests.Session()
    session.mount('https://', requests.adapters.HTTPAdapter(pool_connections, pool_maxsize))
    session.mount('http://', requests.adapters.HTTPAdapter(pool_connections, pool_maxsize))
    return lambda request: _request_handler_impl(request, session)


def _request_handler_impl(request, session):
    method = RequestMethod.tostring(request.method)

    if session is None:
        response = requests.request(method, request.uri, headers=request.headers, data=request.content)
    else:
        response = session.request(method, request.uri, headers=request.headers, data=request.content)

    response_status = response.status_code
    response_content = asstring(response.content)
    response.close()

    return ResponseMessage(
        status = response_status,
        content = asbytes(response_content) if len(response_content) > 0 else None
    )
