# Copyright (C) 2018-2023. Cloud Software Group, Inc. All Rights Reserved. Confidential & Proprietary.

"""Authorization header operations."""
from ctypes import *
from ccauth.util import isempty, asstring, asbytes
from ccauth.error import ensure_success_code
import ccauth.interop as interop


def create_auth_header(bearertoken=None, servicekey=None, quoted=True):
    """Creates and authoration header using the CWSAuth schema

    :param bearertoken  the bearer token identifying the calling user
    :param servicekey   the service key identifying the calling service
    :param quoted       whether to include (double) quotes in the header
    :returns:           a CWSAuth authorization header
    """
    if isempty(bearertoken) and isempty(servicekey):
        raise ValueError("both 'bearertoken' and 'servicekey' cannot be empty.")

    header_ptr = c_char_p()
    
    err = interop.ccauth_create_autheader(
        asbytes(bearertoken),
        asbytes(servicekey),
        1 if quoted else 0,
        byref(header_ptr))

    ensure_success_code(err, 'could not create native authorization header.')

    try:
        return asstring(string_at(header_ptr))
    finally:
        interop.ccauth_free(header_ptr)
