# Copyright Citrix Systems, Inc. All rights reserved.

"""CWC Service Key.

Use :py:class:`cwc.auth.servicekey.CwcServiceKeyParameters` to describe the parameters
required to create service keys.

Create new service keys using the createkey function. It will return a base64-encoded
CWC service key.

Read CWC service keys using the readkey function. It will return a tuple containing
the signature of the target url, the issuer service name, and an optional instance id.

Decode CWC service keys using the decodekey function. It will return an instance of
:py:class:`cwc.auth.servicekey.CwcServiceKey` with the decoded parameters.
"""
from cwc.crypto import sign, verify, randstring
from cwc.util import isempty, getbytes, frombytes, unixtimestamp
from cwc.auth.signing import CwcSigningAlgorithm
import base64

class CwcServiceKeyParameters:
    """Represents a set of parameters required to create CWC service keys."""

    def __init__(self, signingkey, servicename, instanceid = None, signing_algorithm = None):
        """Initializes a new instance.

        :param signingkey:          the private key to use for signing
        :param servicename:         the service name associated with the private key
        :param instanceid:          the service instance id (optional)
        :param signing_algorithm:   the signing algorithm to use (optional)
        :returns:                   a new instance of :py:class:`cwc.auth.servicekey.CwcServiceKeyParameters`
        """
        if isempty(signingkey):
            raise ValueError('signingkey')

        if isempty(servicename):
            raise ValueError('servicename')

        if not signing_algorithm is None and not CwcSigningAlgorithm.isvalid(signing_algorithm):
            raise ValueError('signing_algorithm')

        self.signingkey = signingkey
        self.servicename = servicename
        self.instanceid = instanceid
        self.signing_algorithm = signing_algorithm

class CwcServiceKey:
    """Represents a CWC service key."""

    def __init__(self, signing_algorithm, signature, servicename, instanceid = None, timestamp = None, nonce = None):
        """Initializes a new instance.

        :param signing_algorithm:   the signing algorithm version
        :param signature:           the service key signature
        :param servicename:         the caller service name
        :param instanceid:          the caller service instance id (optional)
        :param timestamp:           the creation date as a Unix timestamp (optional)
        :param nonce:               the nonce value associated with the key (optional)
        :returns:                   a new instance of :py:class:`cwc.auth.servicekey.CwcServiceKey`
        """
        if isempty(signing_algorithm):
            raise ValueError('signingkey')

        if isempty(signature):
            raise ValueError('signature')

        if isempty(servicename):
            raise ValueError('servicename')

        self.servicename = servicename
        self.instanceid = instanceid
        self.signature = signature
        self.timestamp = timestamp
        self.nonce = nonce
        self.signing_algorithm = signing_algorithm

def createkey(keyparams, targeturl):
    """Creates a new service key targeting the given CWC url.

    :param keyparams:   the :py:class:`cwc.auth.servicekey.CwcServiceKeyParameters` to use
    :param targeturl:   the CWC target url
    :returns:           a base64-encoded service key targeting the given url
    """
    algorithm = CwcSigningAlgorithm.MIN_SUPPORTED_VERSION if isempty(keyparams.signing_algorithm) else keyparams.signing_algorithm.upper()

    if algorithm == CwcSigningAlgorithm.RSA_SHA256_V1:
        return _create_rsa_sha256v1_key(keyparams, targeturl)
    elif algorithm == CwcSigningAlgorithm.RSA_SHA256_V2:
        return _create_rsa_sha256v2_key(keyparams, targeturl)
    else:
        raise ValueError(algorithm)

def _create_rsa_sha256v1_key(keyparams, targeturl):
    sig = sign(targeturl, keyparams.signingkey)
    instanceid = None if isempty(keyparams.instanceid) else keyparams.instanceid
    key = '%s;%s%s%s' % (sig, keyparams.servicename, '' if instanceid is None else ';', instanceid or '')
    b64 = base64.b64encode(getbytes(key))
    return frombytes(b64)

def _create_rsa_sha256v2_key(keyparams, targeturl):
    version = CwcSigningAlgorithm.RSA_SHA256_V2
    timestamp = unixtimestamp()
    nonce = randstring(128)
    sigstr = CwcSigningAlgorithm.get_signature_string(targeturl, timestamp, nonce)
    sig = sign(sigstr, keyparams.signingkey)
    instanceid = None if isempty(keyparams.instanceid) else keyparams.instanceid
    key = '%s;%d;%s;%s;%s%s%s' % (version, timestamp, nonce, sig, keyparams.servicename, '' if instanceid is None else ';', instanceid or '')
    b64 = base64.b64encode(getbytes(key))
    return frombytes(b64)

def decodekey(servicekey):
    """Decodes an encoded CWC service key.

    :param servicekey:  the servicekey to decode
    :returns:           an instance of :py:class:`cwc.auth.servicekey.CwcServiceKey`
    """
    b64 = getbytes(servicekey)
    decoded = base64.b64decode(b64)
    key = frombytes(decoded)
    params = key.strip(';').split(';')
    decodedkey = _decode_rsa_sha256v2_key(params)
    return _decode_rsa_sha256v1_key(params) if decodedkey is None else decodedkey

def _decode_rsa_sha256v1_key(params):
    count = len(params)

    if count < 2 or count > 3:
        return None

    signature = params[0]
    servicename = params[1]
    instanceid = params[2] if count == 3 else None
    return CwcServiceKey(CwcSigningAlgorithm.RSA_SHA256_V1, signature, servicename, instanceid)

def _decode_rsa_sha256v2_key(params):
    count = len(params)

    if count < 5 or count > 6:
        return None

    algorithm = params[0]
    timestamp = int(params[1])
    nonce = params[2]
    signature = params[3]
    servicename = params[4]
    instanceid = params[5] if count == 6 else None
    return CwcServiceKey(algorithm, signature, servicename, instanceid, timestamp, nonce)

def readkey(servicekey):
    """Reads the main values from an encoded CWC service key: signature, service name, and instance id.
    See decodekey for a full decoded service key.

    :param servicekey:  the servicekey to decode
    :returns:           a tuple (sig, service, instance) describing the content of the given service key
    """
    key = decodekey(servicekey)

    if key is None:
        return None

    if key.instanceid is None:
        if key.timestamp is not None and key.nonce is not None:
            return tuple([key.signature, key.servicename, key.timestamp, key.nonce])
        else:
            return tuple([key.signature, key.servicename])

    if key.timestamp is not None and key.nonce is not None:
        return tuple([key.signature, key.servicename, key.instanceid, key.timestamp, key.nonce])

    return tuple([key.signature, key.servicename, key.instanceid])
