# Copyright Citrix Systems, Inc. All rights reserved.

"""CWC client request helpers.

Get the required headers for a CWC request using the getheaders function.
Use the py:class:`cwc.client.request.CwcRequestSettings` objects to specify
the requirements.
"""
from cwc.crypto import sign
from cwc.auth import createkey, CwcAuthorizationMode, CwcAuthorizationHeader
from cwc.util import isempty

_CWC_HEADER_AUTHORIZATION = 'Authorization'

class CwcRequestSettings:
    """Represents a set of settings describing a CWC request."""

    def __init__(self, keyparams = None, servicekey = None, bearer = None):
        """Initializes a new instance.

        :param keyparams:   the py:class:`cwc.auth.servicekey.CwcServiceKeyParameters`
                            to use to create service keys
        :param servicekey:  a specific service key (e.g. master key) to use
        :param bearer:      the bearer token to use when sending the request
        :returns:           a new instance of :py:class:`cwc.client.request.CwcRequestSettings`
        """
        self.keyparams = keyparams
        self.servicekey = servicekey
        self.bearer = bearer

def getheaders(mode, settings, targeturl = None):
    """Gets the required headers for a CWC request described by the given settings.

    :param mode:        the CWC authorization mode
    :param settings:    the settings describing the request
    :param targeturl:   optional CWC targeturl, required for per-service keys
    :returns:           a dictionary containing the headers required to make the request
    """
    if settings is None:
        raise ValueError('settings')

    auth = _getauthheader(mode, settings, targeturl)

    return {} if auth is None else {_CWC_HEADER_AUTHORIZATION: str(auth)}

def _getauthheader(mode, settings, targeturl):
    if mode is None:
        return None

    servicekey = settings.servicekey
    bearer = settings.bearer

    if mode.requires_servicekey() and isempty(servicekey):
        if isempty(targeturl) and not mode.requires_bearer():
            raise ValueError('targeturl')

        keyparams = settings.keyparams

        if keyparams is None and not mode.requires_bearer():
            raise ValueError('settings.keyparams')

        if not (keyparams is None) and not isempty(targeturl):
            servicekey = createkey(keyparams, targeturl)

    if mode.requires_bearer() and isempty(bearer) and not mode.requires_servicekey():
        raise ValueError('settings.bearer')

    if isempty(servicekey) and isempty(bearer):
        raise ValueError('settings')

    return CwcAuthorizationHeader(bearer, servicekey)
