#!/usr/bin/python
# Copyright 2016 Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
########################################################################################
#
#  Copyright (c) 2018-2022 Citrix Systems, Inc.
#  All rights reserved.
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions are met:
#      * Redistributions of source code must retain the above copyright
#        notice, this list of conditions and the following disclaimer.
#      * Redistributions in binary form must reproduce the above copyright
#        notice, this list of conditions and the following disclaimer in the
#        documentation and/or other materials provided with the distribution.
#      * Neither the name of the Citrix Systems, Inc. nor the
#        names of its contributors may be used to endorse or promote products
#        derived from this software without specific prior written permission.
#
#  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
#  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
#  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
#  DISCLAIMED. IN NO EVENT SHALL CITRIX SYSTEMS, INC. BE LIABLE FOR ANY
#  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
#  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
#  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
#  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
#  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
#  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
########################################################################################

"""Manage user accounts on a Google Compute Engine instances."""

import datetime
import json
import logging
import logging.handlers
from logging.handlers import RotatingFileHandler
import optparse
import random
"""
NETSCALER begins
"""
import platform
import os
"""
NETSCALER ends.
"""

from google_compute_engine import config_manager
from google_compute_engine import constants
from google_compute_engine import file_utils
from google_compute_engine import logger
from google_compute_engine import metadata_watcher
from google_compute_engine.accounts import accounts_utils
"""
NETSCALER begins
"""
from google_compute_engine.accounts import nsvpxaccounts_utils
from google_compute_engine.accounts import nsvpx_vipscaling_utils
"""
NETSCALER ends.
"""
from google_compute_engine.accounts import oslogin_utils

"""
NETSCALER begins
"""
try:
    from libnitrocli import nitro_cli, nitro_cli_output_parser
except:
    pass

LOCKFILE = '/var/log/gcp/google-accounts.lock'
_LOG_FILE_DIRPATH = '/var/log/gcp/'
_LOG_FILE_NAME = 'google-accounts.log'
_LOG_NEXT_FILE = '/var/log/gcp/nextfile'
_LOG_FILE_MAX_BYTES = 100*1024              # 100 kilobytes, maximum file size after which the log file will be rotated
_LOG_FILEBACKUP_COUNT = 100                 # maximum number of rotated log files that system will keep. Exceeding this value will start deleting older log files 
"""
NETSCALER ends.
"""

class AccountsDaemon(object):
  """Manage user accounts based on changes to metadata."""

  invalid_users = set()
  user_ssh_keys = {}

  def __init__(
      self, groups=None, remove=False, useradd_cmd=None, userdel_cmd=None,
      usermod_cmd=None, groupadd_cmd=None, debug=False):
    """Constructor.
    Args:
      groups: string, a comma separated list of groups.
      remove: bool, True if deprovisioning a user should be destructive.
      useradd_cmd: string, command to create a new user.
      userdel_cmd: string, command to delete a user.
      usermod_cmd: string, command to modify user's groups.
      groupadd_cmd: string, command to add a new group.
      debug: bool, True if debug output should write to the console.
    """
    """
    NETSCALER begins
    """
    if platform.system() == 'FreeBSD':
        if not os.path.exists(_LOG_FILE_DIRPATH):
            os.makedirs(_LOG_FILE_DIRPATH)
        logger_instance = logging.getLogger()
        logging.basicConfig(filename="%s" %(str(_LOG_FILE_DIRPATH + _LOG_FILE_NAME)), format='%(asctime)s %(levelname)s %(message)s', level=logging.INFO)
        #Avoid duplicate logs
        if not len(logger_instance.handlers):
            handler = RotatingFileHandler("%s/%s" % (_LOG_FILE_DIRPATH, _LOG_FILE_NAME), maxBytes=_LOG_FILE_MAX_BYTES, backupCount=_LOG_FILEBACKUP_COUNT)
            handler.setFormatter(logging.Formatter('%(asctime)s %(levelname)s %(message)s'))
            handler.setLevel(logging.INFO)
            logger_instance.addHandler(handler)
        self.logger = logger_instance
        try:
            self.__nitro = nitro_cli()
            self.__parser = nitro_cli_output_parser()
        except:
            self.__nitro = None
            self.__parser = None

        self.watcher = metadata_watcher.MetadataWatcher(logger=self.logger)
        self.logger.info('Platform is NetScaler on GCP')
        self.utils = nsvpxaccounts_utils.NSVPXAccountsUtils(
            logger=self.logger, groups=groups, remove=remove,
            useradd_cmd=useradd_cmd, userdel_cmd=userdel_cmd,
            usermod_cmd=usermod_cmd, groupadd_cmd=groupadd_cmd)
    else:
        facility = logging.handlers.SysLogHandler.LOG_DAEMON
        self.logger = logger.Logger(
            name='google-accounts', debug=debug, facility=facility)
        self.watcher = metadata_watcher.MetadataWatcher(logger=self.logger)
        self.logger.info('Platform is Linux on GCP')
        self.utils = accounts_utils.AccountsUtils(
            logger=self.logger, groups=groups, remove=remove,
            useradd_cmd=useradd_cmd, userdel_cmd=userdel_cmd,
            usermod_cmd=usermod_cmd, groupadd_cmd=groupadd_cmd)

    """
    NETSCALER ends.
    """
    self.oslogin = oslogin_utils.OsLoginUtils(logger=self.logger)

    try:
      with file_utils.LockFile(LOCKFILE):
        self.logger.info('Starting Google Accounts daemon.')
        """
        NETSCALER begins
        """
        #TODO:  Interim workaround until DNS team makes changes to avoid 
        #... removing DNS entries specific for GCP nameserver
        #TODO: we exit the daemon and reinstall nscli commands when metadata
        #... server is not reachable when the daemon restarts.
        #Adding NS Configuration
        nsconfig_info = self.__nitro.get_nsconfig()
        ns_systemtype = str(nsconfig_info[u'nsconfig'][u'systemtype'])
        if ns_systemtype == 'Cluster':
            self.logger.info('DNS LB Vserver will be configured by ADM in cluster setup')
        else:
            self.logger.info('Check for presence of DNS LB vserver and DNS Service.')
            lb_vserver_name = "gcplbdnsvserver"
            result0 = self.__nitro.get_lbvserver(lb_vserver_name)
            service_name = "gcplbdnsservice0"
            result1 = self.__nitro.get_service(service_name)
            if(self.__parser.success(result0) and self.__parser.success(result1)):
                #self.logger.info('DNS LB vserver and DNS service present. Skipping configuration.')
                pass
            else:
                self.logger.info('DNS LB vserver or DNS LB service not present. Adding configuration.')
                feature = "lb"
                result = self.__nitro.enable_nsfeature(feature)
                if(self.__parser.success(result)) :
                    #self.logger.info("enable nsfeat success")
                    pass
                else:
                    self.logger.info("enable nsfeat failure")
                lb_vserver_name = "gcplbdnsvserver"
                lb_protocol = "dns"
                lb_method = "ROUNDROBIN"
                result = self.__nitro.add_lbvserver_gcp(lb_vserver_name, lb_protocol, lb_method)
                if(self.__parser.success(result)) :
                    #self.logger.info("add lbvserver success")
                    dns_nameserver_name = "gcplbdnsvserver"
                    result = self.__nitro.add_dns_nameserver(dns_nameserver_name)
                else:
                    self.logger.info("add lbvserver failure")
                service_name = "gcplbdnsservice0"
                service_ip = "169.254.169.254"
                service_type = "dns"
                service_port = "53"
                result = self.__nitro.add_service_gcp(service_name, service_ip, service_type, service_port)
                if(self.__parser.success(result)) :
                    #self.logger.info("add service success")
                    pass
                else:
                    self.logger.info("add service failure")
                result = self.__nitro.bind_service_lbmonitor(service_name, service_type)
                if(self.__parser.success(result)) :
                    #self.logger.info("bind service and monitor success")
                    pass
                else:
                    self.logger.info("bind service and monitor failure")
                result = self.__nitro.bind_lbvserver_service(lb_vserver_name, service_name)
                if(self.__parser.success(result)) :
                    #self.logger.info("bind lbvserver and service success")
                    pass
                else:
                    self.logger.info("bind lbvserver and service failure")

        self.vip_scaling = nsvpx_vipscaling_utils.vipScaling(self.logger)
        """
        NETSCALER ends.
        """
        timeout = 60 + random.randint(0, 30)
        self.watcher.WatchMetadata(
            self.HandleAccounts, recursive=True, timeout=timeout)
    except (IOError, OSError) as e:
        self.logger.warning(str(e))

  #TODO: Rotate log using NS functionality as we do for ns.log
  def rotate_log(self, log_file_path, log_dir_path):
    cmd = "ps -ax | grep \"google-accounts\" | awk '{print $1}'"
    ret, pid = shellutil.run_get_output(cmd, chk_err=False)
    pid_list = pid.split("\n") 
    curr_pid = os.getpid()

    if (str(curr_pid) in pid_list):
       if not os.path.isfile(self.next_file):
           fileutil.write_file(self.next_file, "0", append=False)
           fileutil.write_file(self.agent_pid, str(curr_pid), append=False)
       else:
           if os.path.isfile(self.agent_pid):
               last_pid = fileutil.read_file(self.agent_pid)
               if (curr_pid != int(last_pid)):
                   last_index = fileutil.read_file(_LOG_NEXT_FILE)
                   new_index = int(last_index) + 1
                   fileutil.write_file(_LOG_NEXT_FILE, str(new_index), append=False)
                   fileutil.write_file(self.agent_pid, str(curr_pid), append=False)
                   cmd = "mv {0} {1}.{2}".format(str(log_file_path), str(log_file_path), str(int(last_index) % 20))
                   ret = shellutil.run(cmd, chk_err=False)
           else:
               #This is upgrade case. So new agent should start from 0 again.
               fileutil.write_file(self.next_file, "0", append=False)
               fileutil.write_file(self.agent_pid, str(curr_pid), append=False)

  def _HasExpired(self, key):
    """Check whether an SSH key has expired.
    Uses Google-specific semantics of the OpenSSH public key format's comment
    field to determine if an SSH key is past its expiration timestamp, and
    therefore no longer to be trusted. This format is still subject to change.
    Reliance on it in any way is at your own risk.
    Args:
      key: string, a single public key entry in OpenSSH public key file format.
          This will be checked for Google-specific comment semantics, and if
          present, those will be analysed.
    Returns:
      bool, True if the key has Google-specific comment semantics and has an
          expiration timestamp in the past, or False otherwise.
    """
    """
    NETSCALER begins
    """
    self.logger.debug('Processing key: %s.', (key[:50]+"**********"))
    """
    NETSCALER ends.
    """

    try:
      schema, json_str = key.split(None, 3)[2:]
    except (ValueError, AttributeError):
      self.logger.debug('No schema identifier. Not expiring key.')
      return False

    if schema != 'google-ssh':
      self.logger.debug('Invalid schema %s. Not expiring key.', schema)
      return False

    try:
      json_obj = json.loads(json_str)
    except ValueError:
      self.logger.debug('Invalid JSON %s. Not expiring key.', json_str)
      return False

    if 'expireOn' not in json_obj:
      self.logger.debug('No expiration timestamp. Not expiring key.')
      return False

    expire_str = json_obj['expireOn']
    format_str = '%Y-%m-%dT%H:%M:%S+0000'
    try:
      expire_time = datetime.datetime.strptime(expire_str, format_str)
    except ValueError:
      self.logger.warning(
          'Expiration timestamp "%s" not in format %s. Not expiring key.',
          expire_str, format_str)
      return False

    # Expire the key if and only if we have exceeded the expiration timestamp.
    return datetime.datetime.utcnow() > expire_time

  def _ParseAccountsData(self, account_data):
    """Parse the SSH key data into a user map.
    Args:
      account_data: string, the metadata server SSH key attributes data.
    Returns:
      dict, a mapping of the form: {'username': ['sshkey1, 'sshkey2', ...]}.
    """
    if not account_data:
      return {}
    lines = [line for line in account_data.splitlines() if line]
    user_map = {}
    for line in lines:
      if not all(ord(c) < 128 for c in line):
        self.logger.info('SSH key contains non-ascii character: %s.', line)
        continue
      split_line = line.split(':', 1)
      if len(split_line) != 2:
        self.logger.info('SSH key is not a complete entry: %s.', split_line)
        continue
      user, key = split_line
      if self._HasExpired(key):
        """
        NETSCALER begins
        """
        self.logger.debug('Expired SSH key for user %s: %s.', user, (key[:50]+"**********"))
        """
        NETSCALER ends.
        """
        continue
      if user not in user_map:
        user_map[user] = []
      user_map[user].append(key)
    """
    NETSCALER begins
    """
    for user in user_map:
        self.logger.debug('Total number of ssh keys: %s, for user %s:', len(user_map[user]), user)
        # Due to log file limitation, commenting it out temprorarily until we have log rotation.
        #for ssh_key in user_map[user]:
            #self.logger.debug('User accounts: %s.', (ssh_key[:50]+"**********"))
    """
    NETSCALER ends.
    """
    return user_map

  def _GetInstanceAndProjectAttributes(self, metadata_dict):
    """Get dictionaries for instance and project attributes.
    Args:
      metadata_dict: json, the deserialized contents of the metadata server.
    Returns:
      tuple, two dictionaries for instance and project attributes.
    """
    metadata_dict = metadata_dict or {}

    try:
      instance_data = metadata_dict['instance']['attributes']
    except KeyError:
      instance_data = {}
      self.logger.warning('Instance attributes were not found.')

    try:
      project_data = metadata_dict['project']['attributes']
    except KeyError:
      project_data = {}
      self.logger.warning('Project attributes were not found.')

    return instance_data, project_data

  def _GetAccountsData(self, metadata_dict):
    """Get the user accounts specified in metadata server contents.
    Args:
      metadata_dict: json, the deserialized contents of the metadata server.
    Returns:
      dict, a mapping of the form: {'username': ['sshkey1, 'sshkey2', ...]}.
    """
    instance_data, project_data = self._GetInstanceAndProjectAttributes(
        metadata_dict)
    valid_keys = [instance_data.get('sshKeys'), instance_data.get('ssh-keys')]
    block_project = instance_data.get('block-project-ssh-keys', '').lower()
    if block_project != 'true' and not instance_data.get('sshKeys'):
      valid_keys.append(project_data.get('ssh-keys'))
      valid_keys.append(project_data.get('sshKeys'))
    accounts_data = '\n'.join([key for key in valid_keys if key])
    return self._ParseAccountsData(accounts_data)

  def _UpdateUsers(self, update_users):
    """Provision and update Linux user accounts based on account metadata.
    Args:
      update_users: dict, authorized users mapped to their public SSH keys.
    """
    for user, ssh_keys in update_users.items():
      if not user or user in self.invalid_users:
        continue
      configured_keys = self.user_ssh_keys.get(user, [])
      if set(ssh_keys) != set(configured_keys):
        if not self.utils.UpdateUser(user, ssh_keys):
          self.invalid_users.add(user)
        else:
          self.user_ssh_keys[user] = ssh_keys[:]

  def _RemoveUsers(self, remove_users):
    """Deprovision Linux user accounts that do not appear in account metadata.
    Args:
      remove_users: list, the username strings of the Linux accounts to remove.
    """
    for username in remove_users:
      self.utils.RemoveUser(username)
      self.user_ssh_keys.pop(username, None)
    self.invalid_users -= set(remove_users)

  def _GetEnableOsLoginValue(self, metadata_dict):
    """Get the value of the enable-oslogin metadata key.
    Args:
      metadata_dict: json, the deserialized contents of the metadata server.
    Returns:
      bool, True if OS Login is enabled for VM access.
    """
    instance_data, project_data = self._GetInstanceAndProjectAttributes(
        metadata_dict)
    instance_value = instance_data.get('enable-oslogin')
    project_value = project_data.get('enable-oslogin')
    value = instance_value or project_value or ''

    return value.lower() == 'true'

  def HandleAccounts(self, result):
    """Called when there are changes to the contents of the metadata server.
    Args:
      result: json, the deserialized contents of the metadata server.
    """
    """
    NETSCALER begins
    """
    try:
        """
        NETSCALER ends.
        """
        self.logger.debug('Checking for changes to user accounts.')
        configured_users = self.utils.GetConfiguredUsers()
        enable_oslogin = self._GetEnableOsLoginValue(result)
        if enable_oslogin:
          desired_users = {}
          self.oslogin.UpdateOsLogin(enable=True)
        else:
          desired_users = self._GetAccountsData(result)
          self.oslogin.UpdateOsLogin(enable=False)
        remove_users = sorted(set(configured_users) - set(desired_users.keys()))
        self._UpdateUsers(desired_users)
        self._RemoveUsers(remove_users)
        self.utils.SetConfiguredUsers(desired_users.keys())
        """
        NETSCALER begins
        """
    except Exception as e:
        self.logger.error("%s"%str(e))
    finally:
        self.vip_scaling.handle_vip_changes(result)
    """
    NETSCALER ends.
    """

def main():
  parser = optparse.OptionParser()
  parser.add_option(
      '-d', '--debug', action='store_true', dest='debug',
      help='print debug output to the console.')
  (options, _) = parser.parse_args()
  instance_config = config_manager.ConfigManager()
  if instance_config.GetOptionBool('Daemons', 'accounts_daemon'):
    AccountsDaemon(
        groups=instance_config.GetOptionString('Accounts', 'groups'),
        remove=instance_config.GetOptionBool('Accounts', 'deprovision_remove'),
        useradd_cmd=instance_config.GetOptionString('Accounts', 'useradd_cmd'),
        userdel_cmd=instance_config.GetOptionString('Accounts', 'userdel_cmd'),
        usermod_cmd=instance_config.GetOptionString('Accounts', 'usermod_cmd'),
        groupadd_cmd=instance_config.GetOptionString(
            'Accounts', 'groupadd_cmd'),
        debug=bool(options.debug))

if __name__ == '__main__':
  main()
