#!/usr/bin/env python

# FILE:         $Id:$
# LAST CHECKIN: $Author:$
#               $DateTime:$
#
# Copyright 2004-2025 Citrix Systems, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to Citrix Systems, Inc.
# None o1f this software and documentation may be copied,
# duplicated or disclosed without the express
# written permission of Citrix Systems, Inc.

import sys
import os
import glob
import re

current = os.path.dirname(os.path.realpath(__file__))
parent = os.path.dirname(current)
sys.path.append(parent)

import json
import time
import datetime
import util
import util_svm

import argparse
import datetime

import constant as cons

import pexpect 

CLI_PROMPT = '>'
SHELL_PROMPT = '#'
SVM_DB_MANAGED_DEVICE_TABLE = "managed_device"
SVM_DEVICE_OWNERSHIP_COLUMN  = "device_ownership"
SVM_LOCAL_LICENSE_FILES = '/mpsconfig/license/*.*'
NOTICE_IDENTIFIER = 'NOTICE='
INTERNAL_NOTICE_VALUE = 'Citrix Systems, Inc.'
SNO_IDENTIFIER = 'SN='
ENABLE_ADM_SERVICE_CONNECT = 'set adm parameter admserviceconnect ENABLED'
SAVE_CONFIG = 'save config'

#=================For setting up logging ==============================================================================================================
import logging
import logging.handlers

log_file_name_local = os.path.basename(__file__)
LOG_FILENAME = '/var/log/' + log_file_name_local + '.log'
LOG_MAX_BYTE = 50*1024*1024
LOG_BACKUP_COUNT = 20

# Set up a specific logger with our desired output level
logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

# Add the log message handler to the logger
logger_handler = logging.handlers.RotatingFileHandler(LOG_FILENAME, maxBytes=LOG_MAX_BYTE, backupCount=LOG_BACKUP_COUNT)
logger_fortmater = logging.Formatter(fmt='%(asctime)s:%(funcName)s:%(lineno)d: [%(levelname)s] %(message)s', datefmt="%Y-%m-%d %H:%M:%S")
logger_handler.setFormatter(logger_fortmater)
logger.addHandler(logger_handler)

def main(argv):
    try:
        creds = {}
        clip = None
        svm_version,_,_ = util_svm.get_version_info()
        if (svm_version == '12.1' or svm_version == '12.0'):
            logger.debug("ADMAutoReg:: SVM is not supported {}".format(svm_version))
            sys.exit(0)
        if sdx_device_license():
            logger.debug("ADMAutoReg:: Starting internal registration daemon")
            ret_value = util_svm.add_column(SVM_DB_MANAGED_DEVICE_TABLE, SVM_DEVICE_OWNERSHIP_COLUMN)
            while 1: 
                db_values = util_svm.get_values_conditional_query(SVM_DB_MANAGED_DEVICE_TABLE, ["ip_address", "profile_name"],"where type ='ns' and instance_state='Up' and (device_ownership is null or device_ownership not ilike '%internal%')")
                if db_values != None:
                    for ip_address, profile_name in db_values:
                        clip,_ = util_svm.get_clip(ip_address)
                        creds = util_svm.get_device_profile_details(profile_name)
                        open_ssh(ip_address,creds)
                        if clip:
                            set_auto_reg(clip, creds)
                        else:
                            set_auto_reg(ip_address, creds)
                        util_svm.update_db_value_internal_flag(ip_address)
                        logger.debug("ADMAutoReg:: VPX IPAddress:{} marked successfuly".format(ip_address))
                time.sleep(3600) #Sleep for 1 hour before you check
        else:
            logger.debug("ADMAutoReg:: SVM is not internally licensed, exiting the daemon")
            sys.exit(0)

    except Exception as e:
        logger.exception("ADMAutoReg:: Failed to start Auto Registration Daemon: {}".format(str(e)))

def open_ssh(ip_address, creds):
    try:
        connect_cmd = 'ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null {}@{}'.format(creds["username"], ip_address)
        child = pexpect.spawn(connect_cmd, timeout=300)
        need_password = send_password(child, creds["password"])
        if need_password:
            child.expect([CLI_PROMPT, SHELL_PROMPT])
        logger.debug("Creating file for VPX: "+ ip_address)  
        send_cmd_shell(child, 'shell')
        send_cmd_shell(child, 'touch /nsconfig/.internal')
        send_cmd_shell(child, 'sh /var/mastools/scripts/admautoreg/admautoregd_ctl restart_if_running')
        return True
    except Exception as e: 
        logger.exception("ADMAutoReg:: Creating a file to mark license internal failed: {}".format(str(e)))
        return False

def set_auto_reg(ip_address, creds):
    try:
        connect_cmd = 'ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null {}@{}'.format(creds["username"], ip_address)
        child = pexpect.spawn(connect_cmd, timeout=300)
        need_password = send_password(child, creds["password"])
        if need_password:
            child.expect([CLI_PROMPT, SHELL_PROMPT])
        logger.debug("ADMAutoReg:: Enabling adm parameter for VPX: "+ ip_address)  
        child.sendline(ENABLE_ADM_SERVICE_CONNECT)
        child.expect(CLI_PROMPT)
        child.sendline(SAVE_CONFIG)
        child.expect(CLI_PROMPT)
        child.terminate()
    except Exception as e: 
        logger.exception("ADMAutoReg:: Enabling admserviceconnect failed: {}".format(str(e)))


def send_password(child, password):
    try:
        need_password = True
        i = child.expect(['assword:', r"yes/no", CLI_PROMPT, pexpect.EOF])
        if i == 0:
            child.sendline(password)
        elif i == 1:
            child.sendline("yes")
            child.expect("assword:", timeout=30)
            child.sendline(password)
        else:
            need_password = False
    except Exception as e:   
        logger.exception("ADMAutoReg:: Password Incorrect: {}".format(str(e)))
    return need_password
        


def send_cmd_shell(child, cmd):    
    logger.debug(cmd)
    child.sendline(cmd)
    child.expect(SHELL_PROMPT)
    logger.debug(child.before)
    logger.debug(child.after)
    return


def sdx_device_license():
    is_internal_ns_from_local_license = False
    try:
        _,is_internal_ns_from_local_license = get_local_license_notice()
    except Exception as e:
        logger.error(str(e))
    return is_internal_ns_from_local_license


# Get local license serial numbers. Each license file can
# have multiple serial numbers. Extracting all unique
# serial numbers by parsing every license file available
# on the local system.

def get_local_license_notice():
    localLicenseSno = []
    localLicenseNoticeList = []
    for licensefile in glob.glob(SVM_LOCAL_LICENSE_FILES):
        with open(licensefile, 'r') as f:
            for line in f:
                if NOTICE_IDENTIFIER in line:
                    notice_match = re.search(r"^.*%s\"(.*)\"\s.*$" % NOTICE_IDENTIFIER, line)
                    if notice_match:
                        notice = notice_match.group(1)
                        if notice not in localLicenseNoticeList:
                            localLicenseNoticeList.append(notice)
                if SNO_IDENTIFIER in line:
                    match = re.search(r"^.*%s(\S*)\s.*$" % SNO_IDENTIFIER, line)
                    if match:
                        sno = match.group(1)
                        # Only add unique sno
                        if sno not in localLicenseSno:
                            localLicenseSno.append(sno)
    if len(localLicenseNoticeList) == 0:
        # Not licensed through Local License
        is_internal_ns = None
    else:
        is_internal_ns = True
        for notice in localLicenseNoticeList:
            if notice != INTERNAL_NOTICE_VALUE:
                # Making is_internal_ns value False if it is not a Citrix test license
                is_internal_ns = False
    return localLicenseSno, is_internal_ns


if __name__ == "__main__":
    main(sys.argv)