#!/usr/bin/env python

# Copyright 2024-2025 Citrix Systems, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to Citrix Systems, Inc.
# None o1f this software and documentation may be copied,
# duplicated or disclosed without the express
# written permission of Citrix Systems, Inc.

import os
import json
import datetime

os.environ["APPLICATION_NAME"] = "ngapi_telemetry"

import util
import constant as cons
from logger import logger

try:
    from ngapi_lib.dbhelper.dbhelper import DbHelper
    from ngapi_lib.ngapictl.controller import Controller
    from ngapi_lib.secure.secure import Secure
    from ngapi_lib.utils.utils import Utils
    NEXT_GEN_API_AVAILABLE = True
except ImportError as e:
    NEXT_GEN_API_AVAILABLE = False

NGAPI_TELEMETRY_DATA_FILE = "/var/mastools/scripts/admautoreg/ngapi_telemetry_data.json"
NGAPI_VERSION_FILE = "/var/nextgen/infra/.ngapi.version"

NGAPI_STATE_ENABLED = "ENABLED"
NGAPI_STATE_DISABLED = "DISABLED"
NGAPI_STATE_INTERNAL_ERROR = "INTERNAL_ERROR"

def dump_telemetry_data(new_telemetry_data):
    # Dumping telemetry data to file so that it can be used by callhome perl script
    logger.debug("NGAPI_TELEMETRY:: Dumping telemetry data to {}".format(NGAPI_TELEMETRY_DATA_FILE))
    with open(NGAPI_TELEMETRY_DATA_FILE, "w", opener=Utils.secure_file_opener) as f:
        json.dump(new_telemetry_data, f)
    os.chmod(NGAPI_TELEMETRY_DATA_FILE, cons.FILE_PERMISSION_CODE)

def collect_telemetry():
    if not NEXT_GEN_API_AVAILABLE:
        logger.info("NGAPI_TELEMETRY:: NextGen API is not available for this build.")
        return {
            "telemetry_collected_at": datetime.datetime.now().strftime("%Y-%m-%dT%H:%M:%SZ"),
            "state": "UNAVAILABLE",
            "version": "unknown",
            "number_of_applications": -1,
            "metadata": {},
            "last_used_at": "",
        }
    logger.debug("NGAPI_TELEMETRY:: Starting ngapi_telemetry...")
    # if NGAPI_TELEMETRY_DATA_FILE exists and is modified today, return the data from the file
    if os.path.exists(NGAPI_TELEMETRY_DATA_FILE):
        today = datetime.datetime.now().date()
        file_modified = datetime.datetime.fromtimestamp(
            os.path.getmtime(NGAPI_TELEMETRY_DATA_FILE)
        ).date()
        if today == file_modified:
            with open(NGAPI_TELEMETRY_DATA_FILE, "r") as f:
                return json.load(f)
    # else, get the latest data from the database and write it to the NGAPI_TELEMETRY_DATA_FILE file
    clt = Controller(logger)
    new_telemetry_data = {
        "telemetry_collected_at": datetime.datetime.now().strftime(
            "%Y-%m-%dT%H:%M:%SZ"
        ),
        "state": "unknown",
        "version": "unknown",
        "number_of_applications": -1,
        "metadata": {},
        "last_used_at": "",
    }

    try:
        with open(NGAPI_VERSION_FILE, "r") as f:
            new_telemetry_data["version"] = f.readline().strip()
    except Exception as e:
        logger.error("NGAPI_TELEMETRY:: Error reading {} file: {}".format(NGAPI_VERSION_FILE, str(e)))

    new_telemetry_data["last_used_at"] = util.get_last_used_time(
        dir_path=cons.NS_LOG_DIR,
        filename_starting_keyword="httpaccess.log",
        search_pattern="nextgen",
    )
    logger.debug(
        "NGAPI_TELEMETRY:: last_used_at: " + new_telemetry_data["last_used_at"]
    )

    _, status_code = clt.status_ngapi()
    if status_code == clt.STOPPED_EXIT_CODE:
        logger.info("NGAPI_TELEMETRY:: NextGen API is DISABLED.")
        new_telemetry_data["state"] = NGAPI_STATE_DISABLED
    elif status_code == clt.STARTED_EXIT_CODE:
        logger.info("NGAPI_TELEMETRY:: NextGen API is ENABLED.")
        new_telemetry_data["state"] = NGAPI_STATE_ENABLED
        secobj = Secure(logger)
        db_helper = DbHelper(logger, secobj)
        filters = {"type": "APPLICATION", "count": True}
        try:
            application_data = db_helper.getall_resource(filters)
            new_telemetry_data["number_of_applications"] = application_data[0]["__count"]
        except Exception as e:
            logger.error("NGAPI_TELEMETRY:: Error getting application count: {}".format(str(e)))
    elif status_code == clt.INTERNAL_FAILURE_RETURN_CODE:
        logger.info("NGAPI_TELEMETRY:: NextGen API is in ERROR state.")
        new_telemetry_data["state"] = NGAPI_STATE_INTERNAL_ERROR
    else:
        logger.info("NGAPI_TELEMETRY:: Unexpected status code: {}".format(status_code))
        new_telemetry_data["state"] = "UNKNOWN"

    dump_telemetry_data(new_telemetry_data)
    logger.info("NGAPI_TELEMETRY:: returning data from {}".format(NGAPI_TELEMETRY_DATA_FILE))
    return new_telemetry_data

if __name__ == "__main__":
    collect_telemetry()
