#!/usr/bin/env python

#
# Copyright 2004-2025 Citrix Systems, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to Citrix Systems, Inc.
# None o1f this software and documentation may be copied,
# duplicated or disclosed without the express
# written permission of Citrix Systems, Inc.

import json
import time
import copy
import os
import random
from logger import logger
import constant as cons
import util
import util_internal_reg
from exception import ProbeError, RequestError, RequestHttpError, RequestCommunicationError, ProbeCommunicationError, ProbeRequestError, ProbeResponseError
import traceback
import threading
import ngapi_telemetry

MANAGED_DEVICE_URI='/nitro/v1/config/managed_device'

class CriticalEvent:
    def __init__(self):
        self.hddError = {}
        self.psuFailure = {}
        self.sslFailure = {}
        self.warmRestart = {}
        self.memoryAnomaly = {}

        self.__last_hddcf_pe_failure_check_time = 0
        self.__psu_failure_count = 0
        self.__last_mem_usage_check_time = 0
        self.__mem_usage_check_frequent = 0
        self.__prev_inuse_mem_per = 0
        self.__mem_increase_count = 0

    def _check_disk_pe_failure(self):
        try:
            currentTime = int(time.time())
            # Check for HDD, CF and PE failure every WAITING_TIME_HDDCF_PE_FAILURE_CHECK
            if currentTime - self.__last_hddcf_pe_failure_check_time >= cons.WAITING_TIME_HDDCF_PE_FAILURE_CHECK:
                logger.debug("Checking for HDD, CF and warm restart due to PE failure")
                hddFailure, cfFailure = util.get_disk_failure_info()
                if hddFailure or cfFailure:
                    if len(self.hddError) == 0:
                        self.hddError['error_type'] = cons.ERROR_TYPE_HDD_ERROR
                        self.hddError['utc_time'] = util.get_system_date()
                        logger.info("HDD/CF failure detected")
                else:
                    if len(self.hddError) > 0:
                        logger.info("HDD/CF failure cleared")
                        self.hddError.clear()
                    else:
                        logger.debug("No HDD failure detected")

                peCrash = util.get_pe_failure_info()
                if peCrash:
                    if len(self.warmRestart) == 0:
                        self.warmRestart['error_type'] = cons.ERROR_TYPE_WARM_RESTART
                        self.warmRestart['utc_time'] = util.get_system_date()
                        logger.info("Warm reboot due to PE crash detected")
                else:
                    if len(self.warmRestart) > 0:
                        logger.info("Warm reboot due to PE crash cleared")
                        self.warmRestart.clear()
                    else:
                        logger.debug("No warm reboot due to PE crash detected")

                self.__last_hddcf_pe_failure_check_time = currentTime
        except Exception as e:
            logger.error("Failed to check HDD, CF and warm restart due to PE failure: {}".format(str(e)))

    def _check_psu_failure(self):
        try:
            logger.debug("Checking for PSU failure")
            ps1status, ps2status, ps3status, ps4status = util.get_psu_status_info()
            if ps1status == cons.PSU_FAILED or ps2status == cons.PSU_FAILED or ps3status == cons.PSU_FAILED or ps4status == cons.PSU_FAILED:
                self.__psu_failure_count += 1
                logger.debug("PSU failure count: {}".format(self.__psu_failure_count))
            else:
                self.__psu_failure_count = 0
                logger.debug("PUS failure count set to 0")

            # Mark PSU failure if its in FAILED state for PSU_FAILED_MAX_LIMIT
            if self.__psu_failure_count >= cons.PSU_FAILED_MAX_LIMIT:
                if len(self.psuFailure) == 0:
                    self.psuFailure['error_type'] = cons.ERROR_TYPE_PSU_FAILURE
                    self.psuFailure['utc_time'] = util.get_system_date()
                    logger.info("PSU failure detected")
            else:
                if len(self.psuFailure) > 0:
                    logger.info("PSU failure cleared")
                    self.psuFailure.clear()
                else:
                    logger.debug("No PUS failure detected")
        except Exception as e:
            logger.error("Failed to check PSU failure: {}".format(str(e)))

    def _check_ssl_card_failure(self):
        try:
            logger.debug("Checking for SSL Card failure")
            totalSslCards, numSslCardsUp = util.get_ssl_card_info()
            if int(totalSslCards) != int(numSslCardsUp):
                if len(self.sslFailure) == 0:
                    self.sslFailure['error_type'] = cons.ERROR_TYPE_SSL_FAILURE
                    self.sslFailure['utc_time'] = util.get_system_date()
                    logger.info("SSL Card failure detected")
            else:
                if len(self.sslFailure) > 0:
                    logger.info("SSL Card failure cleared")
                    self.sslFailure.clear()
                else:
                    logger.debug("No SSL Card failure detected")
        except Exception as e:
            logger.error("Failed to check SSL Card failure: {}".format(str(e)))

    def _check_memory_anomaly(self):
        try:
            check = False
            currentTime = int(time.time())
            if self.__mem_usage_check_frequent == 0:
                if currentTime - self.__last_mem_usage_check_time >= cons.WAITING_TIME_MEM_USAGE_CHECK_NORMAL:
                    self.__last_mem_usage_check_time = currentTime
                    check = True
            else:
                if currentTime - self.__last_mem_usage_check_time >= cons.WAITING_TIME_MEM_USAGE_CHECK_FREQUENT:
                    self.__last_mem_usage_check_time = currentTime
                    check = True

            if check:
                logger.debug("Checking for consistent high memory usage")
                inUseMemPer = float(util.get_system_mem_usage_info())
                if inUseMemPer >= cons.MEM_MONITOR_LOW_LIMIT:
                    logger.debug("In use memory {} is above monitor low limit {}".format(inUseMemPer, cons.MEM_MONITOR_LOW_LIMIT))
                    # Switching check from once in WAITING_TIME_MEM_USAGE_CHECK_NORMAL
                    # to once in every WAITING_TIME_MEM_USAGE_CHECK_FREQUENT
                    self.__mem_usage_check_frequent += 1

                    if inUseMemPer > self.__prev_inuse_mem_per:
                        # Found increase in memory usage. Monitor for next WAITING_TIME_MEM_USAGE_CHECK_NORMAL
                        self.__mem_increase_count += 1
                        self.__prev_inuse_mem_per = inUseMemPer

                    if inUseMemPer >= cons.MEM_MONITOR_HIGH_LIMIT or (self.__mem_increase_count >= cons.MEM_INCREASE_COUNT_LIMIT and self.__mem_usage_check_frequent > cons.MEM_USAGE_CHECK_FREQUENT_LIMIT):
                        # Mark consistent high memory usage since actual in use memory has either
                        #  a. Crossed MEM_MONITOR_HIGH_LIMIT mark OR
                        #  b. Gradual increase in actual inuse memory is seen in last MEM_USAGE_CHECK_FREQUENT_LIMIT from MEM_MONITOR_LOW_LIMIT
                        # Minimum of MEM_INCREASE_COUNT_LIMIT times increase in MEM_USAGE_CHECK_FREQUENT_LIMIT in considered as gradual increase.
                        if len(self.memoryAnomaly) == 0:
                            self.memoryAnomaly['error_type'] = cons.ERROR_TYPE_MEMORY_ANOMALY
                            self.memoryAnomaly['utc_time'] = util.get_system_date()
                            logger.info("Memory anomaly detected")
                        self.__mem_usage_check_frequent = 0
                        self.__mem_increase_count = 0
                        self.__prev_inuse_mem_per = 0
                    elif self.__mem_usage_check_frequent > cons.MEM_USAGE_CHECK_FREQUENT_LIMIT:
                        logger.debug("Memory is not increasing gradually, checked for {} times".format(cons.MEM_USAGE_CHECK_FREQUENT_LIMIT))
                        # Resetting Consistent high memory usage and monitoring
                        # as the memory is not increasing gradually
                        if len(self.memoryAnomaly) > 0:
                            logger.info("Memory anomaly cleared")
                            self.memoryAnomaly.clear()
                        self.__mem_usage_check_frequent = 0
                        self.__mem_increase_count = 0
                        self.__prev_inuse_mem_per = 0
                else:
                    logger.debug("In use memory {} is below monitor low limit {}".format(inUseMemPer, cons.MEM_MONITOR_LOW_LIMIT))
                    # Resetting Consistent high memory usage as
                    # actaul in use fallen below MEM_MONITOR_LOW_LIMIT
                    if len(self.memoryAnomaly) > 0:
                        logger.info("Memory anomaly cleared")
                        self.memoryAnomaly.clear()
                    self.__mem_usage_check_frequent = 0
                    self.__mem_increase_count = 0
                    self.__prev_inuse_mem_per = 0

        except Exception as e:
            logger.error("Failed to check memory anomaly: {}".format(str(e)))

    def check_critical_events(self):
        self._check_disk_pe_failure()
        self._check_psu_failure()
        self._check_ssl_card_failure()
        self._check_memory_anomaly()

    def clear_critical_events_info(self):
        self.hddError.clear()
        self.psuFailure.clear()
        self.sslFailure.clear()
        self.warmRestart.clear()
        self.memoryAnomaly.clear()

class ProbeProxySetting:
    def __init__(self):
        self.proxyMode = False
        self.ipaddress = ""
        self.port = None

    # Refreshes callhome's proxy config:
    # - Reads ipaddress from `proxyAuthService` (by firing 'show service <proxyAuthService_name>' and using 'ipaddress' from O/P) if it is configured
    #   or from `ipaddress` field directly
    #    - Checks if the service is UP
    #   Note: Service is added for mutual SSL authentication between the NetScaler and the proxy server.
    #   Communication to proxy server is over HTTP always.
    def refresh_callhome_proxy_settings(self):
        proxyMode, ipaddress, port, proxyAuthService = util.get_callhome_proxy_info()
        self.proxyMode = proxyMode
        if self.proxyMode:
            if proxyAuthService != None and len(proxyAuthService):
                proxyAuthServiceInfo = util.get_service_info(proxyAuthService)
                if util.is_service_up(proxyAuthServiceInfo):
                    self.ipaddress = proxyAuthServiceInfo["ipaddress"]
                else:
                    # Service isn't configured or isn't up
                    raise Exception("proxyAuthService `{}` is not UP".format(proxyAuthService))
            else:
                self.ipaddress = ipaddress
            self.port = port

    def get_proxy_endpoint(self):
        if self.proxyMode:
            if self.ipaddress == None or self.port == None:
                raise Exception("ipaddress or port is None. Can't get proxy-endpoint.")
            return cons.HTTP + self.ipaddress + ":" + str(self.port)
        else:
            return None

    def get_proxies_dict(self):
        if not self.proxyMode:
            return None
        proxy_ip = self.get_proxy_endpoint()
        proxies_dict = {
            'http': proxy_ip,
            'https': proxy_ip,
        }
        return proxies_dict

class Probe:
    def __init__(self):
        self.criticalEvent = CriticalEvent()

        self.deviceInfo = {}
        self.deviceConfig = {}
        self.deviceMetrics = {}
        self.deviceLicense = {}
        self.deviceEvents = {}
        self.automation_tools = {}
        self.automation_agent_status = {"Ansible": False, "Terraform": False, "NITRO-Python": False, "NITRO-Java": False, "NITRO-Dotnet": False}
        self.__developer_conf = {}
        self.__developer_probe = {}
        self._adc_diag_data = {}
        self.ngapi_telemetry_data = {}

        self.__safehaven_endpoint = None
        self.__last_nsconf_time = 0
        self.__last_license_time = 0
        self.__last_critical_event_time = 0
        self.__last_probe_success_status = None
        self.__claimed_status = None
        self.__registration_status = None
        self.__lodestone_enable_status = None
        self.__last_automation_record_refresh_time = 0
        self.__developer_flag = False
        self.is_internal_ns = False
        self.is_internal_ns_autoreg_blocked = False
        self.internal_ns_tags = {}
        self.adm_cloud_connect_user = None
        self.adm_cloud_connect_password = None
        # NodeId is used only for cluster deployment
        self.__nodeid = ''
        self.__probe_proxy_setting = ProbeProxySetting()
        _lodestone_test_data = util.get_adc_lodestone_test_data()
        self._serialId_test = _lodestone_test_data.get(cons.ADC_LODESTONE_TEST_SERIAL_ID,"")
        self._adm_eps_test = _lodestone_test_data.get(cons.ADC_LODESTONE_TEST_EP,"")
        self.internal_ns_send_adm_svc_api = False
        self.internal_ns_send_liveliness_probe = False

    def _refresh_device_info(self):
        # Deployment Info
        try:
            deployment, ip, nodeid = util.get_deployment_info()
            self.deviceInfo['deployment'] = deployment
            if deployment == cons.CLUSTER_CCO or deployment == cons.CLUSTER_NONCCO:
                self.__nodeid = nodeid
                self.deviceInfo['clusterip'] = ip
            elif deployment == cons.HA_PRIMARY or deployment == cons.HA_SECONDARY:
                self.deviceInfo['hapeernodeip'] = ip
        except Exception as e:
            logger.error(str(e))

        # Hardware Info
        try:
            if 'serial_id' not in self.deviceInfo or self.deviceInfo['serial_id'] == "":
                serialId, encodedSerialId, hostId, netscalerUuid = util.get_hardware_info()
                if self._serialId_test:
                    serialId = self._serialId_test
                self.deviceInfo['serial_id'] = serialId
                self.deviceInfo['encoded_serial_id'] = encodedSerialId
                self.deviceInfo['host_id'] = hostId
                self.deviceInfo['netscaler_uuid'] = netscalerUuid
        except Exception as e:
            logger.error(str(e))

        # Ns Mgmt IP
        try:
            if 'mgmt_ip_address' not in self.deviceInfo or self.deviceInfo['mgmt_ip_address'] == "":
                self.deviceInfo['mgmt_ip_address'] = util.get_mgmt_ip(self.__nodeid)
        except Exception as e:
            logger.error(str(e))

        # Hostname
        try:
            self.deviceInfo['hostname'] = util.get_hostname(self.__nodeid)
        except Exception as e:
            logger.error(str(e))

        # Version Info
        try:
            if 'version' not in self.deviceInfo or self.deviceInfo['version'] == "":
                version, build, type = util.get_version_info()
                self.deviceInfo['version'] = version
                self.deviceInfo['build'] = build
                self.deviceInfo['type'] = type
        except Exception as e:
            logger.error(str(e))

        # License Info
        try:
            if 'features_licensed' not in self.deviceInfo or self.deviceInfo['features_licensed'] == "":
                featuresLicensed, licenseType = util.get_license_info()
                self.deviceInfo['features_licensed'] = featuresLicensed
                self.deviceInfo['license_type'] = licenseType
        except Exception as e:
            logger.error(str(e))

        # Hypervisor Info
        try:
            if 'hypervisor' not in self.deviceInfo or self.deviceInfo['hypervisor'] == "":
                self.deviceInfo['hypervisor'] = util.get_hypervisor_info()
        except Exception as e:
            logger.error(str(e))

        # Cloud Info
        try:
            if 'cloud' not in self.deviceInfo or self.deviceInfo['cloud'] == "":
                self.deviceInfo['cloud'] = util.get_cloud_info()
        except Exception as e:
            logger.error(str(e))

        # Platform Info
        try:
            if 'sysid' not in self.deviceInfo or self.deviceInfo['sysid'] == "":
                sysid, platformDescription, platformType = util.get_platform_info()
                self.deviceInfo['sysid'] = sysid
                self.deviceInfo['platform_description'] = platformDescription
                self.deviceInfo['platform_type'] = platformType
        except Exception as e:
            logger.error(str(e))

        # Enabled Features and Modes
        try:
            featuresEnabled, modesEnabled = util.get_enabled_features_and_modes()
            self.deviceInfo['features_enabled'] = featuresEnabled
            self.deviceInfo['modes_enabled'] = modesEnabled
        except Exception as e:
            logger.error(str(e))

    def _refresh_device_config(self):
        try:
            encodedNsconf, encoding = util.get_nsconf()
            self.deviceConfig['ns.conf'] = encodedNsconf
            self.deviceConfig['encoding'] = encoding
        except Exception as e:
            logger.error(str(e))

    def _refresh_ngapi_telemetry(self):
        try:
            self.ngapi_telemetry_data["ngapi_telemetry"] = ngapi_telemetry.collect_telemetry()
        except Exception as e:
            logger.error(str(e))

    def _refresh_device_metrics(self):
        metrics = {}
        try:
            metrics['utc_time'] = util.get_system_date()
        except Exception as e:
            logger.error(str(e))

        try:
            cpuPercentage, mgmtCpuPercentage, memoryPercentage, rxTput, systemUpTime = util.get_ns_stats()
            metrics['cpu_percentage'] = cpuPercentage
            metrics['mgmt_cpu_percentage'] = mgmtCpuPercentage
            metrics['memory_percentage'] = memoryPercentage
            metrics['rx_tput'] = rxTput
            metrics['system_up_time'] = systemUpTime
        except Exception as e:
            logger.error(str(e))

        try:
            sslNewSessions, sslEncTput, sslDecTput = util.get_ssl_stats()
            metrics['ssl_new_sessions'] = sslNewSessions
            metrics['ssl_enc_tput'] = sslEncTput
            metrics['ssl_dec_tput'] = sslDecTput
        except Exception as e:
            logger.error(str(e))

        self.deviceMetrics['metrics'] = [metrics]


    def _refresh_device_license(self):
        licenseSnoNotice = []
        is_internal_ns_from_local_license = None
        is_internal_ns_from_pooled_license = None
        try:
            localLicenseSno, is_internal_ns_from_local_license = util.get_local_license_sno_notice()
            for sno in localLicenseSno:
                info = {}
                info['lic_sno'] = sno
                info['lic_type'] = cons.LOCAL_TYPE
                licenseSnoNotice.append(info)
        except Exception as e:
            logger.error(str(e))
            logger.debug("{}".format(traceback.format_exc()))

        try:
            pooledLicenseSno, is_internal_ns_from_pooled_license = util.get_pooled_license_sno_notice(self.__nodeid)
            for sno in pooledLicenseSno:
                info = {}
                info['lic_sno'] = sno
                info['lic_type'] = cons.POOLED_TYPE
                licenseSnoNotice.append(info)
        except Exception as e:
            logger.error(str(e))
            logger.debug("{}".format(traceback.format_exc()))
        logger.info("is_internal_ns_from_local_license:{}\n is_internal_ns_from_pooled_license:{}"
                     .format(is_internal_ns_from_local_license,
                             is_internal_ns_from_pooled_license))
        if is_internal_ns_from_local_license is None and is_internal_ns_from_pooled_license is None:
            # Neither licensed by local license nor through ADM
            self.is_internal_ns = False
        elif is_internal_ns_from_pooled_license is None and is_internal_ns_from_local_license is not None:
            # Not licensed through ADM
            # Licensed by local license
            self.is_internal_ns = is_internal_ns_from_local_license
        elif is_internal_ns_from_local_license is None and is_internal_ns_from_pooled_license is not None:
            # Licensed through ADM
            # Not Licensed by local license
            self.is_internal_ns = is_internal_ns_from_pooled_license
        else:
            # Licensed by both local license and through ADM
            # If any of these licenses have a Notice which is not Citrix Systems, Inc, NS not considered as Internal
            self.is_internal_ns = is_internal_ns_from_pooled_license and is_internal_ns_from_local_license
        if os.path.isfile(cons.ADM_AUTOREG_INTERNAL_FILE):
            self.is_internal_ns = True
        util_internal_reg.set_internal_ns(self.is_internal_ns)
        if self.is_internal_ns is True:
            logger.info("Internal netscaler device.")
        self.deviceLicense['licenses'] = licenseSnoNotice

    def _refresh_device_critical_events(self):
        criticalEvent = []

        if len(self.criticalEvent.hddError) > 0:
            criticalEvent.append(self.criticalEvent.hddError)

        if len(self.criticalEvent.psuFailure) > 0:
            criticalEvent.append(self.criticalEvent.psuFailure)

        if len(self.criticalEvent.sslFailure) > 0:
            criticalEvent.append(self.criticalEvent.sslFailure)

        if len(self.criticalEvent.warmRestart) > 0:
            criticalEvent.append(self.criticalEvent.warmRestart)

        if len(self.criticalEvent.memoryAnomaly) > 0:
            criticalEvent.append(self.criticalEvent.memoryAnomaly)

        # Add events only when atleast one event is found
        if len(criticalEvent) > 0:
            self.deviceEvents['device_critical_events'] = criticalEvent

    def _is_safehaven_up(self):
        response = None
        status = False
        try:
            headers = {}
            headers['Accept'] = cons.CONTENT_TYPE_JSON
            url = cons.HTTPS + self.__safehaven_endpoint + cons.SAFEHAVEN_HEALTH_URL
            response = util.request(cons.HTTP_GET, url, headers, proxies=self.__probe_proxy_setting.get_proxies_dict())
            responseJson = response.json()
            if 'health' in responseJson:
                logger.debug("SafeHaven Health: {}".format(responseJson['health']))
                if responseJson['health'] == cons.UP:
                    status = True
            else:
                raise ProbeResponseError("SafeHaven Health response does not contain health")
        except ValueError as e:
            logger.debug("Invalid Json response from SafeHaven Health Service: {}".format(str(e)))
            raise ProbeResponseError("Invalid Json Response from SafeHaven Health Service")
        except RequestCommunicationError as e:
            logger.debug("Failed to connect to SafeHaven Health Service: {}".format(str(e)))
            raise ProbeCommunicationError("SafeHaven Health Service not reachable")
        except RequestHttpError as e:
            logger.debug("Recevied HTTP error from SafeHaven Health Service: {}:{}".format(e.message, e.status_code))
            if e.status_code == 403:
                raise e
            else:
                raise ProbeRequestError(e.message)

        return status

    # to check for developer config file if available
    def _refresh_developer_config(self):
        try:
            logger.debug("Developer Config File Check")
            self.__developer_flag, self.__developer_conf = util.get_developer_details()
            if self.__developer_flag == True:
                logger.info("Developer Config File found")
        except Exception as e:
            logger.error(str(e))

    # update the developer details:
    def _set_developer_config_in_probe(self, probe, mode):
        logger.debug("Devloper Config Update")
        dev_incorrect_key =[]
        try:
            conf_keys = self.__developer_conf.keys()
            if mode == cons.DEVELOPER_MAIN_PROBE:
                for dev_key in conf_keys:
                    # Check if the dev_key is part of the developer key list to be updated
                    if dev_key in cons.DEVELOPER_KEY_CHECK:
                        incorrect_keys =[]
                        # Check for device info
                        if dev_key == 'device_info' :
                            probe_key = probe['device_info'].keys()
                            dev_info = self.__developer_conf['device_info']
                            for k, v in dev_info.items():
                                # Update key-value pair if key is available
                                if k in probe_key:
                                    probe['device_info'][k] = v
                                    if k == 'serial_id':
                                        self.__developer_probe['serial_id'] = v
                                    if k == 'host_id':
                                        self.__developer_probe['host_id'] = v
                                else:
                                    incorrect_keys.append(k)

                            if incorrect_keys:
                                logger.error("INCORRECT KEYS in device_info: {}".format(", ".join(incorrect_keys)))

                        # Check for device events details
                        if dev_key == 'device_events':
                            probe_critical_events = []

                            if 'device_critical_events' in self.__developer_conf['device_events']:
                                # check the data in the device_events
                                dev_critical_info = self.__developer_conf['device_events']['device_critical_events']
                                dev_critical_event = [event['error_type'] for event in dev_critical_info ]

                                if dev_critical_info:
                                    # check for duplicate values and return valid event details to append or modify
                                    event_index = set() # to store unique indexes
                                    for event in dev_critical_info:
                                        event_key = set(event.keys())
                                        if event['error_type'] in cons.CRITICAL_EVENT_LIST and len(event_key) == 2 and event_key == cons.CRITICAL_EVENT_KEYS:
                                            event_index.add(len(dev_critical_event) - 1 - dev_critical_event[::-1].index(event['error_type']))
                                        else:
                                            incorrect_keys.append(event['error_type'])

                                    if 'device_events' in probe:
                                        probe_key = probe['device_events']['device_critical_events']
                                        probe_critical_events = [event['error_type'] for event in probe_key]

                                        # if value exists modify else append
                                        for dev_event_index in event_index:
                                            critical_event = dev_critical_event[dev_event_index]
                                            if critical_event  in probe_critical_events:
                                                probe_event_index = probe_critical_events.index(critical_event)
                                                probe_key[probe_event_index]['utc_time'] = dev_critical_info[dev_event_index]['utc_time']
                                            else:
                                                probe_key.append(dev_critical_info[dev_event_index])

                                        probe['device_events'] = { 'device_critical_events': probe_key}

                                    # if  no device event in probe
                                    else:
                                        probe['device_events'] = { 'device_critical_events': [dev_critical_info[dev_event_index] for dev_event_index in event_index] }

                                    if incorrect_keys:
                                        logger.error("INCORRECT KEYS in device_critical_events : {}".format(",".join(incorrect_keys)))

                                else:
                                    logger.error("DEVICE_EVENTS: No critical events found")

                            else:
                                logger.error("INCORRECT KEYS in device_events: device_critical_events key not found in developer configuration")


                        # Check for device license details
                        if dev_key == 'device_license':
                            temp_license_info = list()
                            if 'licenses' in self.__developer_conf['device_license']:
                                dev_info = self.__developer_conf['device_license']['licenses']
                                if dev_info:
                                    for dev_license in dev_info:
                                        dev_licenses_key = set(dev_license.keys())
                                        if len(dev_licenses_key) == 2 and dev_licenses_key == cons.LICENSE_KEYS and dev_license['lic_type'] in [cons.LOCAL_TYPE, cons.POOLED_TYPE] :
                                            temp_license_info.append(dev_license)
                                        else:
                                            incorrect_keys.append(dev_license)

                                    if temp_license_info:
                                        probe["device_license"] = {"licenses": temp_license_info}
                                        self.__developer_probe['licenses'] = temp_license_info

                                    if incorrect_keys:
                                        logger.error("INCORRECT License details: "+str(json.dumps(incorrect_keys)))
                                else:
                                    logger.error("KEY - VALUES in device_license: license details not found ")

                            else:
                                logger.error("INCORRECT KEY in device_license: licenses key not found in device_license")

                    else:
                        dev_incorrect_key.append(dev_key)

                if dev_incorrect_key:
                    logger.error("INCORRECT KEY in developer_config file: {}".format(",".join(dev_incorrect_key)))

            #lightprobe data update
            else:
                if self.__developer_probe:
                    for items in self.__developer_probe:
                        if items in cons.LIGHT_PROBE_KEY:
                            probe['device_info'][items] = self.__developer_probe[items]
                        else:
                            probe["device_license"] = {"licenses": self.__developer_probe[items]}


        except Exception as e:
            logger.exception("Developer Config Data Update error: {}".format(str(e)))

        logger.info("Updated Probe Data :"+str(json.dumps(probe)))
        return probe


    def refresh_safehaven_endpoint(self):
        response = None
        try:
            headers = {}
            headers['Accept'] = cons.CONTENT_TYPE_JSON
            adm_eps = cons.ADM_ENDPOINTS
            if self._adm_eps_test:
                adm_eps = self._adm_eps_test
            response = util.request(cons.HTTP_GET, adm_eps, headers, proxies=self.__probe_proxy_setting.get_proxies_dict())
            responseJson = response.json()
            if 'endpoints' in responseJson and 'registration_site' in responseJson['endpoints']:
                self.__safehaven_endpoint = responseJson['endpoints']['registration_site']
                logger.info("SafeHaven Endpoint: {}".format(self.__safehaven_endpoint))
            else:
                raise ProbeResponseError("NetScaler Console Endpoints response does not contain registration_site")
        except ValueError as e:
            logger.debug("Invalid Json response from NetScaler Console Endpoints Service: {}".format(str(e)))
            raise ProbeResponseError("Invalid Json Response from NetScaler Console Endpoints Service")
        except RequestCommunicationError as e:
            logger.debug("Failed to connect to NetScaler Console Endpoints Service: {}".format(str(e)))
            raise ProbeCommunicationError("NetScaler Console Endpoints Service not reachable")
        except RequestHttpError as e:
            logger.debug("Recevied HTTP error from NetScaler Console Endpoints Service: {}:{}".format(e.message, e.status_code))
            if e.status_code == 403:
                raise e
            else:
                raise ProbeRequestError(e.message)

    def _refresh_claimed_status(self):
        try:
            if self.__claimed_status == True and self.__registration_status == cons.SUCCESS:
                if util.check_mastools_agentconf() == False:
                    # ADC has been unclaimed by admin on NetScaler Console
                    # Reset claimed and registration status
                    self.__claimed_status = False;
                    self.__registration_status = cons.NOT_APPLICABLE;
                    util.set_claimed_state(False);
        except Exception as e:
            logger.error("Failed to refresh claimed status: {}".format(str(e)))

    def _refresh_automation_tools_events(self):
        automation_tools_record = []
        try:
            if 'automation_tools_record' in self.automation_tools:
                automation_tools_record = self.automation_tools['automation_tools_record']

            if len(automation_tools_record)!= 5:
                for user_agent, status in self.automation_agent_status.items():
                    if not status :
                        record = util.get_automation_details(user_agent)
                        if record:
                            automation_tools_record.append(record)
                            self.automation_agent_status[user_agent] = True
            self.automation_tools['automation_tools_record'] = automation_tools_record
        except Exception as e:
            logger.error("Error in getting automation details: {}".format(str(e)))
            
    def _refresh_diag_data(self):            
        try:
            logger.debug("Refresh Diag Data")
            if self.is_internal_ns and util_internal_reg.get_test_lodestone_state() is False:
                pass
            else:
                self._adc_diag_data = util.get_diag_details()
        except Exception as e:
            logger.error(str(e))

    def get_device_profile_by_name(self):
        util_internal_reg.get_agent_conf()
        url = util_internal_reg.get_device_profile_url_filter_by_name()
        status, _, response = util_internal_reg.send_request(url, 'GET')
        if status:
            response = json.loads(response)
            logger.debug("Response for get on device profile: {}".format(response))
        return response


    def add_snmp_mgr(self):
        try:
            is_cluster, is_cco = util_internal_reg.is_cluster_cco()
            is_ha, is_primary = util_internal_reg.is_ha_vpx_primary()
            if is_cluster and is_cco:
                clip = util_internal_reg.get_clip()
                util_internal_reg.add_snmp_manager(clip)
            elif is_ha and is_primary:
                mgmt_ip = self.deviceInfo['mgmt_ip_address']
                util_internal_reg.add_snmp_manager(mgmt_ip)
            elif not (is_cluster or is_ha):
                mgmt_ip = self.deviceInfo['mgmt_ip_address']
                util_internal_reg.add_snmp_manager(mgmt_ip)
        except Exception as e:
            logger.error("probe: add_snmp_mgr: Failed to add snmp manger with error: {}".format(repr(e)))

    def get_probe_data(self):
        probe = {}
        self._refresh_claimed_status()
        if self.__claimed_status == True:
            if self.is_internal_ns and util_internal_reg.get_test_lodestone_state() is False:
                probe['registration_status'] = util_internal_reg.get_internal_ns_registration_state()
            else:
                probe['registration_status'] = self.__registration_status
        self._refresh_device_info()

        if util_internal_reg.get_is_internal() is True:
            if util_internal_reg.get_update_nsip_deployment_type() is True:
                if self.deviceInfo['mgmt_ip_address'] != util_internal_reg.get_internal_nsip():
                    util_internal_reg.set_internal_ns_ip(self.deviceInfo['mgmt_ip_address'])
                is_cluster, is_cco = util_internal_reg.is_cluster_cco()
                is_ha, is_primary = util_internal_reg.is_ha_vpx_primary()
                if is_cluster:
                    util_internal_reg.set_internal_deployment_type(cons.DEPLOYMENT_TYPE_CLUSTER)
                    util_internal_reg.set_clip(util_internal_reg.get_clip())
                elif is_ha:
                    util_internal_reg.set_internal_deployment_type(cons.DEPLOYMENT_TYPE_HA)
                else:
                    util_internal_reg.set_internal_deployment_type(cons.DEPLOYMENT_TYPE_STANDALONE)
                util_internal_reg.set_update_nsip_deployment_type_to_false()

        probe['device_info'] = self.deviceInfo

        self._refresh_device_metrics()
        probe['device_metrics'] = self.deviceMetrics
        if len(self.deviceConfig) > 0 and self.__last_probe_success_status == True:
            # clearing cached device config
            self.deviceConfig = {}
            self.ngapi_telemetry_data = {}

            # clearing cached automation details
            self.automation_tools ={}
            self.automation_agent_status = {"Ansible": False, "Terraform": False, "NITRO-Python": False, "NITRO-Java": False, "NITRO-Dotnet": False}

        currentTime = int(time.time())

        # Refresh the automation user agent details every 45 min
        if currentTime - self.__last_automation_record_refresh_time >= cons.WAITING_TIME_AUTOMATION_TOOL:
            self._refresh_automation_tools_events()
            self.__last_automation_record_refresh_time = currentTime

        # Refresh device license serial no every WAITING_TIME_LIC_SNO_REFRESH
        if currentTime - self.__last_license_time >= cons.WAITING_TIME_LIC_SNO_REFRESH:
            self._refresh_device_license()
            self.__last_license_time = currentTime

        if self.is_internal_ns and util_internal_reg.get_test_lodestone_state() is False and \
                util_internal_reg.get_is_zero_touch_registration_allowed() is True:
            self.add_snmp_mgr()

        # Refresh device config every WAITING_TIME_NSCONF_REFRESH
        if currentTime - self.__last_nsconf_time >= cons.WAITING_TIME_NSCONF_REFRESH:
            if self.is_internal_ns and util_internal_reg.get_test_lodestone_state() is False and \
                    util_internal_reg.get_internal_ns_registration_state() == cons.SUCCESS:
                # For internal netscaler would be sending NetScaler Console API after one successful post of probe with ns.conf
                if util_internal_reg.check_device_type_suitable_for_reg():
                    self.internal_ns_send_adm_svc_api = True
                    logger.info("Internal netscaler, ns.conf is being sent in probe, set to send NetScaler Console API after successful post of probe.")
                self.internal_ns_send_liveliness_probe = True

            if not self.is_internal_ns:
                self._refresh_device_config()
                self._refresh_ngapi_telemetry()
            self.__last_nsconf_time = currentTime

        if len(self.deviceConfig) > 0:
            probe['device_config'] = self.deviceConfig
            if 'automation_tools_record' in self.automation_tools and self.automation_tools['automation_tools_record']:
                probe['automation_tools'] = self.automation_tools

        if len(self.ngapi_telemetry_data) > 0:
            probe["ngapi_telemetry"] = self.ngapi_telemetry_data["ngapi_telemetry"]

        if self.is_internal_ns and util_internal_reg.get_test_lodestone_state() is False:
            if not util_internal_reg.check_if_admcloudconnectuser_present() and \
                    util_internal_reg.get_internal_ns_registration_state() == cons.SUCCESS and \
                    util_internal_reg.get_is_zero_touch_registration_allowed() is True:
                # Implies that user might have done a clear config
                # agent.conf will be present
                try:
                    new_username, new_password = util_internal_reg.update_device_profile_save_config()
                    self.adm_cloud_connect_user = new_username
                    self.adm_cloud_connect_password = new_password
                except Exception as e:
                    logger.error("Update of device profile failed with error:{}".format(str(e)))
                    logger.error(traceback.format_exc())
                if not probe.get('internal_device_info'):
                    adm_ns_unique_id = util_internal_reg.get_adm_ns_unique_id()
                    is_cluster, is_cco = util_internal_reg.is_cluster_cco()
                    is_ha, is_primary = util_internal_reg.is_ha_vpx_primary()
                    if is_cluster and not is_cco:
                        adm_ns_unique_id = None
                    elif is_ha and not is_primary:
                        adm_ns_unique_id = None
                    deployment_type = util_internal_reg.fetch_type_of_deployment()
                    cloud_details = util_internal_reg.fetch_cloud_info()
                    probe['internal_device_info'] = {
                        'deployment_type': deployment_type,
                        'cloud_details': cloud_details,
                        'ns_unique_id': adm_ns_unique_id
                    }
            else:
                try:
                    if util_internal_reg.get_internal_ns_registration_state() == cons.INPROGRESS:
                        is_timed_out = util_internal_reg.is_registartion_in_progress_timedout()
                        if is_timed_out:

                            logger.info("Internal NS auto registration attempt 1 hour timed out, "
                                        "resetting status to re-trigger registration")
                            util_internal_reg.set_internal_ns_registration_state(cons.NOT_APPLICABLE)
                            util_internal_reg.delete_device_profile()

                    if util_internal_reg.is_internal_registration_required(self.is_internal_ns, self.deviceInfo['mgmt_ip_address']):
                        logger.info("NetScaler is internal and auto-registration is required")
                        if not probe.get('internal_device_info'):
                            username, password = util_internal_reg.add_cloud_connect_user()
                            deployment_type = util_internal_reg.fetch_type_of_deployment()
                            cloud_details = util_internal_reg.fetch_cloud_info()
                            self.internal_ns_tags = cloud_details
                            adm_ns_unique_id = util_internal_reg.get_adm_ns_unique_id()
                            is_cluster, is_cco = util_internal_reg.is_cluster_cco()
                            is_ha, is_primary = util_internal_reg.is_ha_vpx_primary()
                            if is_cluster and not is_cco:
                                adm_ns_unique_id = None
                            elif is_ha and not is_primary:
                                adm_ns_unique_id = None
                            self.adm_cloud_connect_user = username
                            self.adm_cloud_connect_password = password
                            probe['internal_device_info'] = {
                                'username': username,
                                'password': password,
                                'deployment_type': deployment_type,
                                'cloud_details': cloud_details,
                                'ns_unique_id': adm_ns_unique_id
                            }
                            if is_cluster and is_cco:
                                clip_and_member_node_ips_list = util_internal_reg.get_clip_and_member_node_ips(get_active_nodes=False)
                                comma_separator = ","
                                clip_and_member_node_ips_str = comma_separator.join(clip_and_member_node_ips_list)
                                probe['internal_device_info']['clip_and_member_node_ips'] = clip_and_member_node_ips_str
                    else:
                        if not probe.get('internal_device_info'):
                            adm_ns_unique_id = util_internal_reg.get_adm_ns_unique_id()
                            is_cluster, is_cco = util_internal_reg.is_cluster_cco()
                            is_ha, is_primary = util_internal_reg.is_ha_vpx_primary()
                            if is_cluster and not is_cco:
                                adm_ns_unique_id = None
                            elif is_ha and not is_primary:
                                adm_ns_unique_id = None
                            deployment_type = util_internal_reg.fetch_type_of_deployment()
                            cloud_details = util_internal_reg.fetch_cloud_info()
                            probe['internal_device_info'] = {
                                'deployment_type': deployment_type,
                                'cloud_details': cloud_details,
                                'ns_unique_id': adm_ns_unique_id
                            }

                except Exception as e:
                    logger.error("Probe payload preparation for internal_device_info failed with error:{}".format(str(e)))
                    logger.error("{}".format(traceback.format_exc()))

        probe['device_license'] = self.deviceLicense
        
        if util.IS_PLATFORM_SDX ==  False:
            # Check for critical events
            self.criticalEvent.check_critical_events()
            if len(self.deviceEvents) > 0 and self.__last_probe_success_status == True:
                # clearing cached device events
                self.deviceEvents.clear()
                # Clear all previous events
                self.criticalEvent.clear_critical_events_info()
            # Include Critical events only once WAITING_TIME_CRITICAL_EVENTS
            if currentTime - self.__last_critical_event_time >= cons.WAITING_TIME_CRITICAL_EVENTS:
                self._refresh_device_critical_events()
                self.__last_critical_event_time = currentTime

            if len(self.deviceEvents) > 0:
                probe['device_events'] = self.deviceEvents

        self._refresh_diag_data()
        if self._adc_diag_data:
            probe[cons.ADC_DIAG_KEYWORD] = self._adc_diag_data
        las_info = util.extract_las_license_parameters()
        if las_info:
            probe.update(las_info)
        return probe

    def get_dry_run_status(self):
        dry_run_status = {}
        if not os.path.isfile(cons.DRY_RUN_STATUS_FILE):
            return dry_run_status
        try:
            with open(cons.DRY_RUN_STATUS_FILE) as data_file:
                dry_run_status = json.load(data_file)
        except Exception as e:
            print(e)

        return dry_run_status
        
    def get_lightprobe_data(self):
        lightprobe = {}

        self._refresh_claimed_status()
        if self.__claimed_status == True:
            if self.is_internal_ns and util_internal_reg.get_test_lodestone_state() is False:
                lightprobe['registration_status'] = util_internal_reg.get_internal_ns_registration_state()
            else:
                lightprobe['registration_status'] = self.__registration_status
        if self.is_internal_ns and util_internal_reg.get_test_lodestone_state() is False:
            lightprobe["is_internal_ns"] = True
        device_info = {}
        device_info['serial_id'] = self.deviceInfo['serial_id']
        device_info['host_id'] = self.deviceInfo['host_id']
        lightprobe['device_info'] = device_info
        lightprobe['device_license'] = self.deviceLicense
        

        return lightprobe

    def mastools_dry_run(self, dryrunDetails):
        registrationStatus = None
        try:
            deployment = self.deviceInfo['deployment']
            if deployment == cons.STANDALONE or deployment == cons.HA_PRIMARY or deployment == cons.CLUSTER_CCO:
                util.start_dry_run(dryrunDetails['device_profile_name'], dryrunDetails['registration_endpoint'], dryrunDetails['registration_secret'], dryrunDetails['uuid'])
            else:
                logger.error("Dry run rejected, trying to claim {} node".format(deployment))
        except Exception as e:
            logger.exception("Dry run error: {}".format(str(e)))

        return

    def adm_registration(self, claimDetails):
        registrationStatus = None
        try:
            deployment = self.deviceInfo['deployment']
            if deployment == cons.STANDALONE or deployment == cons.HA_PRIMARY or deployment == cons.CLUSTER_CCO:
                if claimDetails['adm_deployment'].lower() == 'cloud' and claimDetails['agent_type'].lower() == 'builtin':
                    registrationStatus = util.mastools_registration(claimDetails['device_profile_name'], claimDetails['registration_endpoint'], claimDetails['registration_secret'])
                else:
                    registrationStatus = util.cli_registration(claimDetails['adm_deployment'], claimDetails['adm_username'], claimDetails['adm_password'], claimDetails['registration_endpoint'], claimDetails['device_profile_name'])
            elif util.IS_PLATFORM_SDX:
                registrationStatus = util.mastools_registration(claimDetails['device_profile_name'], claimDetails['registration_endpoint'], claimDetails['registration_secret'])
            else:
                logger.error("Registration rejected, trying to claim {} node".format(deployment))
                registrationStatus = cons.FAILURE
        except Exception as e:
            logger.exception("Registration error: {}".format(str(e)))
            registrationStatus = cons.FAILURE

        logger.debug("Registration status: {}".format(registrationStatus))
        return registrationStatus

    def track_and_update_status_of_internal_autoreg(self):
        status = util_internal_reg.track_registration_on_adm(self.deviceInfo['mgmt_ip_address'])
        if status:
            # update tags
            util_internal_reg.update_tags_in_managed_device(self.internal_ns_tags)
            util_internal_reg.set_internal_ns_registration_state(cons.SUCCESS)
            logger.info("Internal NS registration status is SUCCESS.")
            util_internal_reg.set_internal_ns_registration_start_time(cons.NOT_APPLICABLE)
            util_internal_reg.set_internal_ns_ip(self.deviceInfo['mgmt_ip_address'])
            is_cluster, _ = util_internal_reg.is_cluster_cco()
            is_ha, _ = util_internal_reg.is_ha_vpx_primary()
            clip = None
            if is_cluster:
                current_deployment_type = cons.DEPLOYMENT_TYPE_CLUSTER
                clip = util_internal_reg.get_clip()
            elif is_ha:
                current_deployment_type = cons.DEPLOYMENT_TYPE_HA
            else:
                current_deployment_type = cons.DEPLOYMENT_TYPE_STANDALONE
            util_internal_reg.set_internal_deployment_type(current_deployment_type)
            if clip:
                util_internal_reg.set_clip(clip)
            else:
                util_internal_reg.set_clip(cons.NOT_APPLICABLE)

            if self.adm_cloud_connect_user and self.adm_cloud_connect_password:
                util_internal_reg.sync_admautreg_config_file_for_ha_cluster(self.adm_cloud_connect_user,
                                                                            self.adm_cloud_connect_password)
        else:
            util_internal_reg.set_internal_ns_registration_state(cons.FAILURE)
            logger.info("Internal NS registration status is FAILURE.")
            if self.adm_cloud_connect_user and self.adm_cloud_connect_password:
                util_internal_reg.sync_admautreg_config_file_for_ha_cluster(self.adm_cloud_connect_user,
                                                                            self.adm_cloud_connect_password)

            # deleting device profile if registration has failed
            util_internal_reg.delete_device_profile()

    def send_api_to_adm_svc(self):
        waiting_time = random.randint(cons.INTERNAL_NS_ADM_SVC_API_WAIT_TIME_MIN,
                                      cons.INTERNAL_NS_ADM_SVC_API_WAIT_TIME_MAX)
        logger.info("Sending NetScaler Console managed_device API after {} seconds...".format(waiting_time))
        time.sleep(waiting_time)
        util_internal_reg.get_agent_conf()
        try:
            is_ha, is_primary = util_internal_reg.is_ha_vpx_primary()
            is_cluster, is_cco = util_internal_reg.is_cluster_cco()
            trust_id = util_internal_reg.get_instanceid()
            managed_device_url = util_internal_reg.get_managed_device_url_by_trust_id(trust_id)
            status, response_code, response = util_internal_reg.send_request(managed_device_url, "GET", time_out=120)
            if status:
                response = json.loads(response)
                if response and "managed_device" in response:
                    if len(response["managed_device"]) == 0:
                        util_internal_reg.set_is_zero_touch_registration_allowed_to_false()
                        util_internal_reg.set_internal_ns_registration_state(cons.NOT_APPLICABLE)
                        logger.info(
                            "Get on managed_device response gives empty, managed_device, modified internal registration status to NA")
                    else:
                        managed_device_ids = []
                        for device in response["managed_device"]:
                            managed_device_ids.append(device["id"])
                        for device in response["managed_device"]:
                            if (device['instance_state'] == 'Out of Service') and \
                                    ((is_ha and is_primary and device['instance_mode'] == "Primary") or
                                     (is_cluster and is_cco and device['instance_mode'] == "Clip") or
                                     (not is_cluster and not is_ha)):
                                logger.info(
                                   "Get on managed_device instance is Out of Service, instance_state: "
                                   "{}, instance_mode: {}, "
                                   "ip_address: {}, creating new user and updating device profile"
                                   "".format(device['instance_state'], device['instance_mode'], device['ip_address']))
                                self.adm_cloud_connect_user, self.adm_cloud_connect_password = util_internal_reg.update_device_profile_save_config()
                else:
                    util_internal_reg.set_is_zero_touch_registration_allowed_to_false()
                    util_internal_reg.set_internal_ns_registration_state(cons.NOT_APPLICABLE)
                    logger.info("Get on managed_device incorrect response, modified internal registration status to NA")
            else:
                if response_code and (response_code == '403' or response_code == '401'):
                    util_internal_reg.set_is_zero_touch_registration_allowed_to_false()
                    util_internal_reg.set_internal_ns_registration_state(cons.NOT_APPLICABLE)
                    logger.info(
                        "Get on managed_device returned response code: {}, modified internal registration status to NA".format(
                            response_code))
        except Exception as e:
            logger.info("Get on managed_device failed {}-{}".format(repr(e), traceback.format_exc()))

    def send_probe(self):
        if util_internal_reg.get_is_internal() is True:
            # add nameserver if doesn't exist for internal onprem netscalers
            if util_internal_reg.fetch_type_of_deployment() == cons.ONPREM_CLOUD_TYPE and util_internal_reg.get_test_lodestone_state() is False:
                logger.info("Adding nameserver 8.8.8.8 if not already exists.")
                util_internal_reg.add_nameserver()

            # If agent.conf is missing, re-trigger registration
            util_internal_reg.mark_for_re_registration_if_agent_conf_not_found()

        if util.IS_PLATFORM_SDX ==  False:
            if util_internal_reg.get_is_internal() is True:
                try:
                    self.__probe_proxy_setting.refresh_callhome_proxy_settings()
                except Exception as e:
                    logger.error("Internal netscaler: refresh_callhome_proxy_settings has failed with error: {}".format(repr(e)))
            else:
                self.__probe_proxy_setting.refresh_callhome_proxy_settings()
        response = None
        try:
            if self.__safehaven_endpoint == None:
                self.refresh_safehaven_endpoint()
            if self.__last_probe_success_status == False:
                if self._is_safehaven_up() == False:
                    raise ProbeCommunicationError("SafeHaven Service is Down")
            if self.__claimed_status == None:
                self.__claimed_status = util.get_claimed_state()
                if self.__registration_status == None:
                    self.__registration_status = util.get_registration_state()
            probe = self.get_probe_data()
            # Check for developer config file
            self._refresh_developer_config()
            if self.__developer_flag == True:
                probe = self._set_developer_config_in_probe(copy.deepcopy(probe), cons.DEVELOPER_MAIN_PROBE)

            # Setting the internal NS registration status to InProgress just before sending the probe
            if "internal_device_info" in probe and "username" in probe["internal_device_info"] \
                    and "password" in probe["internal_device_info"]:
                util_internal_reg.remove_mastools_init_status()
                util_internal_reg.remove_prev_inst()
                if util_internal_reg.check_if_nsip_is_changed(self.deviceInfo['mgmt_ip_address']):
                    logger.info("Internal netscaler IP has been changed.")
                    # IP has been changed, need to pass old IP in payload to remove from managed_device
                    probe["internal_device_info"]["old_ip_address"] = util_internal_reg.get_internal_nsip()
                    util_internal_reg.get_agent_conf()
                    profile_name_from_agent_conf = util_internal_reg.get_profile_name()
                    if profile_name_from_agent_conf:
                        probe['internal_device_info']["device_profile_name"] = profile_name_from_agent_conf
                is_cluster, _ = util_internal_reg.is_cluster_cco()
                is_ha, _ = util_internal_reg.is_ha_vpx_primary()
                if is_cluster:
                    current_deployment_type = cons.DEPLOYMENT_TYPE_CLUSTER
                elif is_ha:
                    current_deployment_type = cons.DEPLOYMENT_TYPE_HA
                else:
                    current_deployment_type = cons.DEPLOYMENT_TYPE_STANDALONE
                deployment_type_in_config_file = util_internal_reg.get_internal_deployment_type()
                if deployment_type_in_config_file != cons.NOT_APPLICABLE and \
                        current_deployment_type != deployment_type_in_config_file:
                    if deployment_type_in_config_file == cons.DEPLOYMENT_TYPE_CLUSTER:
                        clip = util_internal_reg.get_clip_from_config_file()
                        if clip and clip != cons.NOT_APPLICABLE:
                            probe["internal_device_info"]["old_clip_ip_address"] = clip

                util_internal_reg.set_internal_ns_registration_state(cons.INPROGRESS)
                util_internal_reg.set_internal_ns_registration_start_time()
                if self.adm_cloud_connect_user and self.adm_cloud_connect_password:
                    util_internal_reg.sync_admautreg_config_file_for_ha_cluster(self.adm_cloud_connect_user, self.adm_cloud_connect_password)

            if self.is_internal_ns and util_internal_reg.get_is_zero_touch_registration_allowed()==True \
                    and util_internal_reg.get_internal_ns_registration_state() == cons.SUCCESS and not self.internal_ns_send_liveliness_probe:
                logger.info("Not sending probe as it is already registered.")
            else:
                if self.is_internal_ns and util_internal_reg.fetch_type_of_deployment() == 0 and util_internal_reg.get_test_lodestone_state() is False:
                    otp_generated = util_internal_reg.generate_otp(self.deviceInfo['mgmt_ip_address'])
                    probe["internal_device_info"]["otp"] = otp_generated
                    probe_payload = copy.deepcopy(probe)
                    if "internal_device_info" in probe_payload:
                        if "password" in probe_payload["internal_device_info"]:
                            del probe_payload["internal_device_info"]["password"]
                        if "otp" in probe_payload["internal_device_info"]:
                            del probe_payload["internal_device_info"]["otp"]
                    logger.info("Probe: {}".format(json.dumps(probe_payload)))
                headers = {}
                headers['Content-Type'] = cons.CONTENT_TYPE_JSON
                url = cons.HTTPS + self.__safehaven_endpoint + cons.SAFEHAVEN_HEARTBEAT_URL
                response = util.request(cons.HTTP_POST, url, headers, json.dumps(probe), proxies=self.__probe_proxy_setting.get_proxies_dict())
                if response.text != None:
                    logger.info("Probe Response: {}".format(response.text))
                self.__last_probe_success_status = True
                # Check in the probe response for the lodestoneenable status
                responseJson = response.json()
                if 'probe' in responseJson and 'safehaven_internal_ns_auto_reg_state' in responseJson['probe'] and\
                        responseJson['probe']['safehaven_internal_ns_auto_reg_state'] == cons.DISABLED:
                    util_internal_reg.set_internal_ns_registration_state(cons.DISABLED)
                    util_internal_reg.set_internal_ns_registration_start_time(cons.NOT_APPLICABLE)
                    if self.adm_cloud_connect_user and self.adm_cloud_connect_password:
                        util_internal_reg.sync_admautreg_config_file_for_ha_cluster(self.adm_cloud_connect_user, self.adm_cloud_connect_password)
                    # delete the cloud connect user
                    self.adm_cloud_connect_user = None
                    self.adm_cloud_connect_password = None
                elif 'probe' in responseJson and 'lodestone_enabled' in responseJson['probe']:
                    self.__lodestone_enable_status = responseJson['probe']['lodestone_enabled']
                    logger.info("ADC Lodestone Enabled state: {}".format(self.__lodestone_enable_status))
                    util.set_lodestone_enable_state(self.__lodestone_enable_status)


            # This if will be executed id last probe with ns.conf is successful
            if self.is_internal_ns:
                if self.internal_ns_send_liveliness_probe is True:
                    self.internal_ns_send_liveliness_probe = False
                if self.internal_ns_send_adm_svc_api is True and util_internal_reg.get_internal_ns_registration_state() == cons.SUCCESS:
                    thread_send_api_to_adm_svc = threading.Thread(target=self.send_api_to_adm_svc)
                    thread_send_api_to_adm_svc.start()
                    self.internal_ns_send_adm_svc_api = False

        except ValueError as e:
            logger.debug("Invalid Json response from SafeHaven Service: {}".format(str(e)))
            raise ProbeResponseError("Invalid Json Response from SafeHaven Service")
        except RequestCommunicationError as e:
            self.__last_probe_success_status = False
            logger.debug("Failed to connect to SafeHaven Service: {}".format(str(e)))
            raise ProbeCommunicationError("SafeHaven Service not reachable")
        except RequestHttpError as e:
            self.__last_probe_success_status = False
            logger.debug("Recevied HTTP error from SafeHaven Service: {}:{}".format(e.message, e.status_code))
            # Do not raise ProbeRequestError exception for error code 403.
            # SafeHaven will return error code 403 if its gets too many
            # requests from one IP within a certain time period.
            if e.status_code != 403:
                raise ProbeRequestError(e.message)
            else:
                logger.error("Recevied 403 error from SafeHaven Service")

    def send_lightprobe(self):
        claimedInCurrProbe = False
        try:
            lightprobe = self.get_lightprobe_data()

            if self.__developer_flag == True:
                lightprobe = self._set_developer_config_in_probe(copy.deepcopy(lightprobe), cons.DEVELOPER_LIGHT_PROBE)
                
            dry_run_status =  self.get_dry_run_status()   

            if dry_run_status: 
                lightprobe['dry_run_status'] = dry_run_status
            logger.debug("Light Probe: {}".format(json.dumps(lightprobe)))
            headers = {}
            headers['Content-Type'] = cons.CONTENT_TYPE_JSON
            url = cons.HTTPS + self.__safehaven_endpoint + cons.SAFEHAVEN_LIGHTHEARTBEAT_URL
            response = util.request(cons.HTTP_POST, url, headers, json.dumps(lightprobe), proxies=self.__probe_proxy_setting.get_proxies_dict())
            if response.text != None:
                logger.debug("Light Probe Response: {}".format(response.text))
            responseJson = response.json()

            if self.is_internal_ns and util_internal_reg.get_test_lodestone_state() is False and util_internal_reg.check_device_type_suitable_for_reg():
                # Getting registration status for internal NS auto registration
                try:
                    if "is_blocked" in responseJson['probe']['claim_details'] and responseJson['probe']['claim_details']['is_blocked']:
                        self.is_internal_ns_autoreg_blocked = True
                        util_internal_reg.set_internal_ns_registration_state(cons.BLOCKED)
                        util_internal_reg.set_internal_ns_registration_start_time(cons.NOT_APPLICABLE)
                        util_internal_reg.set_internal_ns_ip(cons.NOT_APPLICABLE)
                    if "ns_unique_id" in responseJson['probe']['claim_details'] and responseJson['probe']['claim_details']['ns_unique_id']:
                        util_internal_reg.set_adm_ns_unique_id(responseJson['probe']['claim_details']['ns_unique_id'])
                except Exception:
                    pass
            try:
                safehaven_internal_reg_status = responseJson['probe']['safehaven_internal_ns_auto_reg_state']
            except Exception:
                safehaven_internal_reg_status = ''
            if self.is_internal_ns and util_internal_reg.get_internal_ns_registration_state() == cons.DISABLED and safehaven_internal_reg_status == cons.ENABLED:
                util_internal_reg.set_internal_ns_registration_state(safehaven_internal_reg_status)
            is_registration_allowed = False
            if 'probe' in responseJson and 'claimed' in responseJson['probe']:
                if dry_run_status:
                    util.remove_dry_run_status()
                self.__claimed_status = responseJson['probe']['claimed']
                logger.info("ADC Claimed state: {}".format(self.__claimed_status))
                util.set_claimed_state(self.__claimed_status)
                if self.__claimed_status == True and 'claim_details' in  responseJson['probe']:
                    if not self.is_internal_ns or util_internal_reg.get_test_lodestone_state() is True:
                        # Normal mastools registration
                        self.__registration_status = self.adm_registration(responseJson['probe']['claim_details'])
                        claimedInCurrProbe = True
                        util.set_registration_state(self.__registration_status)
                        if self.__registration_status == cons.SUCCESS:
                            logger.info("Successfully registered to NetScaler Console: {}".format(
                                responseJson['probe']['claim_details']['registration_endpoint']))
                        else:
                            logger.error("Failed to Register to NetScaler Console: {}".format(
                                responseJson['probe']['claim_details']['registration_endpoint']))
                    else:
                        # Internal auto registration
                        if not self.is_internal_ns_autoreg_blocked:
                            is_registration_allowed = responseJson['probe']['claim_details'].get('registration_allowed', False)
                            if is_registration_allowed and responseJson['probe']['claim_details'].get('registration_endpoint')\
                                    and responseJson['probe']['claim_details'].get('registration_secret') \
                                    and responseJson['probe']['claim_details'].get('device_profile_name'):
                                self.__registration_status = self.adm_registration(responseJson['probe']['claim_details'])
                                if not self.internal_ns_tags:
                                    self.internal_ns_tags = util_internal_reg.fetch_cloud_info()
                                self.internal_ns_tags["owner_email_id"] = responseJson['probe']['claim_details'].get('owner_email_id', "")
                                self.internal_ns_tags["business_unit"] = responseJson['probe']['claim_details'].get('business_unit', "")
                                self.internal_ns_tags["manager_email_id"] = responseJson['probe']['claim_details'].get('manager_email_id', "")
                                self.internal_ns_tags["force_upgrades"] = "allowed"
                                self.internal_ns_tags["comments"] = "na"
                                claimedInCurrProbe = True
                                util.set_registration_state(self.__registration_status)
                                if self.__registration_status == cons.SUCCESS:
                                    logger.info("Successfully registered to NetScaler Console: {}".format(
                                        responseJson['probe']['claim_details']['registration_endpoint']))
                                    thread_track_internal_autoreg = threading.Thread(target=self.track_and_update_status_of_internal_autoreg)
                                    thread_track_internal_autoreg.start()
                                else:
                                    logger.error("Failed to Register to NetScaler Console: {}".format(
                                        responseJson['probe']['claim_details']['registration_endpoint']))
                                util_internal_reg.set_is_zero_touch_registration_allowed_to_true()
                            elif not is_registration_allowed :
                                logger.info("Internal Auto-registration is not allowed.")
                                if util_internal_reg.get_is_zero_touch_registration_allowed() == True:
                                    util_internal_reg.set_is_zero_touch_registration_allowed_to_false()
                                if  util_internal_reg.get_clean_up():
                                    logger.info("Cleaning up ...")
                                    # cleanup if registration is not allowed
                                    util_internal_reg.cleanup_internal_auto_registration()
                                    util_internal_reg.set_clean_up_to_false()
                            elif is_registration_allowed:
                                if util_internal_reg.get_is_zero_touch_registration_allowed() == False:
                                    util_internal_reg.set_is_zero_touch_registration_allowed_to_true()

                elif ('probe' in responseJson) and ('dry_run_details' in responseJson['probe']) and ('action' in responseJson['probe']['dry_run_details']):
                    if responseJson['probe']['dry_run_details']['action'] == 'start':
                        logger.info("start dry run")
                        self.mastools_dry_run(responseJson['probe']['dry_run_details'])
            else:
                raise ProbeResponseError("Light Probe response does not contain claimed status")

                    
        except ValueError as e:
            logger.debug("Invalid Json response from SafeHaven Service: {}".format(str(e)))
            raise ProbeResponseError("Invalid Json Response from SafeHaven Service")
        except RequestCommunicationError as e:
            logger.debug("Failed to connect to SafeHaven Service: {}".format(str(e)))
            raise ProbeCommunicationError("SafeHaven Service not reachable")
        except RequestHttpError as e:
            logger.debug("Recevied HTTP error from SafeHaven Service: {}:{}".format(e.message, e.status_code))
            # Do not raise ProbeRequestError exception for error code 403.
            # SafeHaven will return error code 403 if its gets too many
            # requests from one IP within a certain time period.
            if e.status_code != 403:
                raise ProbeRequestError(e.message)
            else:
                logger.error("Recevied 403 error from SafeHaven Service")

        return claimedInCurrProbe
