#!/usr/bin/env python

# Copyright 2004-2025 Citrix Systems, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to Citrix Systems, Inc.
# None o1f this software and documentation may be copied,
# duplicated or disclosed without the express
# written permission of Citrix Systems, Inc.

import subprocess
import shlex
import re
import zlib
import base64
import os
import json
import pexpect
from shutil import copy2
from datetime import datetime
try:
    import ConfigParser
except ImportError:
    import configparser as ConfigParser
import logging
import util_svm
import util_internal_reg
import requests
import time
import fileinput
import glob
import socket
import platform
import string
import sys
import random
try:
    from ctypes import *
except:
    pass

from logger import logger
import constant as cons
from exception import RequestError, RequestHttpError, RequestCommunicationError

SVM_MPS_LIB = '/mps/lib'

def is_python_3_running():
    if platform.python_version().startswith('3.'):
        return True
    return False
    
is_python_3 = is_python_3_running()    

def is_python_2_6_running():
    if platform.python_version().startswith('2.6'):
        return True
    return False


is_python_2_6 = is_python_2_6_running()


def is_platform_linux():
    uname_cmd = 'uname -a'
    try:
        args = shlex.split(uname_cmd)
        process = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=None)
        output = process.communicate()
        unameStr = output[0].decode()
        if 'linux' in unameStr.lower():
            return True
    except Exception as e:
        logger.error("is_platform_linux Exception:"+repr(e))
    return False

IS_PLATFORM_LINUX = is_platform_linux()       
    

def sysctl_exec(input):
    #sysctl is not supported on Linux VPX/BLX
    if IS_PLATFORM_LINUX:
        return ''
    cmd = cons.SYSCTL_CMD + ' ' + input
    sysctlOutput = subprocess.check_output(shlex.split(cmd))
    try:
        if is_python_3:
            sysctlOutput = sysctlOutput.decode("utf-8")
    except AttributeError:
        pass  
    return sysctlOutput.strip()

def is_platform_sdx():
    if os.path.exists(SVM_MPS_LIB):
        return True
    return False

IS_PLATFORM_SDX = is_platform_sdx()

def get_autoreg_state_file():
    if IS_PLATFORM_SDX:
        return cons.SVM_ADM_AUTOREG_STATE_FILE
    else:
        return cons.ADM_AUTOREG_STATE_FILE
        

def get_local_license_file():
    if IS_PLATFORM_SDX:
        return cons.SVM_LOCAL_LICENSE_FILES
    else:
        return cons.ADC_LOCAL_LICENSE_FILES

def set_log_level(level):
    if level == 'debug':
        logger.setLevel(logging.DEBUG)
    else:
        logger.setLevel(logging.INFO)

def init_autoreg_state(is_internal_ns=False):
    initStateFile = False
    initReachableFlag, initClamedFlag, initRegistrationFlag, initLodestoneEnableFlag = False, False, False, False
    initInternalNSRegistrationFlag, initInternalNSRegistrationStartTime, initInternalNSRegistrationNSIP, initIsInternal = False, False, False, False,
    initCleanUp = False
    initNSUniqueID = False
    initTestLoadestone = False
    initIsZeroTouchRegistrationAllowed = False
    initDeploymentType = False
    initUpdateNSIPDeploymentType = False
    initClip = False
    state = ConfigParser.RawConfigParser()
    stateFilePath = get_autoreg_state_file()
    if os.path.isfile(stateFilePath) == True:
        try:
            state.read(stateFilePath)
            if state.has_section(cons.ADM_AUTOREG_SECTION) == True:
                if state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_SAFEHAVEN_REACHABLE_FLAG) == False:
                    initReachableFlag = True
                if state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_ADCCLAIMED_FLAG) == False:
                    initClamedFlag = True
                if state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_ADCREGISTRATION_FLAG) == False:
                    initRegistrationFlag = True
                if state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_LODESTONE_ENABLE_FLAG) == False:
                    initLodestoneEnableFlag = True
                state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_IS_NS_INTERNAL, is_internal_ns)
                if is_internal_ns:
                    if not state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_FLAG):
                        initInternalNSRegistrationFlag = True
                    if not state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_START_TIME):
                        initInternalNSRegistrationStartTime = True
                    if not state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_NSIP):
                        initInternalNSRegistrationNSIP = True
                    if not state.has_option(cons.ADM_AUTOREG_SECTION, cons.TEST_LODESTONE):
                        initTestLoadestone = True
                    if state.has_option(cons.ADM_AUTOREG_SECTION, cons.UPDATE_DEVICE_PROFILE):
                        state.remove_option(cons.ADM_AUTOREG_SECTION, cons.UPDATE_DEVICE_PROFILE)
                    if state.has_option(cons.ADM_AUTOREG_SECTION, cons.DELETE_CORE_DUMP):
                        state.remove_option(cons.ADM_AUTOREG_SECTION, cons.DELETE_CORE_DUMP)
                    if not state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_UNIQUE_ID):
                        initNSUniqueID = True
                    if not state.has_option(cons.ADM_AUTOREG_SECTION, cons.CLEAN_UP):
                        initCleanUp = True
                    if not state.has_option(cons.ADM_AUTOREG_SECTION, cons.IS_ZERO_TOUCH_REGISTRATION_ALLOWED):
                        initIsZeroTouchRegistrationAllowed = True
                    if not state.has_option(cons.ADM_AUTOREG_SECTION, cons.DEPLOYMENT_TYPE):
                        initDeploymentType = True
                    if not state.has_option(cons.ADM_AUTOREG_SECTION, cons.UPDATE_NSIP_DEPLOYMENT_TYPE):
                        initUpdateNSIPDeploymentType = True
                    if not state.has_option(cons.ADM_AUTOREG_SECTION, cons.CLIP):
                        initClip = True
                    if state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_FLAG_OLD):
                        # util_internal_reg.remove_cloud_connect_user_if_exists(cons.OLD_CLOUD_CONNECT_USER_1)
                        # util_internal_reg.remove_cloud_connect_user_if_exists(cons.OLD_CLOUD_CONNECT_USER_2)
                        state.remove_option(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_FLAG_OLD)
                    if (state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_FLAG) and
                        util_internal_reg.get_internal_ns_registration_state() == cons.NOT_ALLOWED) or \
                            ((not os.path.isfile(cons.MASTOOLS_AGENT_CONF_FILE)) and
                             state.has_option(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_FLAG) and
                             util_internal_reg.get_internal_ns_registration_state() == cons.SUCCESS):
                        state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_FLAG, cons.NOT_APPLICABLE)
            else:
                # Recreate state file in case the ADM_AUTOREG_SECTION is missing
                open(stateFilePath, 'w').close()
                initStateFile = True
        except Exception as e:
            logger.error("Failed to read {}: {}".format(stateFilePath, str(e)))
            # Recreate state file in case not able to read
            open(stateFilePath, 'w').close()
            initStateFile = True
    else:
        initStateFile = True

    if initStateFile == True:
        logger.info("Initializing state file")
        state = ConfigParser.RawConfigParser()
        state.add_section(cons.ADM_AUTOREG_SECTION)
        initReachableFlag, initClamedFlag, initRegistrationFlag, initLodestoneEnableFlag = True, True, True, True
        initInternalNSRegistrationFlag, initInternalNSRegistrationStartTime, initInternalNSRegistrationNSIP, initTestLoadestone, initIsInternal = \
            True, True, True, True, True
        initNSUniqueID, initCleanUp, initIsZeroTouchRegistrationAllowed, initDeploymentType, initUpdateNSIPDeploymentType = True, True, True, True, True
        initClip = True

    if initReachableFlag == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_SAFEHAVEN_REACHABLE_FLAG, False)
    if initClamedFlag == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_ADCCLAIMED_FLAG, False)
    if initRegistrationFlag == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_ADCREGISTRATION_FLAG, cons.NOT_APPLICABLE)
    if initLodestoneEnableFlag == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_LODESTONE_ENABLE_FLAG, cons.NOT_APPLICABLE)
    if initInternalNSRegistrationFlag == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_FLAG, cons.NOT_APPLICABLE)
    if initInternalNSRegistrationStartTime == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_START_TIME, cons.NOT_APPLICABLE)
    if initInternalNSRegistrationNSIP == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_INTERNAL_REGISTRATION_NSIP, cons.NOT_APPLICABLE)
    if initTestLoadestone == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.TEST_LODESTONE, False)
    if initNSUniqueID == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_NS_UNIQUE_ID, None)
    if initCleanUp == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.CLEAN_UP, True)
    if initIsZeroTouchRegistrationAllowed == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.IS_ZERO_TOUCH_REGISTRATION_ALLOWED, False)
    if initDeploymentType == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.DEPLOYMENT_TYPE, cons.NOT_APPLICABLE)
    if initUpdateNSIPDeploymentType == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.UPDATE_NSIP_DEPLOYMENT_TYPE, True)
    if initClip == True:
        state.set(cons.ADM_AUTOREG_SECTION, cons.CLIP, cons.NOT_APPLICABLE)

    with open(stateFilePath, 'w') as configfile:
        state.write(configfile)

def get_safehaven_service_state():
    state = ConfigParser.RawConfigParser()
    state.read(get_autoreg_state_file())
    return state.getboolean(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_SAFEHAVEN_REACHABLE_FLAG)

def set_safehaven_service_state(isReachable):
    state = ConfigParser.RawConfigParser()
    state.read(get_autoreg_state_file())
    state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_SAFEHAVEN_REACHABLE_FLAG, isReachable)
    with open(get_autoreg_state_file(), 'w') as configfile:
        state.write(configfile)

def get_claimed_state():
    state = ConfigParser.RawConfigParser()
    state.read(get_autoreg_state_file())
    return state.getboolean(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_ADCCLAIMED_FLAG)

def set_claimed_state(claimed_status):
    state = ConfigParser.RawConfigParser()
    state.read(get_autoreg_state_file())
    state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_ADCCLAIMED_FLAG, claimed_status)
    if claimed_status == False:
        # Reset registration state to NA
        state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_ADCREGISTRATION_FLAG, cons.NOT_APPLICABLE)
    with open(get_autoreg_state_file(), 'w') as configfile:
        state.write(configfile)

def set_lodestone_enable_state(lodestoneenable_status):
    state = ConfigParser.RawConfigParser()
    state.read(get_autoreg_state_file())
    state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_LODESTONE_ENABLE_FLAG, lodestoneenable_status)
    with open(get_autoreg_state_file(), 'w') as configfile:
        state.write(configfile)

def get_lodestone_enable_sate():
    state = ConfigParser.RawConfigParser()
    state.read(get_autoreg_state_file())
    return state.getboolean(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_LODESTONE_ENABLE_FLAG)


def set_registration_state(registration_status):
    state = ConfigParser.RawConfigParser()
    state.read(get_autoreg_state_file())
    state.set(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_ADCREGISTRATION_FLAG, registration_status)
    with open(get_autoreg_state_file(), 'w') as configfile:
        state.write(configfile)

def get_mastools_registration_status():
    logger.debug("In get_mastools_registration_status")
    start_time = time.time()
    value = None
    try:
        if os.path.isfile(cons.MASTOOLS_INIT_STATUS):
            while True:
                with open(cons.MASTOOLS_INIT_STATUS) as f:
                    for line in f:
                        tokens = line.split("=")
                        if tokens[0] == cons.REGISTRATION:
                            value = tokens[1].strip()
                            break
                if value == cons.INPROGRESS:
                    if time.time()-start_time > cons.MASTOOLS_REGISTRATION_TIMEOUT:
                        logger.info("Registration is Stuck In Progress, timed-out, returning Failure")
                        os.unlink(cons.MASTOOLS_INIT_STATUS)
                        return cons.FAILURE
                    # Registration is in progress, wait for WAITING_TIME_REGISTRATION_STATUS_CHECK and try again
                    logger.info("Registration is In Progress")
                    time.sleep(cons.WAITING_TIME_REGISTRATION_STATUS_CHECK)
                elif value == cons.SUCCESS or value == cons.FAILURE:
                    # Mastools registration is complete, remove MASTOOLS_INIT_STATUS file
                    os.unlink(cons.MASTOOLS_INIT_STATUS)
                    break
        else:
            if os.path.isfile(cons.MASTOOLS_AGENT_CONF_FILE):
                value = cons.SUCCESS
            else:
                value = cons.FAILURE
    except Exception as e:
        logger.exception("Failed to read {}: {}".format(cons.MASTOOLS_INIT_STATUS, str(e)))
        value = cons.FAILURE
    return value

def check_mastools_agentconf():
    if os.path.isfile(cons.MASTOOLS_AGENT_CONF_FILE):
        return True
    else:
        return False

def get_registration_state():
    logger.debug("In get_registration_state")
    state = ConfigParser.RawConfigParser()
    state.read(get_autoreg_state_file())
    claimed_status = state.getboolean(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_ADCCLAIMED_FLAG)
    registration_status = None
    if claimed_status == True:
        registration_status = state.get(cons.ADM_AUTOREG_SECTION, cons.ADM_AUTOREG_ADCREGISTRATION_FLAG)
        if registration_status == cons.NOT_APPLICABLE:
            # This case will come only when mastools has restarted this daemon before completing registration
            registration_status = get_mastools_registration_status()
            set_registration_state(registration_status)
    else:
        registration_status = cons.NOT_APPLICABLE

    return registration_status


def get_initial_wait_time():
    if IS_PLATFORM_SDX:
        return util_svm.get_initial_wait_time()

    logger.debug("In get_initial_wait_time")
    value = -1
    try:
        for line in fileinput.input(cons.ADMPARAM_CONF_FILE, inplace = True):
            param = line.split("=")
            if (param[0] == cons.INITIAL_WAIT):
                value = int(param[1])
                print("%s=0" % cons.INITIAL_WAIT)
            else:
                print(line.strip())
    except Exception as e:
        logger.exception(str(e))

    return value

def request(method, url, headers, payload=None, timeout=cons.TIMEOUT, retry=cons.RETRY, cert=cons.ADM_CERT_BUNDLE_FILE, proxies = None):
    response = None
    try:
        if proxies:
            logger.debug("Using proxy: %s", str(proxies))
        for i in range(retry):
            try:
                if method == cons.HTTP_GET:
                    response = requests.get(url, headers=headers, timeout=timeout, verify=cert, proxies=proxies)
                elif method == cons.HTTP_POST:
                    response = requests.post(url, data=payload, headers=headers, timeout=timeout, verify=cert, proxies=proxies)
                elif method == cons.HTTP_PUT:
                    response = requests.put(url, data=payload, headers=headers, timeout=timeout, verify=cert, proxies=proxies)
                elif method == cons.HTTP_DELETE:
                    response = requests.delete(url, headers=headers, timeout=timeout, verify=cert, proxies=proxies)
                else:
                    raise RequestError("Invalid Request Method: {}".format(method))
                response.raise_for_status()
                return response
            except (requests.exceptions.Timeout, requests.exceptions.ConnectionError) as e:
                logger.debug("Attempt #{} failed to connect: {}".format(i+1, str(e)))
                continue

        raise RequestCommunicationError("Failed to connect after #{} retries".format(retry))
    except requests.exceptions.HTTPError as e:
        logger.debug("Recevied HTTP error response: {}".format(str(e)))
        raise RequestHttpError(str(e), e.response.status_code)
    except requests.exceptions.RequestException as e:
        logger.debug("Failed to send request to URL {}: {}".format(url, str(e)))
        raise RequestCommunicationError(str(e))

def get_nitroresp_from_clicmdnitro_op(cliNitroOutput):
    try:
        if cliNitroOutput == None:
            return None
        nitroJson = None
        cliNitroOutput = cliNitroOutput.split(" Done\n")[1]
        tokens = cliNitroOutput.split("\n")
        for val in tokens:
            match = re.search(r"^\s*NITRORESP :\s+(.*$)", val.strip())
            if match:
                nitroJson = json.loads(match.group(1))
                break
        return nitroJson
    except Exception as e:
        logger.error(e)
        raise Exception("Failed to get NITRORESP from cliNitroOutput str: {}".format(cliNitroOutput))

def clinitro_exec(clicmd):
    nitroJson = None
    try:
        cmd = cons.NSCLI_CMD + ' "' + cons.CLICMDNITRO_EXEC + " '" + clicmd + "'" + '"'
        cliNitroOutput = subprocess.check_output(shlex.split(cmd), stderr=subprocess.STDOUT)
        if is_python_3:
            cliNitroOutput = cliNitroOutput.decode("utf-8") 
        nitroJson = get_nitroresp_from_clicmdnitro_op(cliNitroOutput)
    except subprocess.CalledProcessError as e:
        # nscli returns exit-code=1 in error-scenario (ex: when resource isn't there)
        if e.returncode == 1:
            nitroJson = get_nitroresp_from_clicmdnitro_op(e.output)
            if nitroJson:
                return nitroJson
        raise Exception("Failed to execute clinitro cmd: {}".format(clicmd))
    except Exception as e:
        logger.error(str(e))
        raise Exception("Failed to execute clinitro cmd: {}".format(clicmd))
    return nitroJson

def cli_exec(clicmd):
    cmd = cons.NSCLI_CMD + ' "' + clicmd + '"'
    cliOutput = subprocess.check_output(shlex.split(cmd), stderr=subprocess.STDOUT)
    if is_python_3:
        cliOutput = cliOutput.decode("utf-8") 
    return cliOutput.split(" Done\n")[1]

def password_interactive_cli_exec(clicmd, password):
    cmd = cons.NSCLI_CMD + ' "' + clicmd + '"'
    proc = pexpect.spawn(cmd)
    while True:
        ret = proc.expect(['Enter password:', pexpect.EOF, pexpect.TIMEOUT])
        if ret == 0:
            proc.sendline(password)
        elif ret == 1 or ret == 2:
            cliOutput = proc.before
            break
    return cliOutput.split(" Done\n")[1]

def is_platform_blx():
    try:
        nitroJson = clinitro_exec(cons.SHOW_HARDWARE)
        if nitroJson != None and 'nshardware' in nitroJson:
            return "blx" in nitroJson['nshardware']['hwdescription'].lower()
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Hardware info")

def mastools_registration(device_profile_name, registration_endpoint, registration_secret):
    if IS_PLATFORM_SDX:
        cmd = cons.MASTOOLS_INIT_SDX % (device_profile_name, registration_endpoint, registration_secret)
    elif is_platform_blx():
        cmd = cons.MASTOOLS_INIT_BLX % (device_profile_name, registration_endpoint, registration_secret)
    else:
        cmd = cons.MASTOOLS_INIT_ADC % (device_profile_name, registration_endpoint, registration_secret)
    logger.debug("Mastool Registration: {}".format(cmd))
    os.system(cmd)
    # Wait for mastools to set the registration status
    time.sleep(cons.WAITING_TIME_REGISTRATION_STATUS_CHECK)
    return get_mastools_registration_status()

def start_dry_run(device_profile_name, dryrun_endpoint, dryrun_secret, dryrun_uuid):
    cmd = cons.DRY_RUN_COMMAND % (device_profile_name, dryrun_endpoint, dryrun_secret, dryrun_uuid)
    logger.debug("Mastool Dryrun: {}".format(cmd))
    os.system(cmd)
    return

def cli_registration(adm_deployment, adm_username, adm_password, registration_endpoint, device_profile_name):
    if IS_PLATFORM_SDX:
        logger.error("Agent based registration not supported for SDX platform")
        return cons.FAILURE

    cmd = cons.ADD_CENTRALMGMTSRV % (adm_deployment, adm_username, registration_endpoint, device_profile_name)
    logger.debug("CLI Registration: {}".format(cmd))
    response = password_interactive_cli_exec(cmd, adm_password)
    logger.info("CLI Registration Response: {}".format(response))
    if cons.CLI_ERROR in response:
        return cons.FAILURE
    return cons.SUCCESS

def get_license_server_info(nodeid):
    if IS_PLATFORM_SDX:
        return util_svm.get_license_server_info()
    logger.debug("In get_license_server_info")
    license_server, license_port = "", ""
    try:
        nitroJson = clinitro_exec(cons.SHOW_LICENSESERVER)
        if nitroJson != None and 'nslicenseserver' in nitroJson:
            nslicenseservers =  nitroJson['nslicenseserver']
            for nslicenseserver in nslicenseservers:
                if nodeid == '' or ('nodeid' in nslicenseserver and nslicenseserver['nodeid'] == nodeid):
                    license_server = nslicenseserver['servername']
                    license_port = nslicenseserver['port']
                    break
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get License Server info")
    return license_server, license_port

############################################################
## Structures defined to communicate with nslped(C)       ##
## daemon to get pooled license Serial Numbers.           ##
## Request and response structures expected by nslped(C)  ##
## daemon is defined in following C header file.          ##
## File: usr.src/netscaler/triscale/lped/nslped.h         ##
############################################################
def struct_license_request():
    latest_greater_than_12_1_58_0 = False
    latest_greater_than_13_1_39_7 = False
    try:
        version, build, type = get_version_info()
        if version == cons.VERSION_12_1 and float(build) > cons.BUILD_58_0:
            latest_greater_than_12_1_58_0 = True
        if version == cons.VERSION_13_1 and float(build) > cons.BUILD_39_7:
            latest_greater_than_13_1_39_7 = True
    except Exception as e:
        logger.error(str(e))

    if latest_greater_than_13_1_39_7 or version == cons.VERSION_14_1:
        class License_request(Structure):
            _fields_ = [("lic_cmd", c_int),
                        ("lic_pool", c_int),
                        ("lic_edition", c_int),
                        ("lic_count", c_uint32),
                        ("license_server", c_char * cons.NS_LPE_MAX_SERVER_NAME_LEN_256),
                        ("hostName", c_char * cons.NS_LPE_MAX_CLIENTHOSTNAME_LEN),
                        ("prodName", c_char * cons.NS_LPE_MAX_PROD_LEN),
                        ("prodUserName", c_char * cons.NS_LPE_MAX_PROD_LEN),
                        ("license_port", c_uint32),
                        ("errCode", c_uint32),
                        ("flag", c_uint32),
                        ("licmode", c_uint8),
                        ("lic_heartbeatinterval", c_uint32),
                        ("lic_inventoryrefreshinterval", c_uint32)]
    elif latest_greater_than_12_1_58_0 or version == cons.VERSION_13_0:
        class License_request(Structure):
            _fields_ = [("lic_cmd", c_int),
                        ("lic_pool", c_int),
                        ("lic_edition", c_int),
                        ("lic_count", c_uint32),
                        ("license_server", c_char * cons.NS_LPE_MAX_SERVER_NAME_LEN_256),
                        ("hostName", c_char * cons.NS_LPE_MAX_CLIENTHOSTNAME_LEN),
                        ("prodName", c_char * cons.NS_LPE_MAX_PROD_LEN),
                        ("prodUserName", c_char * cons.NS_LPE_MAX_PROD_LEN),
                        ("license_port", c_uint32),
                        ("errCode", c_uint32),
                        ("flag", c_uint32),
                        ("licmode", c_uint8)]
    else:
        class License_request(Structure):
            _fields_ = [("lic_cmd", c_int),
                        ("lic_pool", c_int),
                        ("lic_edition", c_int),
                        ("lic_count", c_uint32),
                        ("license_server", c_char * cons.NS_LPE_MAX_SERVER_NAME_LEN_32),
                        ("hostName", c_char * cons.NS_LPE_MAX_CLIENTHOSTNAME_LEN),
                        ("prodName", c_char * cons.NS_LPE_MAX_PROD_LEN),
                        ("prodUserName", c_char * cons.NS_LPE_MAX_PROD_LEN),
                        ("license_srvrip", c_uint32),
                        ("license_port", c_uint32),
                        ("errCode", c_uint32),
                        ("flag", c_uint32),
                        ("licmode", c_uint8)]
    return License_request

def struct_nslped_serialno_header():
    class Nslped_serialno_header(Structure):
        _fields_ = [("item_count", c_uint32)]
    return Nslped_serialno_header

def struct_nslped_serialno_res(count):
    class Lped_serialno_data(Structure):
        _fields_ = [("SerialNo", c_char * cons.NSLPE_MAX_STR)]

    class Nslped_serialno_res(Structure):
        _fields_ = [("lpe_serialno", Lped_serialno_data * count)]
    return Nslped_serialno_res

def struct_nslped_notice_res(count):
    class Lped_serialno_data(Structure):
        _fields_ = [("Notice", c_char * cons.NSLPE_MAX_STR)]

    class Nslped_serialno_res(Structure):
        _fields_ = [("lpe_notice", Lped_serialno_data * count)]
    return Nslped_serialno_res
############################################################

# Fetch pooled license serial numbers from nslped daemon.
# Nspled daemon currently returns serial numbers only for
# CNS_INST_CCS type licenses available on the pooled
# license server.
def get_pooled_license_sno_notice(nodeid):
    pooledLicenseSno = []
    pooledLicenseNotice = []

    try:
        license_server, license_port = get_license_server_info(nodeid)
        logger.debug("Pooled license server not configured license_server, license_port : {} {}".format(license_server, license_port))
        if license_server == "":
            logger.debug("Pooled license server not configured")
            return pooledLicenseSno, None
    except Exception as e:
        logger.error(str(e))
        return pooledLicenseSno, None

    try:
        s = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        if not is_python_3:
            if s < 0:
                logger.error("Failed to create socket to communicate with license server")
                return pooledLicenseSno, None
        s.connect(cons.NSLPED_SOCK)
        logger.debug("Connected to license server socket")
    except Exception as e:
        logger.error("Failed to connect with license server socket: {}".format(str(e)))
        return pooledLicenseSno, None

    try:
        license_request_t = struct_license_request()
        request = license_request_t()
        request.lic_cmd = cons.LPE_CMD_INVENTORY_FILTERED
        request.lic_pool = cons.LPE_POOL_INSTANCE
        request.license_server = bytes(license_server, 'utf-8')
        request.licmode = cons.LIC_MODE
        request.license_port = int(license_port)

        nsent = s.send(request)
        if nsent < 0:
            logger.error("Failed to send request to license server")
            s.close()
            return pooledLicenseSno, None

        buff = s.recv(sizeof(license_request_t))
        response = license_request_t.from_buffer_copy(buff)

        if response.errCode != 0:
            logger.error("Recieved errCode {} from license server".format(response.errCode))
            s.close()
            return pooledLicenseSno, None

        is_notice_available = False
        # check for flag:- If license client is providing Notice info, then flag will be 1
        if response.flag & 0x01:
            is_notice_available = True
        logger.debug("is_notice_available {}".format(is_notice_available))

        Nslped_serialno_header = struct_nslped_serialno_header()
        buff = s.recv(sizeof(Nslped_serialno_header))
        inv_resp = Nslped_serialno_header.from_buffer_copy(buff)
        count = inv_resp.item_count
        logger.debug("Count returned by license server: {}".format(count))

        Nslped_serialno_res = struct_nslped_serialno_res(count)
        buff = s.recv(sizeof(Nslped_serialno_res))
        inv_resp_sno = Nslped_serialno_res.from_buffer_copy(buff)


        if is_notice_available:
            Nslped_notice_res = struct_nslped_notice_res(count)
            buff = s.recv(sizeof(Nslped_notice_res))
            inv_resp_notice = Nslped_notice_res.from_buffer_copy(buff)

        for i in range(count):
            sno = inv_resp_sno.lpe_serialno[i].SerialNo
            logger.debug("#{}: {}".format(i, sno))
            if isinstance(sno, bytes):
                sno = sno.decode()
            if sno not in pooledLicenseSno:
                pooledLicenseSno.append(sno)

        if is_notice_available:
            for i in range(count):
                notice = inv_resp_notice.lpe_notice[i].Notice
                logger.debug("#{}: {}".format(i, notice))
                if isinstance(notice, bytes):
                    notice = notice.decode()
                if notice not in pooledLicenseNotice:
                    pooledLicenseNotice.append(notice)

        s.close()
    except Exception as e:
        import traceback
        logger.error(traceback.format_exc())
        logger.error(str(e))
        s.close()

    if len(pooledLicenseNotice) == 0:
        # Not licensed through ADM
        is_internal_ns = None
    else:
        is_internal_ns = True
        for notice in pooledLicenseNotice:
            if notice != cons.INTERNAL_NOTICE_VALUE_1 and notice != cons.INTERNAL_NOTICE_VALUE_2:
                # Making is_internal_ns value False if it is not a Citrix Test license
                is_internal_ns = False
    logger.info("get_pooled_license_sno_notice:is_internal: {}".format(is_internal_ns))
    if is_notice_available:
        return pooledLicenseSno, is_internal_ns
    else:
        return pooledLicenseSno, None

    # Get local license serial numbers. Each license file can
# have multiple serial numbers. Extracting all unique
# serial numbers by parsing every license file available
# on the local system.
def get_local_license_sno_notice():
    logger.debug("In get_local_license_sno")
    localLicenseSno = []
    localLicenseNoticeList = []
    for licensefile in glob.glob(get_local_license_file()):
        with open(licensefile, 'r') as f:
            for line in f:
                if cons.NOTICE_IDENTIFIER in line:
                    notice_match = re.search(r"^.*%s\"(.*)\"\s.*$" % cons.NOTICE_IDENTIFIER, line)
                    if notice_match:
                        notice = notice_match.group(1)
                        if notice not in localLicenseNoticeList:
                            localLicenseNoticeList.append(notice)
                if cons.SNO_IDENTIFIER in line:
                    match = re.search(r"^.*%s(\S*)\s.*$" % cons.SNO_IDENTIFIER, line)
                    if match:
                        sno = match.group(1)
                        # Only add unique sno
                        if sno not in localLicenseSno:
                            localLicenseSno.append(sno)
    if len(localLicenseNoticeList) == 0:
        # Not licensed through Local License
        is_internal_ns = None
    else:
        is_internal_ns = True
        for notice in localLicenseNoticeList:
            if notice != cons.INTERNAL_NOTICE_VALUE_1 and notice != cons.INTERNAL_NOTICE_VALUE_2:
                # Making is_internal_ns value False if it is not a Citrix test license
                is_internal_ns = False
    logger.info("get_local_license_sno_notice:is_internal: {}".format(is_internal_ns))
    return localLicenseSno, is_internal_ns


def get_hardware_info():
    if IS_PLATFORM_SDX:
        return util_svm.get_hardware_info()
    logger.debug("In get_hardware_info")
    serialId, encodedSerialId, hostId, netscalerUuid = "", "", "", ""
    try:
        nitroJson = clinitro_exec(cons.SHOW_HARDWARE)
        if nitroJson != None and 'nshardware' in nitroJson:
            nshardware =  nitroJson['nshardware']
            serialId = nshardware['serialno']
            encodedSerialId = nshardware['encodedserialno']
            hostId = nshardware['host']
            netscalerUuid = nshardware['netscaleruuid']
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Hardware info")

    return serialId, encodedSerialId, hostId, netscalerUuid

def get_mgmt_ip(nodeid):
    if IS_PLATFORM_SDX:
        return util_svm.get_mgmt_ip()
    logger.debug("In get_mgmt_ip")
    nsip = ""

    try:
        nitroJson = clinitro_exec(cons.SHOW_NSIP)
        if nitroJson != None and 'nsip' in nitroJson:
            nsips =  nitroJson['nsip']
            for nsip in nsips:
                if nodeid == '' or ('ownernode' in nsip and nsip['ownernode'] == nodeid):
                    nsip = nsip['ipaddress']
                    break
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Mgmt IP info")

    return nsip

def get_hostname(nodeid):
    if IS_PLATFORM_SDX:
        return util_svm.get_hostname()
    logger.debug("In get_hostname")
    hostname = ""

    try:
        nitroJson = clinitro_exec(cons.SHOW_HOSTNAME)
        if nitroJson != None and 'nshostname' in nitroJson:
            nshostnames =  nitroJson['nshostname']
            for nshostname in nshostnames:
                if nodeid == '' or ('ownernode' in nshostname and nshostname['ownernode'] == nodeid):
                    if 'hostname' in nshostname:
                        hostname = nshostname['hostname']
                    break
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Hostname info")

    return hostname

def get_version_info():
    if IS_PLATFORM_SDX:
        return util_svm.get_version_info()
    logger.debug("In get_version_info")
    version, build, type = "", "", ""
    try:
        # NetScaler NS13.0: Build 55.3001.nc, Date: Mar 27 2020, 14:42:45   (64-bit)
        versionString = cli_exec(cons.SHOW_VERSION)
        version = versionString.split("NS")[1].split(":")[0]
        buildTokens = versionString.split("Build ")[1].split(",")[0].split(".")
        type = buildTokens.pop()
        build = '.'.join(buildTokens)
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get NS Version info")

    return version, build, type

def get_license_code(license):
    if IS_PLATFORM_SDX:
        return util_svm.get_license_code()
    licenseCode = ""
    try:
        if license.strip() in cons.licenseCode:
            licenseCode = cons.licenseCode[license.strip()]
        else:
            logger.error("Unknow license name: {}".format(license.strip()))
            licenseCode = "unknown"
    except Exception as e:
        logger.exception(str(e))
        licenseCode = "unknown"

    return licenseCode

def get_license_info():
    if IS_PLATFORM_SDX:
        return util_svm.get_license_info()
    logger.debug("In get_license_info")
    featuresLicensed, licenseType = "", ""
    try:
        shlicense = cli_exec(cons.SHOW_LICENSE)
        tokens = shlicense.split("\n")
        for val in tokens:
            match = re.search(r"^\s*(.*):\s+YES.*$", val.strip())
            if match:
                featuresLicensed += get_license_code(match.group(1)) + " "
            else:
                match = re.search(r"\s*(.*):\s+(.*)\s+License$", val.strip())
                if match:
                    licenseType = match.group(2)
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get License info")

    return featuresLicensed.strip(), licenseType

def get_hypervisor_info():
    if IS_PLATFORM_SDX:
        return util_svm.get_hypervisor_info()
    logger.debug("In get_hypervisor_info")
    hypervisors = cons.HYPERVISER_LIST
    response = ""

    try:
        if IS_PLATFORM_LINUX:
            virtual  = int(os.environ.get("HYPERVISED_NS", 0))
        else:
            virtual = int(sysctl_exec(cons.NS_HYPERVISER))
        response = hypervisors[virtual]
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Hypervisor info")

    return response

def get_cloud_info():
    if IS_PLATFORM_SDX:
        return util_svm.get_cloud_info()
    logger.debug("In get_cloud_info")
    clouds = cons.CLOUD_LIST
    response = ""

    try:
        if IS_PLATFORM_LINUX:
            cloud = int(os.environ.get("VPX_ON_CLOUD", 0))
        else:
            cloud = int(sysctl_exec(cons.NS_CLOUD))
        response = clouds[cloud]
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Cloud info")

    return response

def get_platform_info():
    if IS_PLATFORM_SDX:
        return util_svm.get_platform_info()
    logger.debug("In get_platform_info")
    sysid, platformDescription, platformType = "", "", ""

    try:
        if IS_PLATFORM_LINUX:
            sysid = os.environ.get("SYSID", "")
            platformDescription = os.environ.get("DESCR", "")
            virtual = int(os.environ.get("HYPERVISED_NS", 0))
        else:
            sysid = sysctl_exec(cons.NS_SYSID)
            platformDescription = sysctl_exec(cons.NS_DESCR)
        

            virtual = int(sysctl_exec(cons.NS_HYPERVISER))
        if virtual > 0:
            platformType = "VPX"
        else:
            platformType = "MPX"
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Platform info")

    return sysid, platformDescription, platformType

def get_enabled_features_and_modes():
    if IS_PLATFORM_SDX:
        return util_svm.get_enabled_features_and_modes()
    logger.debug("In get_enabled_features_and_modes")
    featuresEnabled, modesEnabled = "", ""

    try:
        shFeature = cli_exec(cons.SHOW_NSFEATURE)
        tokens = shFeature.split("\n")
        for val in tokens:
            match = re.search(r"^\s*(\S+)\s+(\S+(\s\S+)+)\s+(\S+)\s+ON$", val.strip())
            if match:
                featuresEnabled += match.group(4).lower() + " "

        shMode = cli_exec(cons.SHOW_NSMODE)
        tokens = shMode.split("\n")
        for val in tokens:
            match = re.search(r"^\s*(\S+)\s+(\S+(\s\S+)+)\s+(\S+)\s+ON$", val.strip())
            if match:
                modesEnabled += match.group(4).lower() + " "
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get enabled Features and Modes info")

    return featuresEnabled.strip(), modesEnabled.strip()

def get_deployment_info():
    if IS_PLATFORM_SDX:
        return util_svm.get_deployment_info()
    logger.debug("In get_deployment_info")
    deployment, ip, nodeid = "", "", ""

    try:
        # Check for cluster deployment
        nitroJson = clinitro_exec(cons.SHOW_CLUSTER_NODE)
        if nitroJson != None and 'clusternode' in nitroJson:
            clusternodes = nitroJson['clusternode']
            for node in clusternodes:
                if node['islocalnode'] == True:
                    nodeid = node['nodeid']
                    if node['isconfigurationcoordinator'] == True:
                        deployment = cons.CLUSTER_CCO
                    else:
                        deployment = cons.CLUSTER_NONCCO
                    break
            nitroJson = clinitro_exec(cons.SHOW_NSIP_CLIP)
            if nitroJson != None and 'nsip' in nitroJson:
                # CLIP ip
                ip = nitroJson['nsip'][0]['ipaddress']
        else:
            # Check for HA/Standalone deployment
            nitroJson = clinitro_exec(cons.SHOW_HANODE)
            if nitroJson != None and 'hanode' in nitroJson:
                hanodes = nitroJson['hanode']
                nodeCount = len(hanodes)
                if nodeCount == 2:
                    for node in hanodes:
                        # Id uniquely identifies the node.
                        # For self node, it will always be 0
                        if node['id'] == '0':
                            if node['state'] == cons.PRIMARY:
                                deployment = cons.HA_PRIMARY
                            else:
                                deployment = cons.HA_SECONDARY
                        else:
                            # HA peer node ip
                            ip = node['ipaddress']
                else:
                    deployment = cons.STANDALONE
            else:
                logger.error("Failed to HA node")
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Deployment info")

    return deployment, ip, nodeid

def get_nsconf():
    if IS_PLATFORM_SDX:
        return util_svm.get_nsconf()
    logger.debug("In get_nsconf")
    latest_ns_conf = cons.NSCONF
    scrub_ns_conf = cons.TMP_NSCONF
    replStrng = "XXXX"

    encodedNsconf, encoding = "", ""

    try:
        copy2(latest_ns_conf, scrub_ns_conf)
        os.chmod(scrub_ns_conf, cons.FILE_PERMISSION_CODE)
        in_file = open(scrub_ns_conf, "r")
        config = ""
        for line in in_file:
            if line.find("-encrypted"):
                line = re.sub(r"(\w+\s+)(?=-encrypted)", replStrng+" ", line)
                config += line
            elif line.find("-passcrypt"):
                line = re.sub(r"(?<=-passcrypt).*?(?=(-|$))", " "+replStrng+" ", line)
                config += line
            else:
                config += line
        in_file.close()

        if is_python_3:
            config = config.encode('utf-8')
        encodedNsconf = base64.b64encode(zlib.compress(config))
        if is_python_3:
            encodedNsconf = encodedNsconf.decode('utf-8')
        encoding = "gzip,base64"

    except Exception as e:
        logger.exception(str(e))
        os.remove(scrub_ns_conf)
        raise Exception("Failed to get gzip,base64 ns.conf info")

    os.remove(scrub_ns_conf)
    return encodedNsconf, encoding

def get_system_date():
    logger.debug("In get_system_date")
    datestring = ""

    try:
        now = datetime.now()
        datestring = now.strftime("%Y-%m-%dT%H:%M:%SZ")
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get System date")

    return datestring

def get_ns_stats():
    if IS_PLATFORM_SDX:
        return util_svm.get_ns_stats()
    logger.debug("In get_ns_stats")
    cpuPercentage, mgmtCpuPercentage, memoryPercentage, rxTput, systemUpTime = "", "", "", "", ""

    try:
        statSystem = cli_exec(cons.STAT_NS)
        tokens = statSystem.split("\n")
        for val in tokens:
            match = re.search(r"^((\S+\s)+)\s+(.*)$", val.strip())
            if match:
                if match.group(1).strip() == cons.UP_SINCE:
                    systemUpTime = match.group(3).strip()
                elif match.group(1).strip() == cons.PACKET_CPU_PER:
                    cpuPercentage = match.group(3).strip()
                elif match.group(1).strip() == cons.MGMT_CPU_PER:
                    mgmtCpuPercentage = match.group(3).strip()
                elif match.group(1).strip() == cons.MEMORY_PER:
                    memoryPercentage = match.group(3).strip()
                elif match.group(1).strip() == cons.MEGABITS_TX:
                    rxTput = match.group(3).strip().split(" ")[0]

    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get NS STATS")

    return cpuPercentage, mgmtCpuPercentage, memoryPercentage, rxTput, systemUpTime

def get_ssl_stats():
    if IS_PLATFORM_SDX:
        return util_svm.get_ssl_stats()
    logger.debug("In get_ssl_stats")
    sslNewSessions, sslEncTput, sslDecTput = "", "", ""

    try:
        statSsl = cli_exec(cons.STAT_SSL)
        tokens = statSsl.split("\n")
        for val in tokens:
            match = re.search(r"^((\S+\s)+)\s+(\S+)\s+(\S+)$", val.strip())
            if match:
                if match.group(1).strip() == cons.NEW_SSL_SESSIONS:
                    sslNewSessions = match.group(3).strip()
                elif match.group(1).strip() == cons.BYTES_ENC:
                    sslEncTput = match.group(3).strip()
                elif match.group(1).strip() == cons.BYTES_DEC:
                    sslDecTput = match.group(3).strip()
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get SSL STATS")

    return sslNewSessions, sslEncTput, sslDecTput

def get_disk_failure_info():
    logger.debug("In get_disk_failure_info")
    hdd_failure, cf_failure = False, False

    try:
        cmd = subprocess.Popen(shlex.split(cons.CHECK_DISK_FAILURE_CMD))
        _, _ = cmd.communicate()
        returncode = cmd.returncode
        if returncode == cons.EXIT_CODE_HDD_ERRORS:
            hdd_failure = True
        elif returncode == cons.EXIT_CODE_CF_ERRORS:
            cf_failure = True
        elif returncode == cons.EXIT_CODE_HDD_CF_ERRORS:
            hdd_failure, cf_failure =  True, True
        elif returncode == cons.EXIT_CODE_INTERNAL_ERRORS:
            logger.error("Internal error: Failed to check disk errors")
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to check Disk failure info")

    return hdd_failure, cf_failure

def get_pe_failure_info():
    logger.debug("In get_pe_failure_info")
    pe_failure = False

    try:
        cmd = subprocess.Popen(shlex.split(cons.CHECK_PE_FAILURE_CMD))
        _, _ = cmd.communicate()
        returncode = cmd.returncode
        if returncode == cons.EXIT_CODE_PE_CRASH_ERRORS:
            pe_failure = True
        elif returncode == cons.EXIT_CODE_INTERNAL_ERRORS:
            logger.error("Internal error: Failed to check for PE failures")
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to check PE failure info")

    return pe_failure

def get_psu_status_info():
    logger.debug("In get_psu_status_info")
    ps1status, ps2status, ps3status, ps4status = "", "", "", ""

    try:
        statSystem = cli_exec(cons.STAT_SYSTEM)
        tokens = statSystem.split("\n")
        for val in tokens:
            match = re.search(r"^(\S+\s\S+\s\d\s\S+)\s+(.*)$", val.strip())
            if match:
                if match.group(1).strip() == cons.SYSHEALTH_PS1FAIL:
                    ps1status = match.group(2).strip()
                elif match.group(1).strip() == cons.SYSHEALTH_PS2FAIL:
                    ps2status = match.group(2).strip()
                elif match.group(1).strip() == cons.SYSHEALTH_PS3FAIL:
                    ps3status = match.group(2).strip()
                elif match.group(1).strip() == cons.SYSHEALTH_PS4FAIL:
                    ps4status = match.group(2).strip()
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get PSU status info")

    return ps1status, ps2status, ps3status, ps4status

def get_ssl_card_info():
    logger.debug("In get_ssl_card_info")
    totalSslCards, numSslCardsUp = "0", "0"

    try:
        statSsl = cli_exec(cons.STAT_SSL)
        tokens = statSsl.split("\n")
        for val in tokens:
            match = re.search(r"^((\S+\s)+)\s+(\S+)$", val.strip())
            if match:
                if match.group(1).strip() == cons.NUMBER_OF_SSL_CARDS_PRESENT:
                    totalSslCards = match.group(3).strip()
                elif match.group(1).strip() == cons.NUMBER_OF_SSL_CARDS_UP:
                    numSslCardsUp = match.group(3).strip()
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get SSL Card info")

    return totalSslCards, numSslCardsUp

def get_system_mem_usage_info():
    logger.debug("In get_system_mem_usage_info")
    inUseMemPer = "0.0"

    try:
        statMemory = cli_exec(cons.STAT_MEMORY)
        tokens = statMemory.split("\n")
        for val in tokens:
            match = re.search(r"^((\S+\s)+)\s+(\S+)$", val.strip())
            if match:
                if match.group(1).strip() == cons.MEMORY_PER:
                    inUseMemPer = match.group(3).strip()
                    break
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get System Memory usage info")

    return inUseMemPer

# Returns dict for `show service <svc_name>` CLI
def get_service_info(svc_name):
    logger.debug("In get_service_info")
    serviceInfo = None
    try:
        nitroJson = clinitro_exec(cons.SHOW_SERVICE_BY_NAME.format(svc_name))
        if nitroJson != None and 'service' in nitroJson:
            serviceInfo = nitroJson['service']
            if type(serviceInfo) == list and len(serviceInfo):
                serviceInfo = serviceInfo[0]
        elif nitroJson:
            raise Exception("Failed to get Service Setting. Nitro Response: {}".format(nitroJson))
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Service Setting")
    return serviceInfo

# Returns true if service json passed has state equal to UP
def is_service_up(svc_json):
    logger.debug("In is_service_up")
    if svc_json != None and "svrstate" in svc_json and svc_json["svrstate"] == cons.UP:
        return True
    return False

# Returns folllowing proxy related params from `show callhome` CLI if callhome is enabled:
# proxymode, ipaddress, port, proxyauthservice
def get_callhome_proxy_info():
    logger.debug("In get_callhome_proxy_info")
    proxyMode = False
    ipaddress = None
    port = None
    proxyAuthService = None
    try:
        nitroJson = clinitro_exec(cons.SHOW_CALLHOME)
        if nitroJson != None and 'callhome' in nitroJson:
            if isinstance(nitroJson['callhome'], list):
                callhomeInfo = nitroJson['callhome'][0]
            else:
                callhomeInfo = nitroJson['callhome']
            proxyModeStr = callhomeInfo['proxymode']
            if proxyModeStr == 'YES':
                proxyMode = True
            if proxyMode:
                if 'ipaddress' in callhomeInfo:
                    ipaddress = callhomeInfo['ipaddress']
                if 'proxyauthservice' in callhomeInfo:
                    proxyAuthService = callhomeInfo['proxyauthservice']
                if 'port' in callhomeInfo:
                    port = callhomeInfo['port']
                if (ipaddress == None and proxyAuthService == None) or (port == None):
                    raise Exception('Callhome proxy config is incomplete')
        elif nitroJson:
            raise Exception("Failed to get Callhome Proxy Setting. Nitro Response: {}".format(nitroJson))
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Callhome Proxy Setting")
    return proxyMode, ipaddress, port, proxyAuthService

def get_waiting_time(failedProbeCount):
    waiting_time_per_probe = cons.WAITING_TIME_PER_PROBE
    if util_internal_reg.get_is_internal() is True:
        waiting_time_per_probe = random.randint(cons.INTERNAL_NS_PROBE_WAIT_TIME_MIN,
                                                cons.INTERNAL_NS_PROBE_WAIT_TIME_MAX)
    waitingTime = cons.WAITING_TIME_PER_LIGHTPROBE
    if failedProbeCount <= cons.IGNORE_FIRST_FAILED_PROBE_COUNT:
        failedProbeCount = 0
    else:
        failedProbeCount = failedProbeCount - cons.IGNORE_FIRST_FAILED_PROBE_COUNT

    if failedProbeCount != 0:
        waitingTime = ((2**failedProbeCount) * waiting_time_per_probe)
        if waitingTime > cons.WAITING_TIME_MAX_PER_PROBE:
            waitingTime = cons.WAITING_TIME_MAX_PER_PROBE
    return waitingTime

def get_last_used_time(dir_path, filename_starting_keyword, search_pattern):
    """
    Get the last used time for a file that matches the given filename starting keyword
    in the specified directory, using the provided search pattern.

    Args:
        dir_path (str): The directory path to search for files.
        filename_starting_keyword (str): The starting keyword of the filename to match.
        search_pattern (str): The pattern to search for within the file.

    Returns:
        str: The last used time of the file in the format "%Y-%m-%dT%H:%M:%SZ", or "unknown" if no matching file is found.
    """
    logger.debug(
        "In get_last_used_time to get the last used time for :"
        + filename_starting_keyword
        + " in dir :"
        + dir_path
        + " with search pattern :"
        + search_pattern
    )
    files = [os.path.join(dir_path, f) for f in os.listdir(dir_path) if f.startswith(filename_starting_keyword)]
    # sort the files based on modified time such that the latest file comes first
    files.sort(key=lambda f: os.path.getmtime(f), reverse=True)
    last_used_time = "unknown"
    for f in files:
        logger.debug("Checking in file :"+f)
        try:
            if f.endswith(".gz"):
                cmd = "zcat {} | grep {} | tail -n 1".format(f, search_pattern)
            else:
                cmd = "cat {} | grep {} | tail -n 1".format(f, search_pattern)
            result = subprocess.check_output(cmd, shell=True)
            if isinstance(result, bytes):
                result = result.decode()
            if result != "":
                # convert string datetime to GMT time
                date_string = re.findall(r"\d{2}/[A-Za-z]{3,4}/\d{4}:\d{2}:\d{2}:\d{2}", result)[0]
                date_time_obj = datetime.strptime(date_string,'%d/%b/%Y:%H:%M:%S')
                time_set = time.mktime(date_time_obj.timetuple())
                last_used_time = time.strftime("%Y-%m-%dT%H:%M:%SZ", time.gmtime(time_set))
        except Exception as e:
            logger.error("Exception: {}".format(str(e)))
            return last_used_time # No need to raise exception as it is not a critical error
        if last_used_time != "unknown":
            break
    return last_used_time

def get_automation_details(agent):
    logger.debug("In get_automation_details for :"+agent)
    automation_tools_record = {}
    try:
        cmd = r'''egrep -i '''+ cons.AUTOMATION_USER_AGENTS[agent] +'''  '''+ cons.AUTOMATION_LOG_FILE +''' | tail -1'''
        result = subprocess.check_output(cmd,shell=True)
        if isinstance(result, bytes):
            result = result.decode()
        if result != "":
            # convert string datetime to GMT time
            date_string = re.findall(r"\d{2}/[A-Za-z]{3,4}/\d{4}:\d{2}:\d{2}:\d{2}", result)[0]
            date_time_obj = datetime.strptime(date_string,'%d/%b/%Y:%H:%M:%S')
            time_set = time.mktime(date_time_obj.timetuple())
            formated_date = time.strftime("%Y-%m-%dT%H:%M:%SZ", time.gmtime(time_set))

            # store the atomation record details as dictionary and return
            automation_tools_record["automation_tool_name"] = agent
            automation_tools_record["last_used"] = formated_date
    except Exception as e:
        logger.exception(str(e))
        raise Exception("Failed to get Automation Record Details")

    return automation_tools_record

def get_developer_details():
    logger.debug("In Get Developer Details")
    developer_status_flag = False
    developer_conf_data = {}

    try:
        if (os.path.exists(cons.DEVELOPER_CONF_FILE)):
            with open(cons.DEVELOPER_CONF_FILE) as f:
                developer_conf_data = json.load(f)
                developer_status_flag = True
    except Exception as e:
        raise Exception("Json structure exception :{}".format(str(e)))

    return developer_status_flag, developer_conf_data

def get_diag_details(print_diag_output=False):
    logger.debug("In get_diag_details")
    diag_details_json = {}
    try:
        cmd = cons.ADC_DIAG_COMMAND
        diag_return_str = subprocess.check_output(shlex.split(cmd))
        if is_python_3:
            diag_return_str = diag_return_str.decode()
        diag_json_str = diag_return_str.split("result=")[1].strip()
        if print_diag_output:
            diag_output_all = diag_return_str.split("result=")[0]
            print(diag_return_str.split("Collecting Console Advisory Connect related configuration")[0])
        logger.debug("diag_json_str = "+ diag_json_str)
        diag_details_json = json.loads(diag_json_str)
        
    except Exception as e:
        logger.error("Exception in get_diag_details :{}".format(repr(e)))
        
    logger.debug("diag_details_json = "+ json.dumps(diag_details_json))
    return diag_details_json
    
def remove_dry_run_status():
    try:
        if os.path.isfile(cons.DRY_RUN_STATUS_FILE): 
            logger.debug("In remove_dry_run_status")
            cmd = 'rm ' + cons.DRY_RUN_STATUS_FILE
            subprocess.check_output(shlex.split(cmd), stderr=subprocess.STDOUT)
    except Exception as e:
        logger.error(repr(e))
    return

def get_adc_lodestone_test_data():
    if not os.path.exists(cons.ADC_LODESTONE_TEST_FILE):
        logger.debug(cons.ADC_LODESTONE_TEST_FILE + " doesn't exist")
        return {}
    try:
        lodestone_test = open(cons.ADC_LODESTONE_TEST_FILE)
        data = json.load(lodestone_test)
    except Exception as e:
        logger.error(cons.ADC_LODESTONE_TEST_FILE + " is not json file, exception:" + repr(e))
        data = {}
    return data

def extract_las_license_parameters():
    try:
        
        data = {}
        if IS_PLATFORM_SDX:
            las_file_path = cons.SVM_LAS_INFO_FILE
        else:
            las_file_path = cons.VPX_LAS_INFO_FILE

        if not os.path.isfile(las_file_path):
            return {}
        
        with open(las_file_path, 'r') as f:
            data = json.load(f)

        customer_info = data.get("customer", {})
        ccid = customer_info.get("ccid")
        orgid = customer_info.get("orgid")

        activation_info_list = data.get("activation_info", [])
        processed_activation_info = []
        
        for activation in activation_info_list:
            activation_record = {
                "activationid": activation.get("activationid"),
                "pem": activation.get("pem"),
                "lsid": activation.get("lsid")
            }
            processed_activation_info.append(activation_record)

        license_params = {
            "ccid": ccid,
            "orgid": orgid,
            "is_las_licensed": True,
            "las_info": {
                "activation_info": processed_activation_info
            }
        }
        return license_params
    except Exception as e:
        logger.error("Error in extract_las_license_parameters: {}".format(repr(e)))
        return {}
