#!/usr/bin/env python
"""
Copyright (c) 2020-2025 Citrix Systems, Inc.

MAS Tools File Transfer Script using RSYNC

This script provides robust file transfer capabilities using RSYNC instead of SCP,
with comprehensive error handling, retry logic, and performance optimization.

CHANGES FROM SCP TO RSYNC:
- Replaced SCP with RSYNC for better reliability and error reporting
- Added comprehensive error detection and recovery
- Implemented retry logic with exponential backoff
- Enhanced SSH connection options for robust authentication
- Improved logging and error reporting
- Added detailed exit code analysis for RSYNC operations
- No file transfer verification (relies on RSYNC's built-in integrity with checksum verification)
- No RSYNC availability check (assumes RSYNC is available)

Dependencies:
    - pexpect: For handling interactive command execution
    - rsync: Must be available on the system for file transfers
    - ssh: Required for authentication

Features:
    - RSYNC-based file transfer with SSH authentication
    - Retry logic with exponential backoff (max 2 attempts)
    - Built-in checksum verification for file integrity
    - Comprehensive error detection and reporting (20+ error patterns)
    - Detailed RSYNC exit code analysis
    - Robust cleanup and resource management
    - Enhanced SSH options for reliable connections

Usage:
    python mastools_scp.py <destip> <localfile> <remotepath> <user> <pwd>
    
Example:
    python mastools_scp.py 10.221.42.87 /var/nsinstall/build-12.1-55.13_nc_64.tgz /var/joan/ nsroot nsroot
"""

import pexpect
import argparse
import subprocess
import shlex
import os
import sys
import re
import copy
import json
import time
import base64
import xmltodict
import httplib2
import encodings
import argparse

# Python 2/3 compatibility for exception types
try:
    FileNotFoundError
except NameError:
    # Python 2.7 doesn't have FileNotFoundError
    FileNotFoundError = IOError

try:
    PermissionError  
except NameError:
    # Python 2.7 doesn't have PermissionError
    PermissionError = IOError

try:
    TimeoutError
except NameError:
    # Python 2.7 doesn't have TimeoutError
    class TimeoutError(Exception):
        pass

#=================For setting up logging ==============================================================================================================
import logging
import logging.handlers

LOG_FILENAME = '/var/mastools/logs/mastools_scp.log'
LOG_MAX_BYTE = 50*1024*1024
LOG_BACKUP_COUNT = 20

# Set up a specific logger with our desired output level
logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

# Add the log message handler to the logger
logger_handler = logging.handlers.RotatingFileHandler(LOG_FILENAME, maxBytes=LOG_MAX_BYTE, backupCount=LOG_BACKUP_COUNT)
logger_fortmater = logging.Formatter(fmt='%(asctime)s:%(funcName)s:%(lineno)d: [%(levelname)s] %(message)s', datefmt="%Y-%m-%d %H:%M:%S")
logger_handler.setFormatter(logger_fortmater)
logger.addHandler(logger_handler)

def run_shell_command(cmd):
    args = shlex.split(cmd)
    subprocess.call(args)

def send_pwd(child, pwd):
    need_password = True
    try:
        i = child.expect(['assword:', r"yes/no", pexpect.EOF], timeout=60)
        if i == 0:
            # Password prompt received
            #logger.debug("Password prompt detected, sending password")
            child.sendline(pwd)
        elif i == 1:
            # Host key verification prompt
            #logger.debug("Host key verification prompt detected, accepting and sending password")
            child.sendline("yes")
            try:
                child.expect("assword:", timeout=30)
                child.sendline(pwd)
            except pexpect.TIMEOUT:
                logger.error("Timeout waiting for password prompt after host key acceptance")
                raise TimeoutError("Timeout waiting for password prompt after host key acceptance")
        else:
            # EOF - no password needed (key-based auth or other)
            #logger.debug("No password authentication required")
            need_password = False
    except pexpect.TIMEOUT:
        logger.error("Timeout waiting for authentication prompt")
        raise TimeoutError("Timeout waiting for authentication prompt")
    except pexpect.ExceptionPexpect as e:
        logger.error("Pexpect error during authentication: {0}".format(str(e)))
        raise RuntimeError("Authentication error: {0}".format(str(e)))
    except Exception as e:
        logger.error("Unexpected error during authentication: {0}".format(str(e)))
        raise RuntimeError("Authentication failed: {0}".format(str(e)))
    
    return need_password

def file_transfer(ip, local_file, remote_dir, user, pwd):
    """
    Transfer a file via RSYNC with comprehensive error handling.
    
    Args:
        ip (str): Target IP address
        local_file (str): Path to local file to transfer
        remote_dir (str): Remote directory destination
        user (str): SSH username
        pwd (str): SSH password
        
    Raises:
        ValueError: Invalid parameters
        IOError: File access issues (file not found, permissions)
        TimeoutError: Transfer timeout
        RuntimeError: RSYNC process failures
    """
    child = None
    try:
        # Validate input parameters
        if not all([ip, local_file, remote_dir, user, pwd]):
            raise ValueError("All parameters (ip, local_file, remote_dir, user, pwd) must be provided")
        
        # Check if local file exists
        if not os.path.exists(local_file):
            raise IOError(2, "Local file does not exist: {0}".format(local_file))  # 2 = ENOENT
        
        # Check if local file is readable
        if not os.access(local_file, os.R_OK):
            raise IOError(13, "Local file is not readable: {0}".format(local_file))  # 13 = EACCES
        
        file_transfer_cmd_arg = '%s@%s:%s' % (user, ip, remote_dir)
        logger.info("run rsync command: rsync " + local_file + " " + file_transfer_cmd_arg)
        
        # Get SSH options for secure and reliable connection
        ssh_options = get_ssh_options()
        
        # Rsync arguments for reliable transfer
        # SSH ConnectTimeout is set in get_ssh_options (default 30s)
        args = [
            '-av',                          # Archive mode, verbose
            '--progress',                   # Show progress during transfer
            '--timeout=60',                 # Timeout for I/O operations and connection
            '--inplace',                    # Update destination files in-place
            '--compress',                   # Compress file data during transfer
            '--human-readable',             # Output numbers in human-readable format
            '--stats',                      # Give some file-transfer stats
            '--checksum',                   # Use checksums for file verification
            '--rsh=ssh {0}'.format(ssh_options),     # Use ssh with custom options for transport
            local_file, 
            file_transfer_cmd_arg
        ]
        
        child = pexpect.spawn('rsync', args=args, timeout=1200)
        
        # Handle authentication
        need_password = send_pwd(child, pwd)
        
        if need_password:
            # Wait for transfer completion and read any remaining output
            try:
                child.expect(pexpect.EOF, timeout=1200)  # Wait for completion
                data = child.before
                
                if data:
                    #logger.info("RSYNC transfer output: " + str(data))
                    
                    # Check for common error patterns in RSYNC output
                    error_patterns = [
                        "No space left on device",
                        "Permission denied",
                        "cannot create",
                        "failed",
                        "error",
                        "Disk quota exceeded",
                        "Read-only file system",
                        "rsync error",
                        "connection unexpectedly closed",
                        "protocol version mismatch",
                        "timeout",
                        "broken pipe",
                        "host key verification failed",
                        "connection refused",
                        "network is unreachable",
                        "no route to host",
                        "operation timed out",
                        "ssh_exchange_identification",
                        "could not resolve hostname",
                        "rsync: connection closed"
                    ]
                    data_str = str(data).lower()
                    for pattern in error_patterns:
                        if pattern.lower() in data_str:
                            logger.error("RSYNC transfer failed: {0} detected in output".format(pattern))
                            raise RuntimeError("transfer failed: {0}".format(pattern))
                    
                    # Check for positive completion indicators
                    completion_patterns = [
                        "total size is",
                        "sent",
                        "received",
                        "bytes/sec"
                    ]
                    has_completion_indicator = any(pattern.lower() in data_str for pattern in completion_patterns)
                    if has_completion_indicator:
                        logger.debug("RSYNC transfer appears to have completed successfully based on output patterns")
                        
            except pexpect.TIMEOUT:
                logger.error("RSYNC transfer timed out after 1200 seconds")
                raise TimeoutError("transfer timed out")
            except pexpect.EOF:
                # This is expected when transfer completes successfully
                logger.info("RSYNC transfer completed successfully")
        
        # Check exit status
        child.close()
        exit_status = child.exitstatus
        logger.debug("RSYNC transfer exit status: {0}".format(exit_status))
        
        # RSYNC exit status codes and their meanings
        if exit_status == 1:
            logger.error("RSYNC transfer failed: Syntax or usage error (exit code 1)")
            raise RuntimeError("transfer failed: Syntax or usage error")
        elif exit_status == 2:
            logger.error("RSYNC transfer failed: Protocol incompatibility (exit code 2)")
            raise RuntimeError("transfer failed: Protocol incompatibility")
        elif exit_status == 3:
            logger.error("RSYNC transfer failed: Errors selecting input/output files (exit code 3)")
            raise RuntimeError("transfer failed: Errors selecting input/output files")
        elif exit_status == 4:
            logger.error("RSYNC transfer failed: Requested action not supported (exit code 4)")
            raise RuntimeError("transfer failed: Requested action not supported")
        elif exit_status == 5:
            logger.error("RSYNC transfer failed: Error starting client-server protocol (exit code 5)")
            raise RuntimeError("transfer failed: Error starting client-server protocol")
        elif exit_status == 10:
            logger.error("RSYNC transfer failed: Error in socket I/O (exit code 10)")
            raise RuntimeError("transfer failed: Error in socket I/O")
        elif exit_status == 11:
            logger.error("RSYNC transfer failed: Error in file I/O (exit code 11)")
            raise RuntimeError("transfer failed: Error in file I/O")
        elif exit_status == 12:
            logger.error("RSYNC transfer failed: Error in rsync protocol data stream (exit code 12)")
            raise RuntimeError("transfer failed: Error in rsync protocol data stream")
        elif exit_status == 23:
            logger.error("RSYNC transfer failed: Partial transfer due to error (exit code 23)")
            raise RuntimeError("transfer failed: Partial transfer due to error")
        elif exit_status == 30:
            logger.error("RSYNC transfer failed: Timeout in data send/receive (exit code 30)")
            raise TimeoutError("transfer failed: Timeout in data send/receive")
        elif exit_status != 0:
            logger.error("RSYNC transfer failed with exit status: {0}".format(exit_status))
            raise RuntimeError("transfer failed with exit status: {0}".format(exit_status))
        
        logger.info("Successfully transferred {0} to {1}:{2}".format(local_file, ip, remote_dir))
        
    except pexpect.ExceptionPexpect as e:
        logger.error("Pexpect error during file transfer: {0}".format(str(e)))
        raise RuntimeError("process error: {0}".format(str(e)))
    except pexpect.TIMEOUT:
        logger.error("RSYNC process timed out")
        raise TimeoutError("process timed out")
    except IOError as e:
        if hasattr(e, 'errno') and e.errno == 2:
            logger.error("File not found error: {0}".format(str(e)))
        elif hasattr(e, 'errno') and e.errno == 13:
            logger.error("Permission error: {0}".format(str(e)))
        else:
            logger.error("IO error: {0}".format(str(e)))
        raise
    except ValueError as e:
        logger.error("Invalid parameter: {0}".format(str(e)))
        raise
    except Exception as e:
        logger.error("Unexpected error during file transfer: {0}".format(str(e)))
        raise RuntimeError("File transfer failed: {0}".format(str(e)))
    finally:
        # Ensure child process is properly closed
        if child is not None:
            try:
                if child.isalive():
                    child.terminate(force=True)
                    logger.warning("Forcefully terminated RSYNC process")
            except Exception as cleanup_error:
                logger.error("Error during cleanup: {0}".format(str(cleanup_error)))
    
    return

def file_transfer_with_retry(ip, local_file, remote_dir, user, pwd, max_retries=2):
    """
    Transfer a file via RSYNC with retry logic for enhanced reliability.
    
    Args:
        ip (str): Target IP address
        local_file (str): Path to local file to transfer
        remote_dir (str): Remote directory destination
        user (str): SSH username
        pwd (str): SSH password
        max_retries (int): Maximum number of retry attempts
        
    Returns:
        bool: True if transfer successful, False otherwise
        
    Raises:
        ValueError: Invalid parameters
        IOError: File access issues (file not found, permissions)
        RuntimeError: All retry attempts failed
    """
    last_exception = None
    
    for attempt in range(1, max_retries + 1):
        try:
            logger.info("File transfer attempt {0} of {1}".format(attempt, max_retries))
            file_transfer(ip, local_file, remote_dir, user, pwd)
            logger.info("File transfer successful on attempt {0}".format(attempt))
            return True
            
        except (TimeoutError, RuntimeError) as e:
            last_exception = e
            logger.warning("File transfer attempt {0} failed: {1}".format(attempt, str(e)))
            
            if attempt < max_retries:
                # Wait before retry with exponential backoff
                wait_time = min(5 * (2 ** (attempt - 1)), 30)  # Cap at 30 seconds
                logger.info("Waiting {0} seconds before retry...".format(wait_time))
                time.sleep(wait_time)
            else:
                logger.error("All {0} file transfer attempts failed".format(max_retries))
                break
                
        except (ValueError, IOError) as e:
            # These are non-recoverable errors, don't retry  
            # IOError covers both FileNotFoundError and PermissionError in Python 2.7
            logger.error("Non-recoverable error during file transfer: {0}".format(str(e)))
            raise
    
    # If we get here, all retries failed
    if last_exception:
        raise RuntimeError("File transfer failed after {0} attempts. Last error: {1}".format(max_retries, str(last_exception)))
    else:
        raise RuntimeError("File transfer failed after {0} attempts".format(max_retries))

def get_ssh_options():
    """
    Get SSH options for RSYNC to handle various connection scenarios
    Returns a string of SSH options that can be used with rsync --rsh
    """
    ssh_options = [
        '-o', 'StrictHostKeyChecking=no',          # Don't prompt for unknown hosts
        '-o', 'UserKnownHostsFile=/dev/null',      # Don't update known_hosts
        '-o', 'PreferredAuthentications=password', # Prefer password auth
        '-o', 'PubkeyAuthentication=no',           # Disable key-based auth
        '-o', 'ConnectTimeout=10',                 # Connection timeout
        '-o', 'ServerAliveInterval=10',            # Keep connection alive
        '-o', 'ServerAliveCountMax=3',             # Max missed keepalives
        '-o', 'BatchMode=no',                      # Allow interactive prompts
        '-o', 'PasswordAuthentication=yes'         # Enable password auth
    ]
    return ' '.join(ssh_options)

#python mastools_scp.py --destip 10.221.42.87 --localfile /var/nsinstall/build-12.1-55.13_nc_64.tgz --remotepath /var/joan/ --user nsroot --pwd nsroot
def main():
    try:
        argv = sys.argv
        
        # Use RSYNC-based file transfer with retry logic (no verification)
        # Arguments: ip, local_file, remote_dir, user, pwd, max_retries
        file_transfer_with_retry(argv[2], argv[4], argv[6], argv[8], argv[10], max_retries=2)
        logger.info("File transfer completed successfully")
    except Exception as e:
        logger.error('Exception in file transfer: {0}'.format(repr(e)))
        print("Error: %s" % str(e))
        sys.exit(-1)


if __name__ == '__main__':
    main()
    sys.exit(0)
