"""
Copyright 2000-2024 Citrix Systems, Inc. All rights reserved.
This software and documentation contain valuable trade secrets
and proprietary property belonging to Citrix Systems, Inc.
None of this software and documentation may be copied,
Duplicated or disclosed without the express written permission
of Citrix Systems, Inc.
"""


import json
import os
import fcntl
import sys
from functools import wraps
import sys
import threading
import time
import subprocess
import shlex

#TODO: Add azurelinuxagent from NS Python contrib
#from azurelinuxagent.common.osutil.nsvpx import NSVPXOSUtil
from rainman_core.common.logger import RainLogger
from rainman_core.common.exception import *

log = RainLogger.getLogger()


# Decorate config modifying functions with sync/save
def config_readwrite(func):
    def wrapper(*args, **kargs):
        try:
            config_service = args[1]
            config_service._sync_config()
        except config_sync_not_required:
            pass

        ret = func(*args, **kargs)

        config_service._save_config()
        return ret
    return wrapper


# Decorate config reading functions with sync
def config_read(func):
    @wraps(func)
    def wrapper(*args, **kargs):
        try:
            config_service = args[1]
            config_service._sync_config()
        except config_sync_not_required:
            pass

        ret = func(*args, **kargs)
        return ret
    return wrapper


class rainman_config_entry(object):
    log = log

    @config_read
    def get(self, config_service, entry_name=None):
        '''
        Returns all objects with "entry_name".
        '''

        '''
		All config entries are lists at the top level of the object.
		Append an "s" to a rainman_config_entry object name to get the plural list ;)

		ex: json from conf file
		"groups": [
			{
				"name":"OWA",
				"port":80,
				"protocol":"http",
				"drain":"yes",
				"drainTime":300,
			}
		],
		'''
        try:
            config_service.config[self._type_to_json_key()]
        except KeyError:
            config_service.config[self._type_to_json_key()] = []
        finally:
            entries = config_service.config[self._type_to_json_key()]

        if entry_name == None:
            # If no name is specified, return all of the configured entries
            target_entries = entries
        else:
            # Search for entries with a matching name
            try:
                target_entries = [
                    entry for entry in entries if entry['name'] == entry_name]
            except KeyError:
                # All entries should have a name attribute, complain if they dont.
                raise config_file_error(
                    '%s contains a malformed %s object, name is missing'
                    % (config_service.config_file, self._entry_type()))

            # With a name specified, we should only find one object.
            try:
                target_entries[1]
            except IndexError:
                # Only one, great.
                pass
            else:
                # More than one, complain.
                raise config_file_error(
                    '%s contains duplicate entries for %s with name %s'
                    % (config_service.config_file, self._entry_type(), entry_name))

        # For name and non name lookup, we need to return at least one...
        if not target_entries:
            raise config_not_present('%s of type %s is not configured.' % (
                entry_name, self._entry_type()))

        '''
		Get instances of the current config entry object to store the results

		Module would be "rainman.common.rain"
		Class instance fetched via attribute could be "group"
		'''
        module = sys.modules[__name__]

        # Make sure our module actually contains one of these classes
        config_entry_class = getattr(module, self._entry_type())

        result_objs = []
        for target_entry in target_entries:

            # Then get an instance of the class, ex. group()
            result_obj = config_entry_class()

            for attr_name in self._config_attrs():
                '''
                ex: 'group' attrs will be from __init__ only.

                class group(rainman_config_entry):
                        def __init__(self):
                                self.name=None
                                self.port=None
                                self.protocol=None
                                self.drain=None
                                self.drain_time=None
                '''
                try:
                    result_obj.__dict__[attr_name] = target_entry[attr_name]
                except KeyError:
                    '''
                    This happens when the config file entry (target_entry) doesnt match the result object we are trying to fill.
                    If an object is updated in a later version to add a new attribute, this error will be encountered.
                    For now log and error then skip this attribute so that its non-fatal.
                    '''
                    rainman_config_entry.log.error(
                        "%s exists in the object definition, but is missing from the config file object! Skipping the attribute"
                        % (attr_name))
                    continue

            result_objs.append(result_obj)

        # Name based lookup should just return a single result
        if entry_name != None:
            result_objs = result_objs[0]

        return result_objs

    @config_readwrite
    def add(self, config_service, entry_obj):
        entries = config_service.config[self._type_to_json_key()]
        entries.append(entry_obj.__dict__)
        pass

    @config_readwrite
    def update(self, config_service, entry_obj):
        # enumerate the list of objects from the config file of current class type
        for key, entry in enumerate(config_service.config[self._type_to_json_key()]):
            # Find the object we want to update by name
            if entry['name'] == entry_obj.name:
                # Then update the list in place with the replacement object
                config_service.config[self._type_to_json_key(
                )][key] = entry_obj.__dict__

    @config_readwrite
    def remove(self, config_service, entry_obj):
        for key, entry in enumerate(config_service.config[self._type_to_json_key()]):
            if entry['name'] == entry_obj.name:
                del config_service.config[self._type_to_json_key()][key]

    def _config_attrs(self):
        return list(self.__dict__.keys())

    def _entry_type(self):
        return self.__class__.__name__

    def _type_to_json_key(self):
        return self._entry_type() + "s"


class rainman_config(object):

    def __init__(self):
        rainman_config_entry.log = log
        self.config_file = "/nsconfig/rainman.conf"
        self.config_time = 0
        self.init_rainman_config()
        self.config = self._get_config()
        self.config_time = os.path.getmtime(self.config_file)

    def init_rainman_config(self):
        if os.path.isfile(self.config_file) is not True:
            rainman_config_entry.log.error(
                "Config file: %s does not exist." % self.config_file)
            self.create_rainman_config_file()
        elif os.stat(self.config_file).st_size == 0:
            rainman_config_entry.log.error(
                "Config file: %s is empty." % self.config_file)
            self.create_rainman_config_file()

    def create_rainman_config_file(self):
        rainman_config_entry.log.error(
            "Creating config file %s..." % self.config_file)
        data = {"group_lb_alarm_bindings": [], "services": [], "groups": [],
                "loadbalancers": [], "azuretags": []}
        outfile = open(self.config_file, 'w')
        json.dump(data, outfile)
        outfile.close()

    def get_cloud_config_service(self):
        platform = self._get_cloud_platform()

        if platform == "AWS":
            from rainman_core.drivers.aws import aws_config
            return aws_config()
        elif platform == "AZURE":
            from rainman_core.drivers.azurerain import azure_config
            return azure_config()
        elif platform == "GCP":
            from rainman_core.drivers.gcp import gcp_config
            return gcp_config()
        if platform == "OTHER":
            return

    def get_cloudadapter_service(self, aws_config):
        platform = self._get_cloud_platform()
        if platform == "AWS":
            from rainman_core.drivers.aws import aws_cloudAdapter
            return aws_cloudAdapter(aws_config)

    def get_cloud_reporting_service(self):
        platform = self._get_cloud_platform()
        if platform == "AWS":
            from rainman_core.drivers.aws import aws_config
            return aws_config().get_aws_reporting()
        elif platform == "AZURE":
            from rainman_core.drivers.azurerain import azure_config
            return azure_config().get_azure_reporting()
        if platform == "OTHER":
            return

    def get_local_config_service(self, timeout=500):
        if self._get_local_platform() == "NS":
            from rainman_core.drivers.ns import ns
            return ns(timeout)

    def _get_config(self):
        infile = open(self.config_file, 'r')
        fcntl.flock(infile, fcntl.LOCK_EX)

        config_json = infile.read()
        try:
            config = json.loads(config_json)
        except ValueError as e:
            config = {}
            rainman_config_entry.log.error("Configuration is not valid: %s", e)

        # flock goes away when you call fd.close()
        infile.close()
        self.config_time = os.path.getmtime(self.config_file)
        return config

    def _save_config(self):

        outfile = open(self.config_file, 'w')
        fcntl.flock(outfile, fcntl.LOCK_EX)

        try:
            pretty_config = json.dumps(
                self.config, sort_keys=True, indent=8, separators=(',', ': '))
            outfile.write(pretty_config)
        except:
            rainman_config_entry.log.error("Failed to save config!")
            return
        finally:
            outfile.close()

        rainman_config_entry.log.debug(
            "Ugly config written to disk: %s" % (self.config))
        self._get_config()

        rainman_config_entry.log.debug(
            "Config refreshed from disk: %s" % (self.config))

    def _sync_config(self):
        if self.config_time != os.path.getmtime(self.config_file):
            rainman_config_entry.log.debug("Config file changed, reloading it")
            self.config = self._get_config()
        else:
            raise config_sync_not_required("Configuration is unmodified")

    def _get_cloud_platform(self):
        if "NS_APPLIANCE" in os.environ:
            if "VPX_ON_CLOUD" in os.environ:
                vpx_on_cloud = int(os.environ.get("VPX_ON_CLOUD"))
        else:
            cmd_list = ['sysctl', 'netscaler.vpx_on_cloud']
            vpx_on_cloud = int(self.exec_cmd(cmd_list)[-1])
        if vpx_on_cloud == 1:
            return "AWS"
        elif vpx_on_cloud == 3:
            return "AZURE"
        elif vpx_on_cloud == 4:
            return "GCP"
        else:
            raise Exception("Unknown cloud platform : %d" % (vpx_on_cloud))

    def _get_local_platform(self):
        return "NS"

    def exec_cmd(self, cmd_list):
        try:
            process = subprocess.run(
                cmd_list, stdout=subprocess.PIPE, universal_newlines=True)
            return shlex.split(process.stdout)
        except Exception as err:
            log.error('Command execution failed %s' % err)
            raise

    def upgrade_conf(self):
        # upgrade existing config to match current version
        # introduce service in rainman.conf
        log.info("Checking for rainman.conf upgrade.")
        if not self.config:
            return
        try:
            asgs = group().get(self)
            log.debug(f"{asgs!r}")
            sgs = []
            try:
                sgs = service().get(self)
                log.debug(f"{sgs!r}")
            except:
                if asgs:
                    log.warn("Upgrade conf: Older config file, upgrading.")
            cps = group_lb_alarm_binding().get(self)
            log.debug(f"{cps!r}")

            local = self.get_local_config_service()

            for asg in asgs:
                add_sg_for_asg = False
                default_sg_name = service.get_service_name(asg.name, asg.port)
                lb = None
                for cp in cps:
                    if cp.group == asg.name:
                        if not cp.service:
                            cp.service = default_sg_name
                            cp.update(self, cp)
                            log.info(
                                f"Upgrade Conf: CP updated with service {cp.service!r}")
                            try:
                                sg = service().get(self, default_sg_name)
                            except config_not_present:
                                lb = cp.lb_name
                                add_sg_for_asg = True
                                break
                else:
                    for sg in sgs:
                        if sg.group_name == asg.name:
                            break
                    else:
                        add_sg_for_asg = True
                if add_sg_for_asg:
                    sg = service()
                    sg.name = default_sg_name
                    sg.group_name = asg.name
                    sg.port = asg.port
                    sg.protocol = asg.protocol
                    sg.cloud_profile_type = asg.cloud_profile_type
                    sg.azuretag = asg.azuretag
                    sg.enabled = True
                    sg.add(self, sg)
                    try:
                        local.add_group(sg)
                        if lb:
                            lb_obj = loadbalancer().get(self, lb)
                            local.remove_group_from_lb(asg, lb_obj)
                            local.add_group_to_lb(sg, lb_obj)
                        local.move_vservers_from_sg1_to_sg2(asg, sg)
                        local.remove_group(asg)
                    except Exception as err:
                        log.warning("Exception in local conf update: %s", err)
                    log.info(f"Upgrade Conf: service {sg.name!r} added.")
        except Exception as err:
            log.warn(f"Exception seen while upgrading rainman config {err}")


class group_lb_alarm_binding(rainman_config_entry):
    def __init__(self):
        self.name = None
        self.lb_name = None  # binding name also == lb.name
        self.group = None
        self.service = None
        self.alarms = None
        self.type = None

#config in ns


class loadbalancer(rainman_config_entry):
    def __init__(self):
        self.name = None
        self.lbtype = None
        self.ip = None
        self.port = None
        self.protocol = None

#config in ns and cloud


class service(rainman_config_entry):
    def __init__(self):
        self.name = None
        self.group_name = None
        self.port = None
        self.protocol = None
        self.cloud_profile_type = None
        self.azuretag = None
        self.enabled = False

    @staticmethod
    def get_service_name(group_name, port):
        return (f"_{group_name}_{port!s}")


class group(rainman_config_entry):
    def __init__(self):
        self.name = None
        self.zone = None
        self.rg = None
        self.port = None  # Deprecated, kept for backward compability
        self.protocol = None  # Deprecated, kept for backward compability
        self.drain = None
        self.drain_time = None
        self.drain_count = -1
        self.policy_type = None
        self.policy_adjustment = {}
        self.cloud_profile_type = None  # Deprecated, kept for backward compability
        self.azuretag = None  # Deprecated, kept for backward compability
        self.poll_period = None

    def list_all_services_in_group(self, config):
        if self.name:
            try:
                all_services = service().get(config)
                return [svc for svc in all_services if svc.group_name == self.name]
            except:
                pass
        return []

    @classmethod
    def get_group_service_names(cls, config, name_provided, port=None):
        err_msg = ""
        log.debug(
            f"attempting to get autoscale group name and service name from user input name {name_provided!r} and port {port!r}.")
        substrings = name_provided.split("_")
        substring_size = len(substrings)
        if substring_size >= 3 and not substrings[0]:
            if not substrings[1] or not substrings[-1]:
                # WARNING SCENARIO: autoscale group name starts with '_', may fail.
                err_msg = "autoscale settings name beginning or ending with \"_\" are not supported. please match name as \"_<autoscale_settings_name>_<port>\"."
            elif port == None or substrings[-1] == str(port):
                # name matches "_<grp_name>_<port>"
                grp_name = "_".join(substrings[1:-1])
                if grp_name:
                    return (grp_name, name_provided, err_msg)
                err_msg = "autoscale_settings_name is empty in provided servicegroup name. please match name as \"_<autoscale_settings_name>_<port>\"."
            else:
                err_msg = "Invalid servicegroup name: please match name as \"_<autoscale_settings_name>_<port>\"."
        else:
            # name not starting with '_' are "<grp_name>"
            if not port:
                try:
                    grp = group().get(config, name_provided)
                    port = grp.port
                except:
                    log.warn(
                        f"group {name_provided!r} doesnt exist in rain config yet.")
            if port:
                service_name = service.get_service_name(name_provided, port)
                return (name_provided, service_name, err_msg)
            else:
                return (name_provided, None, err_msg)
        log.warning(
            f"Invalid name provided for autoscale group {name_provided}: {err_msg}")
        return (None, None, err_msg)


class azuretag(rainman_config_entry):
    def __init__(self):
        self.name = None
        self.tag_name = None
        self.tag_value = None

    def set(self, tag_name, tag_value):
        self.name = tag_name+tag_value
        self.tag_name = tag_name
        self.tag_value = tag_value


class server(rainman_config_entry):
    def __init__(self):
        self.name = None
        self.ip = None
        self.state = None

#config in cloud


class alarm(rainman_config_entry):
    def __init__(self):
        self.name = None
        self.alarmtype = None  # cloud or local, is this really needed?
        self.metric = None
        self.threshold = None
        self.comparison = None
        self.period = None
        self.action = None

#config in cloud and rainman


class event_queue(rainman_config_entry):
    def __init__(self):
        self.name = None
        self.details = None
        '''
		Ex. Amazon details
		{ sqs_queuename:"name", "sns_topicname":"name", "sqs_arn":"arn", "sns_arn":"arn"}
		'''

# runtime objects


class group_info(object):
    def __init__(self):
        self.drain = None
        self.name = None
        self.locations = None


class drain(object):
    def __init__(self, group, servers, time):
        self.group = group
        self.servers = servers
        self.drain_time = time
        self.last_activity = None


class event(object):
    def __init__(self, event, group, server, action):
        self.event = event
        self.group = group
        self.server = server
        self.action = action


class stat(object):
    def __init__(self, metric, time, value):
        self.metric = metric
        self.time = time
        self.value = value


class aztags_poller(object):
    def __init__(self, group_name):
        rainman_config_entry.log.debug(
            "Creating aztags_poller for group {}".format(group_name))
        self.is_polling = False
        self.group_name = group_name
        self.rainconfig = rainman_config()
        self.rainlocal = self.rainconfig.get_local_config_service()
        self.azconfig = self.rainconfig.get_cloud_config_service()
        self.rg_name = self.azconfig.get_resource_group_name()
        self.group = group().get(self.rainconfig, self.group_name)
        self.azuretag = azuretag().get(self.rainconfig, self.group.azuretag)
        self.filter_str = "tagName eq \'{}\' and tagValue eq \'{}\'".format(
            self.azuretag.tag_name, self.azuretag.tag_value)
        self.nic_ids_from_vms_query = "where type =~ 'Microsoft.Compute/virtualMachines' and tags.{} == '{}' | project nic = tostring(properties['networkProfile']['networkInterfaces'][0]['id']) | where isnotempty(nic) | distinct nic".format(
            self.azuretag.tag_name, self.azuretag.tag_value)
        self.ips_from_nic_list_query = "where type =~ 'microsoft.network/networkinterfaces' and id in ({}) | project ip = tostring(properties.ipConfigurations[0].properties.privateIPAddress)"
        self.ips_from_nics_query = "where type == 'microsoft.network/networkinterfaces' and tags.{} == '{}' | project ip = tostring(properties.ipConfigurations[0].properties.privateIPAddress)".format(
            self.azuretag.tag_name, self.azuretag.tag_value)
        self.poll_period = int(self.group.poll_period)
        self.timer = looped_timer(self.poll_period, self.poll)
        self.lock = threading.Condition()
        self.resource_counter = 0
        self.servers = list()
        self.ip_list = set()
        self.tagged_nics = list()
        self.api_throttling_delay = 0

    def __del__(self):
        rainman_config_entry.log.debug(
            "Deleting poller for group {}".format(self.group_name))

    def fetch_nic(self, item):
        itemType = item.type
        itemId = item.id
        if ("Microsoft.Compute/virtualMachines" in itemType):
            r = self.azconfig.get_vm_nics(itemId)
            vm_name = r[0]
            networkInterfaces = r[1]
            nic_count = len(networkInterfaces)
            for nic in networkInterfaces:
                # Azure REST API returns nic.primary = NULL if there is only one NIC
                # So we need to set this explictly to true
                if ((nic_count == 1) and (nic.primary is None)):
                    nic.primary = True
                if (nic.primary is True):
                    ip_configs = self.azconfig.get_nic_ip(nic)
                    self.tagged_nics.append(itemId)
                    for item in ip_configs:
                        self.ip_list.add(item.private_ip_address)
                        self._add_server(
                            "{}-{}-{}".format(vm_name, item.name, self.resource_counter), item.private_ip_address)
                        self.resource_counter += 1
        elif ("Microsoft.Network/networkInterfaces" in itemType):
            if (item.id not in self.tagged_nics):
                ip_configs = self.azconfig.get_nic_ip(item)
                self.tagged_nics.append(item.id)
                for item in ip_configs:
                    self.ip_list.add(item.private_ip_address)
                    self._add_server(
                        "{}-{}".format(item.name, self.resource_counter), item.private_ip_address)
                    self.resource_counter += 1
            else:
                rainman_config_entry.log.error(
                    "Item {} already processed".format(item.id))

    def fetch_resources(self):
        start_time = time.time()
        resourceList = self.azconfig.resourceMgmt.resources.list(
            filter=self.filter_str)
        for item in resourceList:
            try:
                self.fetch_nic(item)
                if (self.api_throttling_delay > 0):
                    time.sleep(self.api_throttling_delay)
            except Exception as e:
                rainman_config_entry.log.error(
                    "Error in fetch_resources(): {}".format(str(e)))
                continue
        end_time = time.time()
        rainman_config_entry.log.debug("{} resources retrieved in {} secs for group {}".format(
            self.resource_counter, end_time - start_time, self.group_name))

    def fetch_resources_graph(self):
        start_time = time.time()
        vm_ips = None
        nics_ips = None
        try:
            nics = self.azconfig.resource_graph_request(
                self.nic_ids_from_vms_query)
            if len(nics) > 0:
                str_nics = ', '.join(
                    ['\'{}\''.format(str(s)) for s in nics])
                vm_ips = self.azconfig.resource_graph_request(
                    self.ips_from_nic_list_query.format(str_nics))
        except Exception as e:
            rainman_config_entry.log.error(
                "Error in fetching nics for VMs: {}".format(str(e)))
            pass

        if vm_ips is not None and len(vm_ips) > 0:
            for ip in vm_ips:
                self.ip_list.add(ip)

        try:
            nics_ips = self.azconfig.resource_graph_request(
                self.ips_from_nics_query)
        except Exception as e:
            rainman_config_entry.log.error(
                "Error in fetching nics: {}".format(str(e)))
            pass

        if nics_ips is not None and len(nics_ips) > 0:
            for ip in nics_ips:
                self.ip_list.add(ip)

        for ip in self.ip_list:
            self._add_server("{}".format(str(ip)), ip)
            self.resource_counter += 1

        end_time = time.time()
        rainman_config_entry.log.debug("{} resources retrieved in {} secs for group {}".format(
            self.resource_counter, end_time - start_time, self.group_name))

    def update(self):
        cloud_servers = self.get_servers()
        group_servers = self.rainlocal.get_servers_in_group(self.group)

        add_servers = [x for x in cloud_servers if x.ip not in (
            y.ip for y in group_servers)]
        remove_servers = [x for x in group_servers if x.ip not in (
            y.ip for y in cloud_servers)]

        for server in add_servers:
            self.rainlocal.add_server_to_group(server, self.group)
            rainman_config_entry.log.debug(
                "Adding server {} - ip {} from group {}".format(server.name, server.ip, self.group_name))

        for server in remove_servers:
            self.rainlocal.remove_server_from_group(server, self.group)
            rainman_config_entry.log.debug(
                "Remove server {} - ip {} from group {}".format(server.name, server.ip, self.group_name))

    def poll(self):
        rainman_config_entry.log.debug(
            "Polling for {} started...".format(self.group_name))
        try:
            self.lock.acquire()
            self.clear()
            #TODO: Add azurelinuxagent from NS Python contrib
	     #if NSVPXOSUtil.check_ns_on_azstack():
            #    self.fetch_resources()
            #else:
            self.fetch_resources_graph()
            self.update()
        finally:
            self.lock.notify()
            self.lock.release()
        rainman_config_entry.log.debug(
            "Polling for {} ended...".format(self.group_name))

    def start_polling(self, now=False):
        if not self.is_polling:
            rainman_config_entry.log.debug('Start polling')
            self.is_polling = True
            if (now == True):
                self.poll()
            self.timer.start()
        else:
            rainman_config_entry.log.error(
                "Poller is already polling, can't start again")

    def stop_polling(self):
        if self.is_polling:
            rainman_config_entry.log.debug('Stop polling')
            self.lock.acquire()
            while (self.timer.is_executing):
                rainman_config_entry.log.debug(
                    "Actual polling in progress. Waiting to finish...")
                self.lock.wait()
            rainman_config_entry.log.debug("Actual polling finished!")
            self.is_polling = False
            self.lock.release()
            self.timer.cancel()
        else:
            rainman_config_entry.log.error("Poller is not polling, can't stop")

    def clear(self):
        self.ip_list.clear()
        self.tagged_nics[:] = []
        self.servers[:] = []
        self.resource_counter = 0

    def _add_server(self, name, ip):
        new_server = server()
        new_server.name = name
        new_server.ip = ip
        self.servers.append(new_server)

    def get_servers(self):
        return self.servers


class looped_timer(object):
    def __init__(self, duration, func, *args, **kwargs):
        self.is_active = False
        self.is_executing = False
        self._loop_duration = duration
        self._thread = None
        self._func = func
        self._args = args
        self._kwargs = kwargs

    def _start_exec(self):
        rainman_config_entry.log.debug(
            "Looped Timer -> function execution started")
        self.is_executing = True
        try:
            self._func(*(self._args), **(self._kwargs))
        except Exception as e:
            rainman_config_entry.log.error(
                "Looped Timer -> function execution errored: %s", e)
        self.is_executing = False
        rainman_config_entry.log.debug(
            "Looped Timer -> function execution finished")
        self._start_timer()

    def _start_timer(self):
        if self.is_active:
            self._thread = threading.Timer(
                self._loop_duration, self._start_exec)
            self._thread.daemon = True
            self._thread.start()
            rainman_config_entry.log.debug("Looped Timer started")

    def start(self):
        if not self.is_active and not self.is_executing:
            self.is_active = True
            self._start_timer()
        else:
            rainman_config_entry.log.error(
                "Timer is already running, can't start again")

    def cancel(self):
        if self._thread:
            self.is_active = False
            self._thread.cancel()
            self._thread = None
        else:
            rainman_config_entry.log.error(
                "Timer is not running, can't cancel")
