"""
Copyright 2000-2022 Citrix Systems, Inc. All rights reserved.
This software and documentation contain valuable trade secrets
and proprietary property belonging to Citrix Systems, Inc.
None of this software and documentation may be copied,
duplicated or disclosed without the express written permission
of Citrix Systems, Inc.
"""

import sys
import json

from rainman_core.common import rain
from rainman_core.common.exception import rainman_exception, aws_authetication_failure, azure_authentication_failure
from rainman_core.common.ns_errors import NS_BASE_ERRORS


def res_exit(resp, errno, errmsg):
    result = {'response': resp, 'errno': errno, 'errmsg': errmsg}
    print(json.dumps(result))
    sys.exit(errmsg)


def init_rain_globals():
    config = None
    local = None
    cloud = None
    try:
        config = rain.rainman_config()
    except Exception as err:
        errmsg = f"Unexpected error received trying read rainman config: {err!r}"

    try:
        local = config.get_local_config_service(15)
    except rainman_exception as err:
        errno = str(err).split()[0].rstrip(',')
        errmsg = ' '.join(str(err).split()[1:])
        res_exit('failed', errno, errmsg)
    except Exception as err:
        errmsg = f"Unexpected error received trying connect to local: {err!r}"
        res_exit('failed', 1, errmsg)

    try:
        cloud = config.get_cloud_config_service()
    except (aws_authetication_failure, azure_authentication_failure) as err:
        errno = NS_BASE_ERRORS['NSERR_CLOUD_CRED']['errno']
        desc = NS_BASE_ERRORS['NSERR_CLOUD_CRED']['desc']
        res_exit(desc, errno, str(err))
    except rainman_exception as err:
        errno = NS_BASE_ERRORS['NSERR_IO']['errno']
        desc = NS_BASE_ERRORS['NSERR_IO']['desc']
        res_exit(desc, errno, str(err))
    except Exception as err:
        errno = 1
        errmsg = f"Unexpected error received trying connect to cloud: {err!r}"
        res_exit('failed', 1, errmsg)
    return (config, local, cloud)
