"""
Copyright 2000-2022 Citrix Systems, Inc. All rights reserved.
This software and documentation contain valuable trade secrets
and proprietary property belonging to Citrix Systems, Inc.
None of this software and documentation may be copied,
duplicated or disclosed without the express written permission
of Citrix Systems, Inc.
"""


import logging
import json
import os
import fcntl
import sys
import pkg_resources
import shutil


class stats_config(object):

    def __init__(self):
        self.log = logging.getLogger(__name__)
        self.config_file = "/nsconfig/rain_stats.json"
        self.config_time = 0
        self.init_stats_config()
        self.config = self._get_config()
        self.config_time = os.path.getmtime(self.config_file)

    def init_stats_config(self):
        if os.path.isfile(self.config_file) is not True:
            self.log.warn("Config file: %s does not exist." % self.config_file)
            self.create_stats_config_file()
        elif os.stat(self.config_file).st_size == 0:
            self.log.warn("Config file: %s is empty." % self.config_file)
            self.create_stats_config_file()
    
    def reload_config(self):
        conf = self._get_config()
        self.config = conf

    def create_stats_config_file(self):
        self.log.info("Creating config file %s..." % self.config_file)
        data = pkg_resources.resource_stream(__name__, "rain_stats.json")
        with open(self.config_file, 'wb') as outfile:
            shutil.copyfileobj(data, outfile)
            outfile.close()

    def _get_config(self):
        config = {}
        self.init_stats_config()
        infile = None
        try:
            infile = open(self.config_file, 'r')
            fcntl.flock(infile, fcntl.LOCK_EX | fcntl.LOCK_NB)
            config_json = infile.read()
            config = json.loads(config_json)
        except ValueError as e:
            self.log.error("Configuration is not valid: %s", e)
        except IOError as e:
            self.log.warn("Unable to open or lock config file: %s", e)
        if infile:
            infile.close()
        self.config_time = os.path.getmtime(self.config_file)
        return config

    def get_stat_resolution(self, resolution):
        self.log.debug("Resolution: %s", resolution)
        if resolution == 'high':
            return 1 # 1 sec
        return 60 # default resolution 'standard'
