"""
Copyright 2000-2024 Citrix Systems, Inc. All rights reserved.
This software and documentation contain valuable trade secrets
and proprietary property belonging to Citrix Systems, Inc.
None of this software and documentation may be copied,
duplicated or disclosed without the express written permission
of Citrix Systems, Inc.
"""


import json
import time
import ast
import configparser
import datetime
import re
import os
import urllib3
from functools import wraps

from botocore.client import Config
from botocore.exceptions import ClientError
import boto3
import boto.utils

from rainman_core.common.logger import RainLogger
from rainman_core.common.base import base_cloud_driver
from rainman_core.common.rain import rainman_config, server, group, group_info, event, event_queue
from rainman_core.common.exception import *
from rainman_core.common.ns_errors import *


""" Defining metadata server ip common part """
METADATA_URL = "http://169.254.169.254/latest"
# Any alarm (scale down in our case) can only have 5 entries
# (1 policy ARN + 4 Topic arn in our case)"""
MAX_TOPC_ARN_IN_ALARM = 4

log = RainLogger.getLogger()


class aws_config(base_cloud_driver):
    cur_configured_groups = []
    endpoint_url = {"iam": None ,"ec2": None, "autoscaling":None, "sns":None, "sqs":None, "cloudwatch":None}

    def __init__(self):
        self.http = urllib3.PoolManager()
        self.log = log
        self.missing_iam = False
        region = self.get_own_region()
        if region.startswith("us-isob-"):
            self.endpoint_url["iam"] ="https://iam.us-isob-east-1.sc2s.sgov.gov"
            self.endpoint_url["ec2"] ="https://ec2.us-isob-east-1.sc2s.sgov.gov"
            self.endpoint_url["autoscaling"] ="https://autoscaling.us-isob-east-1.sc2s.sgov.gov"
            self.endpoint_url["sns"] ="https://sns.us-isob-east-1.sc2s.sgov.gov"
            self.endpoint_url["sqs"] ="https://sqs.us-isob-east-1.sc2s.sgov.gov"
            self.endpoint_url["cloudwatch"] ="https://monitoring.us-isob-east-1.sc2s.sgov.gov"
        elif region.startswith("us-iso-"):
            self.endpoint_url["iam"] ="https://iam.us-iso-east-1.c2s.ic.gov"
            self.endpoint_url["ec2"] ="https://ec2.us-iso-east-1.c2s.ic.gov"
            self.endpoint_url["autoscaling"] ="https://autoscaling.us-iso-east-1.c2s.ic.gov"
            self.endpoint_url["sns"] ="https://sns.us-iso-east-1.c2s.ic.gov"
            self.endpoint_url["sqs"] ="https://sqs.us-iso-east-1.c2s.ic.gov"
            self.endpoint_url["cloudwatch"] ="https://monitoring.us-iso-east-1.c2s.ic.gov"
        self.connect_instance_autoscale()
        self.connect_instance_ec2()
        self.timer_duration = 24*60*60
        self.log_folder = "/flash/nsconfig/.AWS/"
        self.SERVER_STANDBY_STR = "Standby"
        self.SERVER_ACTIVE_STR = "InService"

    def do_timer_events(self, queue_conf):
        # Nothing to do in AWS as of now.
        pass

    def is_authenticated(self):
        return True

    def aws_managed_connection(func):
        @wraps(func)
        def wrapper(*args, **kargs):
            self = args[0]
            '''
                    Do any token refresh related activities here.
                    Today in AWS we are taking hard coded tokens.
                    In the future we expect IAM role and that tokens
                    will have some expiry. An expiry variable will
                    need to get added to this class, and then we
                    can use this decorator.

                    EX:

                    if time.time() >= self.nitro_client_expiry:
                            self.nitro_client, self.nitro_client_expiry = self._connect_instance_nitro()
                            self.log.info("Refreshing nitro token at %s, expired at %s" % (str(time.time()), str(self.nitro_client_expiry)))
                    else:
                            pass
            '''
            return func(*args, **kargs)
        return wrapper

    def quick_api_call(self, func):
        @wraps(func)
        def wrapper(*args, **kargs):
            # Default values
            orig_num_retries = 5

            del self.autoscale_connection
            config = Config(retries=dict(max_attempts = 1))
            self.connect_instance_autoscale(config)

            return_val = func(*args, **kargs)

            del self.autoscale_connection
            config = Config(retries=dict(max_attempts = orig_num_retries))
            self.connect_instance_autoscale(config)

            return return_val
        return wrapper

    '''
    Group managment
    '''

    '''
    def add_group(self, group):
            pass
    '''

    @aws_managed_connection
    def get_group(self, group_names=None):
        try:
            result = []
            groups = []
            groups.append(group_names)
            autoscaling_groups = self._get_autoscaling_groups(groups)
            if len(autoscaling_groups) == 0:
                return result
            for autoscaling_group in autoscaling_groups:
                this_group = group()
                this_group.name = autoscaling_group['AutoScalingGroupName']
                self.log.info("Group Name %s " % this_group.name)
                result.append(this_group)
        except:
            raise
        return result

    @aws_managed_connection
    def get_group_info(self, group_names=None):
        try:
            result = []
            groups = []
            groups.append(group_names)
            autoscaling_groups = self._get_autoscaling_groups(groups)
            if len(autoscaling_groups) == 0:
                return result

            self.log.info("get_group_info _get_autoscaling_groups sucess %d" % len(autoscaling_groups))

            for autoscaling_group in autoscaling_groups:
                this_group = group_info()
                this_group.name = autoscaling_group['AutoScalingGroupName']
                this_group.locations = ','.join(autoscaling_group['AvailabilityZones'])
                self.log.info("Group Name %s " % this_group.name)
                self.log.info("Group AZ %s " % this_group.locations)

                if self.get_scale_down_policy(autoscaling_group):
                    this_group.drain=True
                else:
                    this_group.drain=False

                result.append(this_group)
        except Exception as e:
            self.log.exception("get_group_info Exception %s" % str(e))
            raise
        return result

    '''
    def remove_group(self, group):
            pass
    '''


    '''
    Min Server constraint in group
    '''
    @aws_managed_connection
    def get_min_servers_in_group(self, group_name):
        try:
            autoscaling_groups = self._get_autoscaling_groups([group_name])
            if len(autoscaling_groups) == 0:
                return 0
            for autoscaling_group in autoscaling_groups:
                if autoscaling_group['AutoScalingGroupName'] == group_name :
                    return autoscaling_group['MinSize']
        except Exception as e:
            self.log.exception("get_min_server_in_group Exception %s" % str(e))
            pass
        return 0

    '''
    Servers in groups
    '''

    @aws_managed_connection
    def get_servers_in_group(self, group):
        result = []
        self.connect_instance_autoscale()
        self.connect_instance_ec2()
        groups = []
        try:
            self.log.info("Get_servers_in_group %s" % group.name)
            groups.append(group.name)
            autoscaling_groups = self._get_autoscaling_groups(groups)
            if len(autoscaling_groups) == 0:
                return result
            for autoscale_group in autoscaling_groups:
                if autoscale_group['AutoScalingGroupName'] != group.name:
                    continue

                if (len(autoscale_group['Instances']) == 0):
                    self.log.debug("Cloud group %s is empty." % (group.name))
                    return result

                for instance in autoscale_group['Instances']:
                    try:
                        details = self._get_instance_details(instance['InstanceId'])
                        instance_state = details['State']['Name']
                        self.log.debug("State of the instance %s is %s" % (instance['InstanceId'], instance_state))
                        if instance_state != "running" or 'PrivateIpAddress' not in details or details['PrivateIpAddress'] == None:
                            continue
                        this_server = server()
                        this_server.name = instance['InstanceId']
                        this_server.ip = details['PrivateIpAddress']
                        this_server.state = instance['LifecycleState']
                        result.append(this_server)
                        self.log.debug("Cloud group %s contains: %s %s %s" % (group.name, instance['InstanceId'], this_server.ip, this_server.state))
                    except Exception as e:
                        self.log.exception("get_servers_in_group: Exception for get_instance_details %s", str(e))
        except Exception as e:
            self.log.exception("get_servers_in_group Exception %s" % str(e))
            return result
        return result

    @aws_managed_connection
    def add_server_to_group(self, server, group):
        '''
        Scale up for AWS
        Might ignore server details? Need to look into amazon API docs.
        '''
        pass

    @aws_managed_connection
    def remove_server_from_group(self, server, group):
        '''
        Scale down for AWS
        Might ignore server details? Need to look into amazon API docs.
        '''
        self.connect_instance_autoscale()
        try:
            self.autoscale_connection.terminate_instance_in_auto_scaling_group(InstanceId=server.name, ShouldDecrementDesiredCapacity=True)
        except Exception as e:
            self.log.exception("Remove server exception: %s." % (str(e)))


    '''
    Alarms
    TODO: Not ported in from rain_stats.py yet
    '''

    '''
    Alarms in groups
    '''

    '''
    Events
    '''
    def _get_queue_by_name_local(self, queue_name):
        region = self.get_own_region()
        sqs_res = boto3.resource('sqs', region_name=region, endpoint_url=self.endpoint_url["sqs"], verify=False)
        queue = sqs_res.get_queue_by_name(QueueName=queue_name)
        return queue

    def get_event_queue_details(self):
        queue_name = "RAINMAN_QUEUE_%s" % (self.get_own_instanceid())
        topic_name = "RAINMAN_TOPIC_%s" % (self.get_own_instanceid())
        self.log.info("get_event_queue: SQS queue %s and SNS topic %s" % (queue_name, topic_name))

        self.connect_instance_sqs()
        self.connect_instance_sns()

        topic_arn = ""
        queue_arn = ""

        # Check if Queue and Topic are already created.
        self.log.info("add_event_queue_sbr: Get_queue_by_name" )
        try :
            queue = self._get_queue_by_name_local(queue_name)
            # Queue is already created so lets get the information from the config file.

            queue_name = json.loads(queue_conf.details)['sqs_queuename']
            topic_name = json.loads(queue_conf.details)['sns_topicname']
            queue_url = queue.url

            try:
                queue_arn = json.loads(queue_conf.details)['sqs_arn']
            except Exception as e:
                try:
                    queue_arn= self.sqs_connection.get_queue_attributes(QueueUrl=queue_url, AttributeNames=['All'])['Attributes']['QueueArn']
                except Exception as e:
                    self.log.info("Exception getting SQS attributes: %s ", str(e))
            try:
                topic_arn = json.loads(queue_conf.details)['sns_arn']
            except Exception as e:
                self.log.info("Old version of rainman.conf : %s ", str(e))
                topic_arn = self.topic_arn_from_name(self.get_own_region(), topic_name)
                self.log.info("Old version of rainman.conf : Topic_arn %s ", topic_arn)

        except Exception as e:
            self.log.info("Get_event_details: this should happen only in context of rain_config: %s." % (str(e)))
            retry = 3
            while(retry):
                try :
                    self.log.info("Creating queue: %s...: " % queue_name)
                    queue_url = self.sqs_connection.create_queue(QueueName=queue_name)['QueueUrl']
                    queue_arn = self.sqs_connection.get_queue_attributes(QueueUrl=queue_url, AttributeNames=['QueueArn'])['Attributes'] ['QueueArn']
                    self.log.info("Queue created: %s Q-ARN %s." % (queue_name, queue_arn))
                    retry = 0
                except Exception as e:
                    self.log.debug("Create queue exception: %s." % (str(e)))
                    if "You must wait 60 seconds" in str(e):
                        if retry > 0:
                            retry -= 1
                            self.log.info("Retry queue creation in 60 seconds.")
                            time.sleep(60)
                        else:
                            self.log.info("Failed to create queue: %s" % queue_name)
                            raise
                    else:
                        raise
            try:
                self.log.info("Creating topic: %s...: " % topic_name)
                topic_arn = self.sns_connection.create_topic(Name=topic_name)['TopicArn']
                subscription_arn = self.attach_queue_to_topic(queue_url, queue_arn, topic_arn)
            except Exception as e:
                self.log.exception("Create Topic exception: %s." % (str(e)))
        return json.dumps({'sqs_queuename':queue_name, 'sns_topicname':topic_name, "sqs_arn":queue_arn, "sns_arn":topic_arn, "sqs_url": queue_url})


    @aws_managed_connection
    def add_event_queue(self, queue_conf):
        return

    @aws_managed_connection
    def remove_event_queue(self, queue_conf):
        queue_name = json.loads(queue_conf.details)['sqs_queuename']
        topic_name = json.loads(queue_conf.details)['sns_topicname']

        self.connect_instance_sns()
        self.connect_instance_sqs()
        try:
            topic_arn = json.loads(queue_conf.details)['sns_arn']
        except Exception as e:
            self.log.info("Old version of rainman.conf : %s ", str(e))
            topic_arn = self.topic_arn_from_name(self.get_own_region(), topic_name)

        try:
            queue_arn = json.loads(queue_conf.details)['sqs_arn']
        except Exception as e:
            self.log.info("Old version of rainman.conf : %s ", str(e))
            queue_arn = self.queue_arn_from_name(self.get_own_region(), queue_name)

        self.log.info("Deleting SQS queue %s and SNS topic %s" % (queue_name, topic_name))
        subscription_list = self.sns_connection.list_subscriptions_by_topic(TopicArn=topic_arn)['Subscriptions']
        for sub in subscription_list:
            if queue_arn != None and sub['Endpoint'] == queue_arn:
                self.sns_connection.unsubscribe(SubscriptionArn=sub['SubscriptionArn'])
                self.sns_connection.delete_topic(TopicArn=topic_arn)
                break
        if queue_name != None:
            queue = self._get_queue_by_name_local(queue_name)
            queue.delete()

        return

    @aws_managed_connection
    def get_events_from_queue(self, queue_conf):

        queue_name = json.loads(queue_conf.details)['sqs_queuename']
        try:
            queue_url = json.loads(queue_conf.details)['sqs_url']
        except Exception as e:
            queue = self._get_queue_by_name_local(queue_name)
            queue_url = queue.url

        self.log.info("Fetching SQS queue %s" % (queue_name))
        self.connect_instance_sqs()

        messages = []
        queue = None
        try:
            messages = self.sqs_connection.receive_message(QueueUrl=queue_url, MaxNumberOfMessages=10, WaitTimeSeconds=20)
        except Exception as e:
            self.log.exception("Getting message from queue failed. %s" % (str(e)))

        events = []
        if 'Messages' in messages: # when the queue is exhausted, the response dict contains no 'Messages' key
            for message in messages['Messages']:
                try:
                    event = self.message_to_event(message)
                    if event != None:
                        events.append(event)
                except Exception as e:
                    self.log.exception("Converting message to event queue failed. %s" % (str(e)))
                    self.sqs_connection.delete_message(QueueUrl=queue_url,ReceiptHandle=message['ReceiptHandle'])
                    continue
                finally:
                    self.sqs_connection.delete_message(QueueUrl=queue_url,ReceiptHandle=message['ReceiptHandle'])
        return events

    # configure_events_for_groups called by rain_scale.
    # ASG Policy, Alarm and Notification is already done during rain_config.
    # Hence verify configuration already done when rain_scale is running.

    @aws_managed_connection
    def configure_events_for_groups(self, queue_conf, group_names):
        added_configured_groups = []
        removed_configured_groups = []

        for group_name in group_names:
            if group_name not in self.cur_configured_groups:
                if self._check_configure_notification_asg(group_name, queue_conf):
                    added_configured_groups.append(group_name)

        for group_name in self.cur_configured_groups:
            if group_name not in group_names:
                removed_configured_groups.append(group_name)

        for group_name in removed_configured_groups:
            self.cur_configured_groups.remove(group_name)

        for group_name in added_configured_groups:
            self.cur_configured_groups.append(group_name)
        return

    def _check_configure_notification_asg(self, asg_name, queue_conf):

        topic_name = json.loads(queue_conf.details)['sns_topicname']
        self.log.debug("Fetching SNS topic %s" % (topic_name))
        try:
            topic_arn = json.loads(queue_conf.details)['sns_arn']
        except Exception as e:
            self.log.info("Old version of rainman.conf : %s ", str(e))
            topic_arn = self.topic_arn_from_name(self.get_own_region(), topic_name)

        self.log.info("asg '%s' -> Check notification for '%s'" % (asg_name, topic_arn))
        try:
            notification_list = self.autoscale_connection.describe_notification_configurations(
                    AutoScalingGroupNames=[asg_name])['NotificationConfigurations']

            for notification in notification_list:
                if (notification['AutoScalingGroupName'] == asg_name) and (notification['TopicARN'] == topic_arn):
                    self.log.info("asg '%s'-> notification configured" % asg_name)
                    return True
        except Exception as err:
            self.log.exception("Failed to describe notification in Auto Scaling group: %s" % str(err))
            return False


    def _configure_notification_asg(self, asg, topic_arn):
        self.log.info("asg '%s' -> Configuring notification for '%s'" % (asg['AutoScalingGroupName'], topic_arn))
        try:
            response = self.autoscale_connection.put_notification_configuration(
                    AutoScalingGroupName=asg['AutoScalingGroupName'],
                    NotificationTypes=[
                            'autoscaling:EC2_INSTANCE_LAUNCH',
                            'autoscaling:EC2_INSTANCE_LAUNCH_ERROR',
                            'autoscaling:EC2_INSTANCE_TERMINATE',
                            'autoscaling:EC2_INSTANCE_TERMINATE_ERROR'
                    ],
                    TopicARN=topic_arn)
        except Exception as err:
            if "LimitExceeded" in str(err):
                self.log.warn("Maximum limit for count of Auto Scaling group '%s'-> notification(or topic) has reached. Please manually delete existing notification from group" % asg['AutoScalingGroupName'])
            else:
                self.log.exception("Failed to create notification in Auto Scaling group: %s" % str(err))
            return False
        self.log.info("asg '%s'-> notification configured" % asg['AutoScalingGroupName'])
        return True

    def remove_notification_config_from_group(self, queue_conf, group_name):
        self.log.info("asg '%s' -> Remove notification" % (group_name))
        self.connect_instance_autoscale()
        self.connect_instance_sns()

        topic_name = json.loads(queue_conf.details)['sns_topicname']
        self.log.debug("Fetching SNS topic %s" % (topic_name))
        try:
            topic_arn = json.loads(queue_conf.details)['sns_arn']
        except Exception as e:
            self.log.info("Old version of rainman.conf : %s ", str(e))
            topic_arn = self.topic_arn_from_name(self.get_own_region(), topic_name)

        try:
            self.autoscale_connection.delete_notification_configuration(AutoScalingGroupName=group_name, TopicARN=topic_arn)
            self.log.info("Removed notification configuration from asg '%s'" % group_name)
            return True
        except Exception as err:
            self.log.exception("Exception seen while removing notification configuration: %s" % str(err))
        self.log.info("No notification configuration found for asg '%s'. Continuing..." % group_name)
        return False

    '''
            Input : Policy
            Output : Policy_adjustment series- Policy_name:<adjustments1>,<adjustment2>
    '''
    def _scale_down_policy_str(self, p):
        if (p['PolicyType'] == "SimpleScaling"):
            if p['ScalingAdjustment'] :
                new_description = "(%s:%d)" % (p['PolicyName'], p['ScalingAdjustment'])
                return new_description
        if p['PolicyType'] == "StepScaling" :
            adjust_description = ""
            description = "(%s: ["% p['PolicyName']
            for adjustment in p['StepAdjustments']:
                if adjustment['ScalingAdjustment'] < 0 :
                    # Just to maintain the conf format put a value 999 to show negative infinity.
                    if 'MetricIntervalLowerBound' not in adjustment:
                        LowerBound = -999
                        self.log.info("Lower bound not found, setting it up")
                    else:
                        LowerBound = adjustment['MetricIntervalLowerBound']
                    adjust_description += " [{}, {}, {}],".format(LowerBound, \
                    adjustment['MetricIntervalUpperBound'], \
                    adjustment['ScalingAdjustment'])
                    self.log.info("_scale_down_policy_str %s", adjust_description)
            if len(adjust_description):
                adjust_description = adjust_description[:-1] # Remove last comma
                new_description = description + adjust_description + "]" + ")"
                self.log.info("_scale_down_policy_str Final %s", new_description)
                return new_description
        return ""

    def _configure_scaling_policy_asg(self, asg, topic_arn):
        self.log.info("Configuring scaling policies for asg '%s'" % asg['AutoScalingGroupName'])
        try:
            down_policies = self.get_scale_down_policy(asg)
            if not down_policies:
                self.log.info("Did not find a scale down policy for %s" % asg['AutoScalingGroupName'])
                return False
            for p in down_policies:
                self.log.info("Configuring scaling policy '%s'" % p['PolicyName'])
                old_alarms = p['Alarms']
                new_description=""
                scale_adjust = 0
                policy_arn = p['PolicyARN']

                new_description = self._scale_down_policy_str(p)

                config = rainman_config()
                target_group = group()
                target_group = target_group.get(config, asg['AutoScalingGroupName'])
                target_group.policy_adjustment[p['PolicyName']] = new_description
                target_group.policy_type = p['PolicyType']
                target_group.update(config, target_group)


                self.log.info("Configure scaling Policy -> Looking at Alarms ")
                asg_entry = {'Name':'AutoScalingGroupName', 'Value':asg['AutoScalingGroupName']}

                is_topic_added = False

                for alarm in old_alarms:
                    self.log.info("Configure scaling Policy -> Describe_alarm %s" % alarm['AlarmName'])
                    # TBD: handle the case of same alarm on multiple ASG

                    this_alarm = self.cloudwatch_connection.describe_alarms(AlarmNames=[alarm['AlarmName']])['MetricAlarms'][0]

                    self.log.info("Configure scaling Policy -> Check if ASG in Alarm dimension %s" % asg_entry)

                    if asg_entry not in this_alarm['Dimensions']:
                        self.log.info("Configure Scaling policy -> skipping alarm %s", this_alarm['AlarmName'])
                        continue

                    self.log.info("Configure scaling Policy -> Check alarm_actions and update")

                    alarm_actions = this_alarm['AlarmActions']

                    if policy_arn not in this_alarm['AlarmActions']:
                        self.log.info("Configure scaling Policy -> Append PolicyArn %s", policy_arn)
                        alarm_actions.append(policy_arn)
                    else:
                        self.log.info("Configure scaling Policy -> PRESENT in alarm PolicyArn %s", policy_arn)

                    if len(alarm_actions) >= MAX_TOPC_ARN_IN_ALARM:
                        self.log.info("Configure Topic ARN -> Topic ARN cannot be added as there are other entries in alarm. If there are any stale entries please cleanup! Topic :%s", topic_arn)
                    elif topic_arn not in this_alarm['AlarmActions']:
                        self.log.info("Configure Topic ARN -> Append Topic ARN %s", topic_arn)
                        alarm_actions.append(topic_arn)
                        is_topic_added = True
                    else:
                        self.log.warn("ALARM ALREADY CONFIGURED WITH TOPIC_ARN. USING SAME ALARM WITH MULTIPLE POLICIES IS NOT SUPPORTED CURRENTLY (MIGHT BE SUPPORTED IN FUTURE). THIS DEPLOYMENT IS UNSUPPORTED AND MIGHT FAIL.")


                    # TBD: Bug if AlarmDescription is not there.

                    self.log.info("Configure scaling Policy -> Alarm Description")
                    alarm_description = ""

                    if 'AlarmDescription' in this_alarm:
                        alarm_description += this_alarm['AlarmDescription']

                    self.log.info("Configure scaling Policy -> AlarmDescr %s len %d" % (alarm_description, len(alarm_description)))

                    if 'RainmanScaleDownAlarm' not in alarm_description:
                        if len(alarm_description) > 150:
                            alarm_description = alarm_description[0:145]
                            alarm_description += "...;"
                        alarm_description += 'RainmanScaleDownAlarm'
                    alarm_description += new_description
                    if len(alarm_description) > 255:
                        self.log.warn("Alarm description crossed max char limit of 255. truncating to 255 length.")
                        alarm_description = alarm_description[0:254]

                    self.log.info("Configure scaling policy -> Modifying alarm %s" % (this_alarm['AlarmName']))

                    self.cloudwatch_connection.put_metric_alarm(AlarmName=this_alarm['AlarmName'],
                                    AlarmDescription=alarm_description,
                                    MetricName=this_alarm['MetricName'],
                                    Namespace=this_alarm['Namespace'],
                                    Period=this_alarm['Period'],
                                    EvaluationPeriods=this_alarm['EvaluationPeriods'],
                                    Threshold=this_alarm['Threshold'],
                                    ComparisonOperator=this_alarm['ComparisonOperator'],
                                    AlarmActions=alarm_actions,
                                    Dimensions=this_alarm['Dimensions'],
                                    Statistic=this_alarm['Statistic'])
                    self.log.info("Configure scaling policy -> Alarm modified with topic_arn '%s'" % (topic_arn))

                if is_topic_added == True:
                    # Modify the alarm only if topic ARN was added.
                    # Simple Scaling for now - AdjustmentType = ChangeInCapacity
                    # Step Scaling MetricAggregationType is supported.

                    # Target Tracking - Set TargetTrackingConfiguration
                    if (p['PolicyType'] == "SimpleScaling") :
                        new_policy_arn = self.autoscale_connection.put_scaling_policy(
                                AutoScalingGroupName=asg['AutoScalingGroupName'],
                                PolicyName=p['PolicyName'],
                                AdjustmentType=p['AdjustmentType'],
                                PolicyType =p['PolicyType'],
                                ScalingAdjustment=0)['PolicyARN']

                    if (p['PolicyType'] == "StepScaling") :
                        for step_adjust in p['StepAdjustments']:
                            if step_adjust['ScalingAdjustment'] < 0:
                                step_adjust['ScalingAdjustment'] = 0

                        new_policy_arn = self.autoscale_connection.put_scaling_policy(
                                AutoScalingGroupName=asg['AutoScalingGroupName'],
                                PolicyName=p['PolicyName'],
                                PolicyType =p['PolicyType'],
                                AdjustmentType=p['AdjustmentType'],
                                StepAdjustments=p['StepAdjustments']) ['PolicyARN']

                    if not new_policy_arn:
                        self.log.warn("Configure scaling policy -> Unable to find newly created policy.")
                    else:
                        if (policy_arn != new_policy_arn):
                            self.log.warn("Old and new policy DO NOT MATCH******")
                            policy_arn = new_policy_arn
                        self.log.info("Configure scaling policy -> Policy %s changed Scaling to 0" % (p['PolicyName']))
                else:
                    self.log.info("Topic ARN was not updated in scale Down alarm. Hence skipping modifying step_adjust in %s",p['PolicyName'])
                    self.log.info("Connection draining feature will not work! But scale down will happen")

            self.log.info("All scaling policies/Alarms configured in asg '%s'" % asg['AutoScalingGroupName'])
            return True
        except Exception as err:
            self.log.exception("Failed to configure scaling policies in Auto Scaling group %s: %s" % (asg['AutoScalingGroupName'],str(err)))
            raise config_failed("Did not find a scaling  policy for %s" % asg['AutoScalingGroupName'])
        return False

    '''
    _revert_scaling_policy
            - Find all Scale down policies for this ASG.
            - For all Scale down Polices Find Alarms
            - For each Alarm for this policy ensure this alarm is in context of this ASG.
            - Find the description stored and extract scaling Adjustment
            - Program the Scaling Policy by reverting to old Scaling Adjustment
            - Obtain new Policy_arn
            - FOr all alarms for the old policy
            - Revert the SNS notification and change the description to reflect original user set description.
            - Prgram the new Alarm.
    '''

    def alarm_modified_by_ns(self,alarm, asg):
        if ('AlarmDescription' in alarm ) and ("RainmanScaleDownAlarm" in alarm['AlarmDescription']):
            return True
        self.log.info("alam modified by ns-> skipping alarm %s", alarm['AlarmName'])
        return False

    '''

            # scale adjust is one element for simple scaling
            # Scale adjust is dictionarylist for step scaling

    '''



    def get_scale_adjustment(self, p, description ):
        self.log.info("Revert_policy - Found NS configured Policy")
        if p['PolicyType'] == 'SimpleScaling':
            scale_adjust = 0
            split_list1 = description.split("(%s:" % p['PolicyName'])
            if len(split_list1) == 2:
                scale_adjust = int(split_list1[-1].split(")", 1)[0])
                self.log.info("Revert_policy - Policy Scaling extracted%d " % scale_adjust)

            else:
                config = rainman_config()
                target_group = group().get(config, asg['AutoScalingGroupName'])
                scale_adjust = target_group.policy_adjustment.get(p['PolicyName'])

            remove_description = "({}:{})".format(p['PolicyName'], scale_adjust)
            return_both = (scale_adjust, remove_description)
            return return_both

        if p['PolicyType'] == 'StepScaling':
            found = False
            description = description.replace("RainmanScaleDownAlarm", "")
            this_policy_set=","
            temp = description

            while found == False and len(temp) :
                result = re.search('((.*)]])', temp) # Get the first set
                this_policy_set = result.group(1) + "),"
                if p['PolicyName'] in this_policy_set:
                    found = True
                else :
                    temp = temp.replace(this_policy_set, "") # Take one Policy set out

            remove_description = this_policy_set[:-1]
            self.log.info("Revert Policy Remove Description", remove_description)

            split_list1 = description.split("(%s: [" % p['PolicyName'])
            scale_adjust_str = (split_list1[-1]).replace(")","")
            scale_adjust_str = (split_list1[-1]).replace("]]","")
            self.log.info("StepScaling: SCALE_ADJUST_STR {}".format(scale_adjust_str))

            scale_adjust_list = []

            for each_adjust in scale_adjust_str.split("],"):
                scale_adjust = {}
                each_adjust = each_adjust.replace("[", "")
                each_adjust = each_adjust.replace(")", "")
                each_set = str(each_adjust).split(",")

                self.log.info("each set 0:{} 1 :{} 2:{}".format(each_set[0], each_set[1], each_set[2]))

                if "-999" not in each_set[0]:
                    scale_adjust['MetricIntervalLowerBound'] = float(each_set[0])
                scale_adjust['MetricIntervalUpperBound'] = float(each_set[1])
                scale_adjust['ScalingAdjustment'] = int(each_set[2])
                scale_adjust_list.append(scale_adjust)

            return_both =(scale_adjust_list, remove_description)

            return return_both

    def _revert_scaling_policy_asg(self, asg, topic_arn):
        self.log.info("Revert configuration of scaling policies for asg '%s'" % asg['AutoScalingGroupName'])
        try:
            down_policies = self.get_scale_down_policy(asg)
            for p in down_policies:
                old_alarms = p['Alarms']
                remove_from_description = ""
                new_policy = None
                policy_arn = p['PolicyARN']

                if not self.policy_modified_by_ns(asg, p):
                    continue

                self.log.info("Revert policy configuration of '%s'" % p['PolicyName'])

                try:
                    for alarm in p['Alarms']:
                        these_alarms = self.cloudwatch_connection.describe_alarms(AlarmNames=[alarm['AlarmName']])['MetricAlarms']
                        for this_alarm in these_alarms:
                            if not self.alarm_modified_by_ns(this_alarm, asg):
                                continue


                            if str("(%s:" % (p['PolicyName'])) not in this_alarm['AlarmDescription']:
                                continue

                            scale_adjust, remove_from_description = self.get_scale_adjustment(p, this_alarm['AlarmDescription'])
                            self.log.info("Revert policy => scale_adjust string %s", remove_from_description)

                            if (p['PolicyType'] == "SimpleScaling") :
                                new_policy = self.autoscale_connection.put_scaling_policy(
                                AutoScalingGroupName=asg['AutoScalingGroupName'],
                                PolicyName=p['PolicyName'],
                                AdjustmentType=p['AdjustmentType'],
                                PolicyType =p['PolicyType'],
                                ScalingAdjustment=scale_adjust)

                            if (p['PolicyType'] == "StepScaling") :
                                new_policy = self.autoscale_connection.put_scaling_policy(
                                        AutoScalingGroupName=asg['AutoScalingGroupName'],
                                        PolicyName=p['PolicyName'],
                                        PolicyType=p['PolicyType'],
                                        AdjustmentType=p['AdjustmentType'],
                                        StepAdjustments=scale_adjust)

                            if new_policy:
                                new_policy_arn = new_policy['PolicyARN']
                                self.log.info(
                                        "Revert policy -> policy reverted for Policy:%s in ASG:%s"
                                        % ( p['PolicyName'], asg['AutoScalingGroupName']))
                                policy_arn = new_policy_arn
                            else:
                                self.log.warn("Revert policy -> Unable to find newly created policy")
                except Exception as err:
                    self.log.exception("Exception seen while reverting Policy: %s" % str(err))
                    continue

                for alarm in old_alarms:
                    self.log.info("Revert policy -> Found alarm '%s'" % (alarm['AlarmName']))
                    these_alarms = self.cloudwatch_connection.describe_alarms(AlarmNames=[alarm['AlarmName']])['MetricAlarms']

                    for this_alarm in these_alarms:
                        if not self.alarm_modified_by_ns(this_alarm, asg):
                            continue

                        self.log.info("Revert policy -> Modifying alarm '%s'" % (alarm['AlarmName']))

                        alarm_actions = this_alarm['AlarmActions']

                        self.log.info("Revert policy -> Modifying alarm '%s'" % (alarm['AlarmName']))
                        self.log.info("Revert policy => removeDescription %s", remove_from_description)
                        self.log.info("Revert policy => thisAlarmDesc %s", this_alarm['AlarmDescription'])



                        if new_policy and policy_arn not in alarm_actions:
                            self.log.info("Revert policy -> Alarm action missing policy! This is Weird..")
                            alarm_actions.append(policy_arn)

                        if topic_arn in alarm_actions:
                            self.log.info("Revert policy -> Alarm remving Topic ")
                            alarm_actions.remove(topic_arn)

                        if remove_from_description and 'AlarmDescription' in this_alarm and \
                                remove_from_description in this_alarm['AlarmDescription']:
                            # Alarm -> description

                            ns_configured_actions = [a for a in alarm_actions if "RAINMAN_TOPIC" in a ]
                            if not ns_configured_actions:
                                self.log.info("Revert policy -> resetting description")
                                this_alarm['AlarmDescription'] = this_alarm['AlarmDescription'].replace('RainmanScaleDownAlarm', '')
                            this_alarm['AlarmDescription'] = this_alarm['AlarmDescription'].replace(remove_from_description, '')

                        self.cloudwatch_connection.put_metric_alarm(AlarmName=this_alarm['AlarmName'],
                                AlarmDescription=this_alarm['AlarmDescription'],
                                MetricName=this_alarm['MetricName'],
                                Namespace=this_alarm['Namespace'],
                                Period=this_alarm['Period'],
                                EvaluationPeriods=this_alarm['EvaluationPeriods'],
                                Threshold=this_alarm['Threshold'],
                                ComparisonOperator=this_alarm['ComparisonOperator'],
                                AlarmActions=alarm_actions,
                                Dimensions=this_alarm['Dimensions'],
                                Statistic=this_alarm['Statistic'])

                        self.log.info("Revert policy -> Alarm modified")

            self.log.info("All scaling policies reverted in asg '%s'" % asg['AutoScalingGroupName'])
            return True
        except Exception as err:
            self.log.exception("Failed to revert configuration to scaling policies in Auto Scaling group: %s" % str(err))
        return False



    def configure_events_for_group(self, queue_conf, group):
        self.connect_instance_sns()
        self.connect_instance_cloudwatch()
        self.connect_instance_autoscale()


        topic_name = json.loads(queue_conf.details)['sns_topicname']
        self.log.debug("Fetching SNS topic %s" % (topic_name))
        try:
            topic_arn = json.loads(queue_conf.details)['sns_arn']
        except Exception as e:
            self.log.info("Old version of rainman.conf : %s ", str(e))
            topic_arn = self.topic_arn_from_name(self.get_own_region(), topic_name)

        self.log.info("configure_events_for_group SNS ARN %s" % (topic_arn))

        asg = self.autoscale_connection.describe_auto_scaling_groups(AutoScalingGroupNames=[group.name])['AutoScalingGroups'][0]
        self.log.info("configure_events_for_group : %s ", asg['AutoScalingGroupName'])

        if asg :
            if not self._configure_notification_asg(asg, topic_arn):
                return False
            if group.drain_time != '0':
                if not self._configure_scaling_policy_asg(asg, topic_arn):
                    return False
            return True
        return False

    def cleanup_on_group_remove(self, group):
        try:
            self.log.info("Cleanup routine for ASG %s started" % group.name)
            queue_conf = event_queue().get(rainman_config(), "default")
            topic_name = json.loads(queue_conf.details)['sns_topicname']
            self.connect_instance_sns()

            self.log.debug("Fetching SNS topic %s" % (topic_name))
            try:
                topic_arn = json.loads(queue_conf.details)['sns_arn']
            except Exception as e:
                self.log.info("Old version of rainman.conf : %s ", str(e))
                topic_arn = self.topic_arn_from_name(self.get_own_region(), topic_name)

            self.log.info("cleanup_on_group_remove: GroupName %s SNS ARN : %s" % (group.name, topic_arn))

            asg = self.autoscale_connection.describe_auto_scaling_groups(AutoScalingGroupNames=[group.name])['AutoScalingGroups'][0]

            if topic_arn:
                self._revert_scaling_policy_asg(asg, topic_arn)
                self.log.info("Cleanup: revert scaling Policy '%s'" % topic_arn)
            else:
                self.log.warn("Cleanup: Failed to get topic arn")
            self.log.info("Cleanup for asg '%s' done" % group.name)
            return True
        except Exception as err:
            self.log.exception("Cleanup failed for Auto Scaling group: %s" % str(err))
        return False


        '''
        Misc
        '''
    # Scale down policies which are pre-configured by NS
    def policy_modified_by_ns(self, asg, p):
        for alarm in p['Alarms']:
            alarm_name = alarm['AlarmName']
            self.log.info("get_scale_down_policy_subr: Alarm %s " % alarm_name)
            these_alarms = self.cloudwatch_connection.describe_alarms(AlarmNames=[alarm_name]) ['MetricAlarms']
            for this_alarm in these_alarms:
                self.log.info("get_scale_down_policy_subr: This_alarm Name %s" % this_alarm['AlarmName'])
                if 'AlarmDescription' in this_alarm and ("RainmanScaleDownAlarm" in this_alarm['AlarmDescription']):
                    self.log.info("get_scale_down_policy_subr: This_alarm decsription %s" % this_alarm['AlarmDescription'])
                    return True
        return False

    def get_scale_down_policy(self, asg):
        self.connect_instance_autoscale()
        self.connect_instance_cloudwatch()
        self.log.info("get_scale_down_policy_subr: ASG %s " % asg['AutoScalingGroupName'])
        matched = []

        try:
            policies = self.autoscale_connection.describe_policies(AutoScalingGroupName=asg['AutoScalingGroupName'])['ScalingPolicies']

            for p in policies:
                if (p['PolicyType'] == "SimpleScaling" and (p['ScalingAdjustment'] < 0)):
                    matched.append(p)
                    continue

                if (p['PolicyType'] == "SimpleScaling" and p['ScalingAdjustment'] == 0):
                    if self.policy_modified_by_ns(asg, p):
                        matched.append(p)
                        continue

                if (p['PolicyType'] == "StepScaling") :
                    for adjustment in p['StepAdjustments']:
                        if adjustment['ScalingAdjustment'] < 0 :
                            matched.append(p)
                            break;
                        if (adjustment['ScalingAdjustment'] == 0) and self.policy_modified_by_ns(asg, p):
                            matched.append(p)
                            break

            self.log.info("Scale down policies for asg '%s': %s" % (asg['AutoScalingGroupName'], matched))
        except Exception as e:
            self.log.exception("get_scale_down_policy Exception : %s" % str(e))
        return matched

    def parse_slow_server_alarm(self, body):
        region = self.get_own_region()
        boto3clientCW = boto3.client('cloudwatch', region_name=region)
        try:
            alarm_name = body['AlarmName']
            response = boto3clientCW.describe_alarms(AlarmNames=[alarm_name])['MetricAlarms']
            for alarm in response:
                state_reason_data = json.loads(alarm['StateReasonData'])
                return state_reason_data['asg_name'],state_reason_data['slow_server']
            return None,None
        except Exception as e:
            self.log.info("Exception in parse_slow_server_alarm: %s"% str(e))
            return None,None
    '''
            Sample Message on Alarm:
                    "Message" : "{\"AlarmName\":\"STEP_SCALED_LOW_CPU\",
                            \"AlarmDescription\":\" ABCRainmanScaleDownAlarm(STEP_SCALE_DOWN:-3 )\",
                            \"AWSAccountId\":\"074724716397\",
                            \"NewStateValue\":\"ALARM\",
                            \"NewStateReason\":\"Threshold Crossed: 1 datapoint [0.0999999999999848 (04/12/18 21:17:00)] was less than or equal to the threshold (2.0).\",
                            \"StateChangeTime\":\"2018-12-04T21:22:59.517+0000\",
                            \"Region\":\"US East (N. Virginia)\",
                            \"OldStateValue\":\"OK\",
                            \"Trigger\":{
                                    \"MetricName\":\"CPUUtilization\",
                                    \"Namespace\":\"AWS/EC2\",
                                    \"StatisticType\":\"Statistic\",
                                    \"Statistic\":\"AVERAGE\",
                                    \"Unit\":null,
                                    \"Dimensions\":[
                                            {
                                                    \"value\":\"STEP_SCALED_ASG\",
                                                    \"name\":\"AutoScalingGroupName\"
                                            }],
                                    \"Period\":300,
                                    \"EvaluationPeriods\":1,
                                    \"ComparisonOperator\":\"LessThanOrEqualToThreshold\",
                                    \"Threshold\":2.0,\"TreatMissingData\":\"\",
                                    \"EvaluateLowSampleCountPercentile\":\"\"
                            }
                    }",
            Sample message for test notification :
                    "Message" : "{
                            \"AccountId\":\"074724716397\",
                            \"RequestId\":\"0c7e11a9-f80a-11e8-b002-35513163c285\",
                            \"AutoScalingGroupARN\":\"arn:aws:autoscaling:us-east-1:074724716397:autoScalingGroup:5fb5d107-80ba-4013-aad8-6751a741f017:autoScalingGroupName/STEP_SCALED_ASG\",
                            \"AutoScalingGroupName\":\"STEP_SCALED_ASG\",
                            \"Service\":\"AWS Auto Scaling\",
                            \"Event\":\"autoscaling:TEST_NOTIFICATION\",
                            \"Time\":\"2018-12-04T21:17:48.106Z\"
                            }"

    '''
    def message_to_event(self, message):
        self.log.info("=="*80)

        main_body = message['Body']
        self.log.info("Received Body: %s" % (main_body))

        try:
            body = json.loads(main_body).get("Message")
            body = json.loads(body)
        except Exception as e:
            self.log.exception("Exception during Processing Message %s", str(e))
            return

        if "Event" in list(body.keys()):
            autoscaling_type = body['Event'].split(':')[-1]
            if len(autoscaling_type):
                EC2InstanceId = ""
                self.log.info("Received event, Not Alarm =%s" % autoscaling_type)
                if 'EC2InstanceId' in body: # will we ever have this field?
                    EC2InstanceId = body['EC2InstanceId']
                try:
                    this_event = event(
                            autoscaling_type,
                            body['AutoScalingGroupName'],
                            EC2InstanceId,
                            'sync'
                    )
                    return this_event
                except KeyError as e:
                    self.log.debug("Message has missing payload. %s" % (str(e)))
                    pass

        if "AlarmName" in list(body.keys()):
            self.log.debug("Received alarm: %s" % (body))
            comparison = body['Trigger']['ComparisonOperator']
            new_state_reason = body['NewStateReason'].split(':')[0]
            slow_server = None

            if 'LessThan' in comparison:
                self.log.info("Received a alarm for scale down. Processing...")
            else:
                self.log.info("Not a alarm for scale down. Ignored.")
                return

            if new_state_reason == "SLOW_SERVER" and body['AlarmName'] == "ADC_CLOUDADAPTER_ALARM":
                asg_name, slow_server = self.parse_slow_server_alarm(body)
                if not asg_name or not slow_server:
                    self.log.info("CloudAdapter alarm sent wrong data")
                    return
                self.log.info("slow_server received: ASG:%s server:%s" % (asg_name,slow_server))
            else:
                self.log.info("no slow server in alarm")
                dimensions = body['Trigger']['Dimensions']
                for dimension in dimensions:
                    try:
                        if dimension['name'] == 'AutoScalingGroupName' :
                            asg_name = dimension['value']
                    except KeyError:
                        '''
                        There might be multiple dimensions, so we have to keep trying.
                        If we fail to find a dimension for AutoScalingGroupName we
                        will fail to initialize asg_name which will be caught as exception
                        when we try to use it in the event() arguments.
                        '''
                        pass
                    continue

            self.log.info("Received an alarm for scale down event in asg: %s" % asg_name)

            try:
                this_event = event(
                        body['NewStateValue'],
                        asg_name,
                        slow_server,
                        'drain'
                )
                return this_event
            except (KeyError, NameError) as e:
                # Body key we expected was missing
                self.log.debug("Message has missing payload. %s" % (str(e)))
                return

        self.log.debug("Skipped message: %s" % (body))

    # Assumption SQS queue and SNS Topicare already created.
    def attach_queue_to_topic(self, queue_url, queue_arn, topic_arn):

        self.connect_instance_sqs()
        self.connect_instance_sns()
        policy = {
                'Version':'2012-10-17',
                'Id':'%s_queue_to_topic' % (self.get_own_instanceid()),
                'Statement':[{
                        'Sid':'rule1',
                        'Effect':'Allow',
                        'Principal':{
                                'AWS':'*'
                        },
                        'Action':'SQS:*',
                        'Resource':queue_arn,
                        'Condition':{
                                'ArnEquals':{
                                        'aws:SourceArn':topic_arn
                                }
                        }
                }]
        }
        subscription_arn = ""
        try:
            self.sqs_connection.set_queue_attributes(QueueUrl=queue_url, Attributes={'Policy': json.dumps(policy)})
            subscription_arn = self.sns_connection.subscribe(TopicArn=topic_arn, Protocol='sqs', Endpoint=queue_arn)['SubscriptionArn']
            self.log.info(" SQS Subscribed to SNS Subscription ID %s" % subscription_arn)
        except Exception as e:
            self.log.exception("Attach Topic to Queue - exception: %s." % (str(e)))
        return subscription_arn

    def _get_instance_details(self, instanceid):
        reservations = self.ec2_connection.describe_instances(InstanceIds=[instanceid])
        if reservations :
            instance = reservations['Reservations'][0]['Instances'][0]
            return instance

    def _get_autoscaling_groups(self, groups):
        # boto expects list argument here, hide this from the user.
        get_all_flag = False
        asg_list = []
        if groups is not None:
            if type(groups) is not list:
                groups = [ groups ]
            elif len(groups) > 0 and groups[0] is None:
                # Some functions simply send a [None] to this function
                get_all_flag = True
        else:
            get_all_flag = True
        try:
            if get_all_flag :
                asg_list = self.autoscale_connection.describe_auto_scaling_groups()['AutoScalingGroups']
            else:
                asg_list = self.autoscale_connection.describe_auto_scaling_groups(AutoScalingGroupNames=groups)['AutoScalingGroups']
        except boto.exception.BotoServerError as e:
            self.log.critical(str(e))
            raise aws_authetication_failure(str(e))
        else:
            return asg_list

    def get_own_instanceid(self):
        user_data_file = "/nsconfig/.AWS/user_data" ##HAPeerInstanceId
        instance_id_file = "/nsconfig/.AWS/instance-id"

        found_password = None

        if os.path.exists(user_data_file):
            with open(user_data_file, 'r') as f:
                for line in f:
                    if "HAPeerInstanceId" in line:
                        found_password = line.split("=")[1].strip()
                        break

        if found_password is None and os.path.exists(instance_id_file):
            with open(instance_id_file, 'r') as f:
                found_password = f.readline()

        return found_password

    def get_own_instance_name(self, instanceid, region):
        # When given an instance ID as str e.g. 'i-1234567', return the instance 'Name' from the name tag.
        ec2 = boto3.resource('ec2', region_name=region, endpoint_url=self.endpoint_url["ec2"], verify=False)
        ec2instance = ec2.Instance(instanceid)
        instancename = ''
        for tags in ec2instance.tags:
            if tags["Key"] == 'Name':
                instancename = tags["Value"]
        return instancename

    def get_account_id(self):
        #iam = boto3.resource('iam')
        #return(iam.CurrentUser().arn.split(':')[4])
        return boto3.client('sts').get_caller_identity()['Account']


    def topic_arn_from_name(self, region, name):
        return ":".join(["arn", "aws", "sns", region, self.get_account_id(), name])

    def queue_arn_from_name(self, region, name):
        return ":".join(["arn", "aws", "sqs", region, self.get_account_id(), name])

    def __get_token(self):
        headers = {'X-aws-ec2-metadata-token-ttl-seconds': '600',}
        req_ip = METADATA_URL + '/api/token'
        response = self.http.request('PUT', req_ip, headers=headers)
        return response.data.decode('utf-8')

    def __get_meta_data_v2(self,fields):
        url = METADATA_URL + '/meta-data' + fields
        token = self.__get_token()
        resp = self.http.request('GET', url, headers={"X-aws-ec2-metadata-token":token}, timeout=10)
        if resp.status != 200:
            self.log.error("Meta-data server 169.254.169.254 returned error code : %s", resp.status)
            return 0
        return resp.data.decode('utf-8')

    def get_own_privateip(self):
        privateip = self.__get_meta_data_v2('/local-ipv4')
        return privateip

    def get_own_region(self):
        region = self.__get_meta_data_v2('/placement/availability-zone')
        if region:
            return region[:-1]
        else:
            return None

    def get_instance_freeips(self, mac=None):
        macs = self.__get_meta_data_v2('/network/interfaces/macs')
        for macx in macs.split('\n'):
            if not macx:
                continue
            mac = macx[:-1]
            fields = '/network/interfaces/macs/' + mac + '/device-number'
            device_nu = self.__get_meta_data_v2(fields)
            if device_nu == '1':
                fields_ip = '/network/interfaces/macs/' + mac + '/local-ipv4s'
                freeips = self.__get_meta_data_v2(fields_ip)
                return freeips.split("\n")
        return []

    def get_instance_count(self):
        macs = self.__get_meta_data_v2('/network/interfaces/macs')
        return macs.count('/')

    def connect_instance_autoscale(self, config=None):
        try:
            region = self.get_own_region()
            self.autoscale_connection
        except AttributeError:
            try:
                if config == None:
                    self.autoscale_connection = boto3.client('autoscaling', region_name=region, endpoint_url=self.endpoint_url["autoscaling"], verify=False )
                else:
                    self.autoscale_connection = boto3.client('autoscaling', region_name=region,  config=config, endpoint_url=self.endpoint_url["autoscaling"], verify=False)
            except boto.exception.NoAuthHandlerFound as e:
                self.log.critical("Fail to connect autoscale: %s" % str(e))
                raise aws_authetication_failure(str(e))
            except Exception as e:
                self.log.exception("Fail to connect autoscale: %s" % str(e))
                raise e

    def connect_instance_ec2(self):
        try:
            region = self.get_own_region()
            self.ec2_connection
        except AttributeError:
            try:
                self.ec2_connection = boto3.client('ec2', region_name=region, endpoint_url=self.endpoint_url["ec2"], verify=False)
            except boto.exception.NoAuthHandlerFound as e:
                self.log.critical("Fail to connect ec2: %s" % str(e))
                raise aws_authetication_failure(str(e))
            except Exception as e:
                self.log.exception("Fail to connect ec2: %s" % str(e))
                raise e

    def connect_instance_sns(self):
        try:
            self.log.debug("Connect_instance_sns")
            region = self.get_own_region()
            self.sns_connection
        except AttributeError:
            try:
                self.sns_connection = boto3.client('sns',region_name=region, endpoint_url=self.endpoint_url["sns"], verify=False)
            except boto.exception.NoAuthHandlerFound as e:
                self.log.critical("Fail to connect sns: %s" % str(e))
                raise aws_authetication_failure(str(e))
            except Exception as e:
                self.log.exception("Fail to connect sns: %s" % str(e))
                raise e

    def connect_instance_sqs(self):
        try:
            region = self.get_own_region()
            self.sqs_connection
        except AttributeError:
            try:
                self.sqs_connection = boto3.client('sqs', region_name=region,endpoint_url=self.endpoint_url["sqs"], verify=False)
            except boto.exception.NoAuthHandlerFound as e:
                self.log.critical("Fail to connect sqs: %s" % str(e))
                raise aws_authetication_failure(str(e))
            except Exception as e:
                self.log.exception("Fail to connect sqs: %s" % str(e))
                raise e

    def connect_instance_cloudwatch(self):
        try:
            region = self.get_own_region()
            self.cloudwatch_connection
        except AttributeError:
            try:
                self.cloudwatch_connection = boto3.client('cloudwatch', region_name=region, endpoint_url=self.endpoint_url["cloudwatch"],verify=False)
            except boto.exception.NoAuthHandlerFound as e:
                self.log.critical("Fail to connect autoscale: %s" % str(e))
                raise aws_authetication_failure(str(e))
            except Exception as e:
                self.log.exception("Fail to connect autoscale: %s" % str(e))
                raise e
    def get_cloud_platform(self):
        return 'AWS'

    def validate_intf_count(self, intf_count):
        return (intf_count < 3)

    def check_event_queue(self):
        return True

    def get_ftu_filename(self):
        return '/flash/nsconfig/.AWS/ftumode'

    def get_daemon_pid_file(self):
        return '/flash/nsconfig/.AWS/rain_scale.pid'

    def get_rain_stats_daemon_pid_file(self):
        return '/flash/nsconfig/.AWS/rain_stats.pid'

    def clouadapter_iam_perrmissions(self):
        iam_privileges = [
            "iam:GetRole",
            "iam:PassRole",
            "iam:SimulatePrincipalPolicy",
            "autoscaling:*",
            "sns:*",
            "sqs:*",
            "cloudwatch:*",
            "ec2:*",
            "events:DescribeRule",
            "events:PutRule",
            "events:DeleteRule",
            "events:PutTargets",
            "events:RemoveTargets"
                 ]
        try:
           self.check_iam(iam_privileges)
           return 0
        except Exception as e:
           self.log.error("check_iam failed: %s" % str(e))
           return -1

    def check_privileges(self, feature):
        region = self.get_own_region()
        if region.startswith("us-isob-"):
            #IAM check not supported in SC2S
            return 0;
        if region.startswith("us-iso-"):
            #IAM check not supported in C2S
            return 0;
        iam_privileges = {
                "ha"    :
                        [
                                "ec2:DescribeInstances",
                                "ec2:AttachNetworkInterface",
                                "ec2:DetachNetworkInterface",
                                "ec2:StopInstances",
                                "ec2:StartInstances",
                                "ec2:RebootInstances"
                        ],
                "hainc" :
                        [
                                "ec2:DescribeInstances",
                                "ec2:DescribeAddresses",
                                "ec2:AssociateAddress",
                                "ec2:DisassociateAddress"
                        ],
                "appautoscale" :
                        [
                                "sns:CreateTopic",
                                "sns:DeleteTopic",
                                "sns:ListTopics",
                                "sns:Subscribe",
                                "sqs:CreateQueue",
                                "sqs:ListQueues",
                                "sqs:DeleteMessage",
                                "sqs:GetQueueAttributes",
                                "sqs:SetQueueAttributes"
                        ]
        }

        if feature in iam_privileges:
            return self.check_iam(iam_privileges[feature], do_print = False)
        else :
            return -1

    def check_iam(self, Actions, do_print=True):
        self.missing_iam = False
        client = boto3.client("iam", region_name=self.get_own_region())
        resp = self.__get_meta_data_v2('/iam/security-credentials')
        if resp == 0:
            if do_print:
                err_str = " Meta-data server 169.254.169.254 returned error code "
                print(json.dumps({'response':
                            NS_BASE_ERRORS["NSERR_CLOUD_IAM"]["desc"],
                            'errno':
                            NS_BASE_ERRORS['NSERR_CLOUD_IAM']['errno'],
                            'errmsg':err_str}))
            self.missing_iam = True
            return 0
        roleName = client.get_role(RoleName=resp)["Role"]["Arn"]
        result = client.simulate_principal_policy(PolicySourceArn=roleName,
                                                ActionNames=Actions)
        if result["ResponseMetadata"]["HTTPStatusCode"] != 200:
            self.log.error("Call to simulate_principal_policy failed ")
            if do_print:
                print(json.dumps({'response':
                    NS_BASE_ERRORS["NSERR_CLOUD_IAM"]["desc"],
                    'errno':
                    NS_BASE_ERRORS['NSERR_CLOUD_IAM']['errno'],
                    'errmsg':"Call to simulate_principal_policy failed"}))
            self.missing_iam = True
            return 0
        iam_missing = ""
        for i in result["EvaluationResults"]:
            if i["EvalDecision"] != "allowed":
                iam_missing += ( i['EvalActionName'] + ", ")
                self.log.error("IAM permission not configured : %s" %i['EvalActionName'])
        if iam_missing != "" :
            self.log.info("Missing IAMs: " + iam_missing)
            err_str = "Missing IAMs: " + iam_missing
            if do_print:
                print(json.dumps({'response':
                    NS_BASE_ERRORS["NSERR_CLOUD_IAM"]["desc"],
                    'errno':
                    NS_BASE_ERRORS['NSERR_CLOUD_IAM']['errno'],
                    'errmsg':err_str}))
            self.missing_iam = True
            return 0
        self.log.info("NO Missing IAMs")
        if do_print:
            print(json.dumps({'response':'done', 'errno':0, 'errmsg':""}))
        return 0

    def get_aws_reporting(self):
        return aws_reporting(self)

class aws_cloudAdapter():
    def __init__(self, aws_config):
        self.log = aws_config.log
        self.aws_config = aws_config
        self.region = self.aws_config.get_own_region()
        self.boto3resource_sqs = boto3.resource('sqs', region_name=self.region)
        self.boto3client_sqs = boto3.client('sqs', region_name=self.region)
        self.boto3client_sns = boto3.client('sns', region_name=self.region)
        self.boto3client_events = boto3.client('events', region_name=self.region)
        self.boto3clientASG = boto3.client('autoscaling', region_name=self.region)
        self.boto3clientCW = boto3.client('cloudwatch', region_name=self.region)
        self.boto3client_ec2 = boto3.client('ec2', region_name=self.region)

    def get_own_az(self, instanceid):
        try:
            response = self.boto3client_ec2.describe_instances(InstanceIds=[instanceid])
            res = response['Reservations'][0]
            instance = res['Instances'][0]
            az = instance['Placement']['AvailabilityZone']
            return az
        except Exception as e:
            self.log.info("Exception in get_own_az %s" % str(e))
            return None

    def find_instance(self, feild):
        try:
            response = self.boto3client_ec2.describe_instances(Filters=[{'Name':feild,'Values':values}])
            instance = response['Reservations'][0]['Instances'][0]
            return instance
        except Exception as e:
            self.log.info("Exception in find_instance %s" % str(e))
            return None

    def get_instance_details(self, instanceid, feild):
        try:
            response = self.boto3client_ec2.describe_instances(InstanceIds=[instanceid])
            instance = response['Reservations'][0]['Instances'][0]
            return instance[feild]
        except Exception as e:
            self.log.info("Exception in get_instance_details %s" % str(e))
            return None

    def put_tag_on_instances(self,instances, tagKey, tagValue):
        try:
            response = self.boto3client_ec2.create_tags(Resources=instances,
                            Tags=[{'Key': tagKey, 'Value':tagValue },])
            return response
        except Exception as e:
            self.log.info("put_tag_on_instance: %s"%str(e))
            return None

    def get_tag_by_instance(self, instanceId, tagKey):
        try:
            instanceFilter = {
                    'Name' : 'resource-id',
                    'Values' : [instanceId]
            }
            tagFilter = {
                    'Name' : 'key',
                    'Values' : [tagKey]
            }
            tag = None
            response = self.boto3client_ec2.describe_tags(Filters=[instanceFilter, tagFilter])
            if len(response['Tags']) != 0:
                tag = response['Tags']
            return tag
        except Exception as e:
            self.log.info("get_tag_by_instance %s" % str(e))
            return None

    def get_instances_by_tag(self, tagKey, tagValue):
        try:
            tagFilter = {
                    'Name' : 'tag:' + tagKey,
                    'Values' : [tagValue]
            }
            instances = None
            response = self.boto3client_ec2.describe_instances(Filters=[tagFilter])
            self.log.info(response)
            if len(response['Reservations']) != 0:
                instances = response['Reservations'][0]['Instances']
                instance = instances[0]['InstanceId']
            self.log.info(instance)
            return instance
        except Exception as e:
            self.log.info("get_instances_by_tag %s" % str(e))
            return None

    def create_sqs_queue(self, name):
        try:
            queue_url = self.boto3resource_sqs.create_queue(
                                            QueueName=name)
            self.log.debug("url_queue : %s" % str(queue_url))
            return queue_url
        except Exception as e:
            self.log.info("Exception received in create_sqs_queue %s" % str(e))
            return None

    def get_sqs_queue(self, name):
        try:
            queue_url = self.boto3resource_sqs.get_queue_by_name(QueueName=name)
            return queue_url.url
        except Exception as e:
            self.log.info("failed to fetch queue %s" % str(e))
            return None

    def set_sqs_queue_attributes(self, queue_url, sns_arn_list, sqs_arn):
        policy = {
                        'Version':'2012-10-17',
                        'Id':'%s_queue_to_topic' % (self.aws_config.get_own_instanceid()),
                        'Statement':[{
                                'Sid':'rule1',
                                'Effect':'Allow',
                                'Principal':{
                                        'AWS':'*'
                                },
                                'Action':'SQS:*',
                                'Resource':sqs_arn,
                                'Condition':{
                                        'ArnEquals':{
                                                'aws:SourceArn':sns_arn_list
                                        }
                                }
                        }]
                }
        try:
            response = self.boto3client_sqs.set_queue_attributes(
                                    QueueUrl=queue_url,
                                    Attributes={'Policy': json.dumps(policy)}
                        )
            self.log.debug("queue attributes set : "+str(response))
            return True #response is None
        except Exception as e:
            self.log.info("Unknown Exception received in set_sqs_queue_attributes %s" % str(e))
            return False


    def get_sqs_queue_attributes(self, queue_url, attribute):
        try:
            response = self.boto3client_sqs.get_queue_attributes(
                            QueueUrl=queue_url,
                            AttributeNames=[attribute]
                        )
            self.log.debug("url_queue attributes %s " % response)
            if response and 'Attributes' in response:
                return response['Attributes'][attribute]
            return None
        except Exception as e:
            self.log.info("Exception received in get_sqs_queue_attributes %s" % str(e))
            return None

    def poll_sqs_queue(self, url_queue, max_msg, wait_time):
        try:
            response = self.boto3client_sqs.receive_message(
                            QueueUrl=url_queue,
                            MaxNumberOfMessages=max_msg,
                            WaitTimeSeconds=wait_time,
                            MessageAttributeNames=['All'],
                       )
            self.log.debug("received message on SQS :" + str(response))
            return response
        except Exception as e:
            self.log.info("Exception received in poll_sqs_queue %s" % str(e))
            return None

    def dequeue_sqs_queue(self, queue_url, receiptHandle):
        try:
            response = self.boto3client_sqs.delete_message(
                        QueueUrl=queue_url,
                        ReceiptHandle=receiptHandle
                       )
            self.log.debug("message de-queued : "+str(response))
            return True #response is None
        except Exception as e:
            self.log.info("Exception received in dequeue_sqs_queue %s" % str(e))
            return False

    def create_sns_topic(self, name):
        try:
            response = self.boto3client_sns.create_topic(Name=name)
            sns_arn = response['TopicArn']
            self.log.debug("sns_arn : %s" % sns_arn)
            if not self.set_sns_topic_attributes(sns_arn):
                return None
        except Exception as e:
            self.log.info("caught exception in create_sns_topic %s" % str(e))
            return None
        return response

    def get_sns_topic_attributes(self, sns_arn):
        try:
            attrs = self.boto3client_sns.get_topic_attributes(TopicArn=sns_arn)
            self.log.debug(attrs)
            return attrs
        except Exception as e:
            self.log.info("exception in get_sns_topic_attributes %s" % str(e))
            return None

    def set_sns_topic_attributes(self, sns_arn):
        account_id = self.aws_config.get_account_id()
        self.log.debug("account id is : "+str(account_id))
        policy = {
            "Version": "2012-10-17",
            "Id": "__default_policy_ID_CloudAdapter",
            "Statement": [
                {
                    "Sid": "__default_statement_ID_CloudAdapter",
                    "Effect": "Allow",
                    "Principal": {
                       "AWS": "*"
                     },
                    "Action": [
                        "SNS:GetTopicAttributes",
                         "SNS:SetTopicAttributes",
                        "SNS:AddPermission",
                          "SNS:RemovePermission",
                        "SNS:DeleteTopic",
                       "SNS:Subscribe",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:Publish",
                         "SNS:Receive"
                      ],
                    "Resource": sns_arn,
                    "Condition": {
                    "StringEquals": {
                            "AWS:SourceOwner": account_id
                        }
                    }
                },
                {
                    "Sid": "CloudAdapter_Events_to_Topic",
                    "Effect": "Allow",
                    "Principal": {
                        "Service": "events.amazonaws.com"
                    },
                    "Action": "sns:Publish",
                    "Resource": sns_arn
                }
                ]
        }
        try:
            response = self.boto3client_sns.set_topic_attributes(
                                TopicArn=sns_arn,
                                AttributeName='Policy',
                                AttributeValue=json.dumps(policy)
                      )
            self.log.debug("SNS attributes set : "+str(response))
            return True #response is None here
        except Exception as e:
            self.log.info("caught exception in set_sns_topic_attributes %s" % str(e))
            return False

    def unsubscribe_sqs_to_sns_topic(self, subscription_arn):
        try:
            response = self.boto3client_sns.unsubscribe(
                           SubscriptionArn=subscription_arn
                        )
            self.log.debug("subscription deleted")
            return True
        except Exception as e:
            self.log.info("unsubscribe_sqs_to_sns_topic %s" % str(e))
            return False

    def subscribe_sqs_to_sns_topic(self, sns_arn, sqs_arn):
        try:
            subscription = self.boto3client_sns.subscribe(
                            TopicArn=sns_arn,
                            Protocol='sqs',
                            Endpoint=sqs_arn
                        )
            self.log.debug("subscription : %s" % str(subscription))
            if 'SubscriptionArn' in subscription:
                return subscription['SubscriptionArn']
            return None
        except Exception as e:
            self.log.info("caught exception in subscribe_sqs_to_sns_topic %s" % str(e))
            return None

    def fetch_subscription_attributes(self, subs_arn):
        try:
            response = self.boto3client_sns.get_subscription_attributes(SubscriptionArn=subs_arn)
            return response
        except Exception as e:
            self.log.info("fetch_subscription_attributes %s" % str(e))
            return None

    def get_subscriptions(self, sns_arn):
        try:
            response = self.boto3client_sns.list_subscriptions_by_topic(
                                            TopicArn=sns_arn)
            self.log.info(response)
            return response
        except Exception as e:
            self.log.info("get_subscritions %s" % str(e))
            return None

    def create_rule(self, rule_name, event_pattern, state):
        try:
            response = self.boto3client_events.put_rule(
                        Name=rule_name,
                        EventPattern=event_pattern,
                        State=state
                        )
            self.log.debug("rule created : %s" % str(response))
            return response
        except Exception as e:
            self.log.info("caught exception in create_rule %s" % str(e))
            return None

    def get_rule(self, name):
        try:
            rule = self.boto3client_events.describe_rule(Name=name)
            return rule
        except Exception as e:
            self.log.info("caught exception in get_rule %s" % str(e))
            return None

    #appends to existing list
    def assign_target_to_rule(self, rule_name, sns_arn):
        try:
            response = self.boto3client_events.put_targets(
                                Rule=rule_name,
                                Targets=[
                                    {
                                        'Arn': sns_arn,
                                        'Id': 'citrixADC_rule_id123'
                                    }
                                ]
                        )
            self.log.debug("SNS target to rule configured: "+str(response))
            return response
        except Exception as e:
            self.log.info("caught exception in assign_target_to_rule %s" % str(e))
            return None

    def delete_sns(self, sns_arn):
        try:
            response = self.boto3client_sns.delete_topic(TopicArn=sns_arn)
            self.log.debug("sns topic deleted : %s" % str(response))
            return True
        except Exception as e:
            self.log.info("caught exception in delete_sns %s" % str(e))
            return False

    def delete_sqs(self, url_queue):
        try:
            response = self.boto3client_sqs.delete_queue(QueueUrl=url_queue)
            self.log.debug("sqs queue deleted %s" % str(response))
            return True #response will be None
        except Exception as e:
            self.log.info("caught exception in delete_sqs %s" % str(e))
            return False

    def delete_targets_from_rule(self, rule_name):
        try:
            response = self.boto3client_events.remove_targets(
                                Rule=rule_name,
                                Ids=['citrixADC_rule_id123']
                            )
            self.log.debug("targets removed :"+str(response))
            return True
        except Exception as e:
            self.log.info("caught exception in delete_targets_from_rule %s" % str(e))
            return False

    def delete_aws_rule(self, rule_name):
        try:
            response = self.boto3client_events.delete_rule(
                Name=rule_name
            )
            self.log.debug("rule deleted %s" % str(response))
            return True
        except Exception as e:
            self.log.info("delete_aws_rule %s" % str(e))
            return False

    def describe_asg_policies(self, asg_name):
        try:
            response = self.boto3clientASG.describe_policies(AutoScalingGroupName=asg_name)
            self.log.debug("describe_asg_policies: %s" % str(response))
            return response
        except Exception as e:
            self.log.info("caught expection in describe_asg_policies %s" % str(e))
            return None

    def describe_asg(self, asg_name):
        try:
            response = self.boto3clientASG.describe_auto_scaling_groups(
                                    AutoScalingGroupNames=[asg_name,],)
            self.log.debug("describe asg : %s" % str(response))
            return response
        except Exception as e:
            self.log.info("exception occured in describe_asg %s" % str(e))
            return None

    def update_asg_desired_capacity(self, asg_name, desired_capacity):
        try:
            response = self.boto3clientASG.update_auto_scaling_group(
                        AutoScalingGroupName=asg_name,
                        DesiredCapacity=desired_capacity
                    )
            return True
        except Exception as e:
            self.log.info("exception in update_asg_desired_capacity %s" % str(e))
            return False

    def update_asg_max_capacity(self, asg_name, max_capacity):
        try:
            response = self.boto3clientASG.update_auto_scaling_group(
                        AutoScalingGroupName=asg_name,
                        MaxSize=max_capacity
                    )
            return True
        except Exception as e:
            self.log.info("exception in update_asg_max_capacity %s" % str(e))
            return False

    def create_metric_alarm(self, name, sns_arn_list, asg_name):
        try:
            response = self.boto3clientCW.put_metric_alarm(
                AlarmName=name,
                ComparisonOperator='LessThanThreshold',
                Statistic='SampleCount',
                EvaluationPeriods=1,
                MetricName='NetworkIn',
                Namespace='AWS/EC2',
                Period=60,
                Dimensions= [
                {
                   'Name': 'AutoScalingGroupName',
                   'Value': asg_name
                 },
                ],
                Threshold=0.0000001,
                ActionsEnabled=True,
                AlarmActions=sns_arn_list
            )
            self.log.debug("alarm created")#response retunred in None
            return True
        except Exception as e:
            self.log.info("create_metric_alarm failed %s" % str(e))
            return False

    def get_alarms(self, actionPrefix):
        try:
            if actionPrefix:
                response = self.boto3clientCW.describe_alarms(
                        ActionPrefix=actionPrefix
                       )
            else:
                response = self.boto3clientCW.describe_alarms()
            self.log.debug("got alarms: "+str(response))
            return response
        except Exception as e:
            self.log.info("Exception in get_alarms %s" % str(e))
            return None

    def enable_alarm(self, name):
        try:
            response = self.boto3clientCW.enable_alarm_actions(AlarmNames=[name])
            self.log.debug("alarm enabled : "+name)
            return True
        except Exception as e:
            self.log.info("exception occurred while enabling alarm %s" % str(e))
            return False

    def disable_alarm(self, name):
        try:
            response = self.boto3clientCW.disable_alarm_actions(AlarmNames=[name])
            self.log.info("alarm disabled : "+name)
            return True
        except Exception as e:
            self.log.info("Exception in disable_alarm %s"  % str(e))
            return False

    def update_alarm_state(self, name, state, reason, data):
        try:
            response = self.boto3clientCW.set_alarm_state(
                    AlarmName=name,
                    StateValue=state,
                    StateReason=reason,
                    StateReasonData=data
                  )
            self.log.debug("update_alarm_state : %s" % str(response))
            return True
        except Exception as e:
            self.log.info("update_alarm_state %s" % str(e))
            return False

    def delete_aws_alarms(self,names):
        try:
            response = self.boto3clientCW.delete_alarms(AlarmNames=names)
            return True
        except Exception as e:
            self.log.info("delete_aws_alarms %s" % str(e))
            return False

    def change_scaling_policy(self, scaling_adjustment, asg_name, policy_name, status, adjust_type):
        try:
            response = self.boto3clientASG.put_scaling_policy(
                            ScalingAdjustment=scaling_adjustment,
                            AutoScalingGroupName=asg_name,
                            PolicyName=policy_name,
                            Enabled=status,
                            AdjustmentType=adjust_type
                    )
            self.log.debug("changed scalig policy : "+str(response))
            return response
        except Exception as e:
            self.log.info("Exception in change_scaling_policy %s" % str(e))
            return None

    def get_status_of_scaling_activities(self, asg_name):
        try:
            response = self.boto3clientASG.describe_scaling_activities(
                            AutoScalingGroupName=asg_name,MaxRecords=1)
            self.log.debug("got scaling activities : "+str(response))
            return response
        except Exception as e:
            self.log.info("Exception in get_status_of_scaling_activities %s" % str(e))
            return None

    def get_template_data(self, instanceId):
        try:
            response = self.boto3client_ec2.get_launch_template_data(
                    InstanceId=instanceId)
            self.log.debug(response)
            LaunchTemplateData = response["LaunchTemplateData"]
            return LaunchTemplateData
        except Exception as e:
            self.log.info("get_template_data %s" % str(e))
            return None

    def get_templates(self, names):
        try:
            response = self.boto3client_ec2.describe_launch_templates(
                                LaunchTemplateNames=names
                       )
            return response
        except Exception as e:
            self.log.info("get_templates %s" % str(e))
            return None

    def create_launch_temp(self, name, data):
        try:
            response = self.boto3client_ec2.create_launch_template(
                    LaunchTemplateName=name, LaunchTemplateData=data)
            LaunchTemplateId = response["LaunchTemplate"]["LaunchTemplateId"]
            return LaunchTemplateId
        except Exception as e:
            self.log.info("create_launch_temp %s" % str(e))
            return None

    def get_launch_template_versions(self, name):
        try:
            response = self.boto3client_ec2.describe_launch_template_versions(
                    LaunchTemplateName=name, Versions=["$Default", "$Latest"])
            self.log.debug(response)
            return response
        except Exception as e:
            self.log.info("get_launch_template_versions %s" % str(e))
            return None

    def create_instance(self, name):
        try:
            LaunchTemplateInfo = {
            'LaunchTemplateName': name,
            'Version': '$Default'
            }
            self.log.debug("Create Instance from $Latest Version of Template")
            response = self.boto3client_ec2.run_instances(
                             MaxCount=1, MinCount=1, LaunchTemplate=LaunchTemplateInfo)
            self.log.debug(response)
            InstanceId = response["Instances"][0]["InstanceId"]
            self.log.debug("NewInstance:")
            self.log.debug(InstanceId)
            return InstanceId
        except ClientError as e:
            self.log.info(e.response)
            if e.response['Error']['Code'] == 'InvalidIPAddress.InUse':
                return "InvalidIPAddressInUse"
            return None

    def attach_instance_to_asg(self,InstanceId,AutoScalingGroupName):
        try:
            response = self.boto3clientASG.attach_instances(
                          InstanceIds=[InstanceId], AutoScalingGroupName=AutoScalingGroupName)
            return True
        except Exception as e:
            self.log.info("attach_instance_to_asg %s" % str(e))
            return False

    def detach_instance_from_asg(self, InstanceId, AutoScalingGroupName):
        try:
            response = self.boto3clientASG.terminate_instance_in_auto_scaling_group(
                                                InstanceId=InstanceId,
                                                ShouldDecrementDesiredCapacity=True
                       )
            return True
        except Exception as e:
            self.log.info("detach_instance_from_asg %s" % str(e))
            return False

    def terminate_ec2_instances(self, InstanceIds):
        try:
            response = self.boto3client_ec2.terminate_instances(
                                             InstanceIds=InstanceIds
                       )
            return True
        except Exception as e:
            self.log.info("terminate_ec2_instances %s" % str(e))
            return False

    def delete_template(self, name):
        try:
            self.log.debug("Deleting the Launch Template")
            response = self.boto3client_ec2.delete_launch_template(
                               LaunchTemplateName=name)
            self.log.debug(response)
            return True
        except Exception as e:
            self.log.info("delete_template %s" % str(e))
            return False

    def get_instance_status(self, InstanceId):
        try:
            response = self.boto3client_ec2.describe_instance_status(InstanceIds=[InstanceId])
            return response
        except Exception as e:
            self.log.info("get_instance_status %s" % str(e))
            return None

class aws_reporting():

    UNIT = 0
    LIMIT = 1

    #format is statname:[unit, limit], ...
    sys_statinfo = {
            "cpuusagepcnt":["Percent", 80],
            "mgmtcpuusagepcnt":["Percent", 80],
            "memsizemb":["Megabytes", None],
            "memuseinmb":["Megabytes", None],
            "memusagepcnt":["Percent", 90]}

    server_statinfo = {
            "avgsvrttfb":["Milliseconds", 100],
            "surgecount":["Count", 30],
            "cursrvrconnections":["Count", 30],
            "svrestablishedconn":["Count", 30]}

    def __init__(self, cloud_config):
        name = self.__class__.__name__
        self.cloud_config = cloud_config
        self.logger = log
        self.instanceid = cloud_config.get_own_instanceid()
        self.ipaddr = cloud_config.get_own_privateip()
        self.region = cloud_config.get_own_region()
        self.instanceName = cloud_config.get_own_instance_name(self.instanceid, self.region)
        self.cw_client = self.connect_instance_cloudwatch()
        pass

    def connect_instance_cloudwatch(self):
        return boto3.client('cloudwatch', region_name=self.region, endpoint_url=self.cloud_config.endpoint_url["cloudwatch"], verify=False)

    def send_cloudwatch_thresholds(self):
        pass

    def formulate_metric(self, metric, stat_value, stats):
        resource = metric["resource"]
        property = metric["property"]
        value = float(stat_value[resource][property])
        resolution = stats.get_stat_resolution(metric["resolution"])
        data = {
                'MetricName': metric["name"],
                'Dimensions': metric["dimensions"],
                'Timestamp': datetime.datetime.now(),
                'StatisticValues': {
                        'SampleCount': 1,
                        'Sum': value,
                        'Minimum': value,
                        'Maximum': value
                        },
                'Unit': metric["unit"],
                'StorageResolution': resolution
                }
        self.logger.debug("New metric sample for %s (count=%s, sum=%s, ts=%s)" % (
                data.get('MetricName'), data.get('StatisticValues').get('SampleCount'), data.get('StatisticValues').get('Sum'),
                data.get('Timestamp')))
        return data


    def publish_metrics(self, metric_data, name_space):
        # publich metrics to cloudwatch
        self.cw_client.put_metric_data(
                Namespace = name_space + " <" + self.instanceid + ">",
                MetricData = metric_data
        )

    def create_cloudwatch_alarms_for_topic(self, target_statsinfo, topic_arn):
        namespace = "NS Cloudwatch"

        for stat in list(target_statsinfo.keys()):
            if target_statsinfo[stat][self.LIMIT]:
                #AUTOSCALE POLICY
                alarm1 = cloudwatch.alarm.MetricAlarm(
                        name="%s%s" % (stat, "1"),
                        metric="%s" % (stat),
                        namespace=namespace,
                        statistic="Average",
                        comparison=">=",
                        threshold=target_statsinfo[stat][self.LIMIT],
                        period=60,
                        evaluation_periods=1,
                        unit=target_statsinfo[stat][self.UNIT],
                        dimensions={"autoScalingGroupName":"OWA"},
                        alarm_actions=['arn:aws:autoscaling:us-east-1:074724716397:scalingPolicy:08dccf55-a87a-46e3-9f58-8416b3c247b3:autoScalingGroupName/OWA:policyName/Increase Group Size'])

                alarm2 = cloudwatch.alarm.MetricAlarm(
                        name="%s%s" % (stat, "2"),
                        metric="%s" % (stat),
                        namespace=namespace,
                        statistic="Average",
                        comparison=">=",
                        threshold=target_statsinfo[stat][self.LIMIT],
                        period=60,
                        evaluation_periods=1,
                        unit=target_statsinfo[stat][self.UNIT],
                        dimensions={"autoScalingGroupName":"SkypeForBusiness"},
                        alarm_actions=['arn:aws:autoscaling:us-east-1:074724716397:scalingPolicy:f5b36d8f-cd42-4d76-9148-390701db175f:autoScalingGroupName/SkypeForBusiness:policyName/Increase Group Size'])

                alarm3 = cloudwatch.alarm.MetricAlarm(
                        name="%s%s" % (stat, "3"),
                        metric="%s" % (stat),
                        namespace=namespace,
                        statistic="Average",
                        comparison=">=",
                        threshold=target_statsinfo[stat][self.LIMIT],
                        period=60,
                        evaluation_periods=1,
                        unit=target_statsinfo[stat][self.UNIT],
                        dimensions={"autoScalingGroupName":"SharePoint"},
                        alarm_actions=['arn:aws:autoscaling:us-east-1:074724716397:scalingPolicy:51880090-7205-4040-b393-afd78ee32e51:autoScalingGroupName/SharePoint:policyName/Increase Group Size'])

                self.cw_client.update_alarm(alarm1)
                time.sleep(1)
                self.cw_client.update_alarm(alarm2)
                time.sleep(1)
                self.cw_client.update_alarm(alarm3)
    def  configure_polling_for_groups(self, group_names):
        raise NotImplementedError('%s: configure_polling_for_groups' % (self.__class__.__name__))
