# Copyright 2025 Citrix Systems, Inc. All rights reserved.
# This software and documentation contain valuable trade
# secrets and proprietary property belonging to Citrix Systems, Inc.
# None of this software and documentation may be copied,
# duplicated or disclosed without the express
# written permission of Citrix Systems, Inc

import argparse
import json
import jwt
import sys
import logging
import os
import time
from cryptography import x509
from cryptography.hazmat.backends import default_backend

def setup_logging():
    log_file = '/var/tmp/extension_decoder.log'
    logger = logging.getLogger(__name__)
    logger.setLevel(logging.INFO)
    if not logger.handlers:
        formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(message)s')
        fh = logging.FileHandler(log_file)
        fh.setFormatter(formatter)
        sh = logging.StreamHandler(sys.stderr)
        sh.setFormatter(formatter)
        logger.addHandler(fh)
        logger.addHandler(sh)
    return logger

LOGGER = setup_logging()

def load_cert_with_target_cn(logger=LOGGER):
    """
    Load a PEM bundle (multiple certs) and return public key of first cert
    whose Common Name matches one of target_cns.
    """
    pem_path = "/nsconfig/license/las_cert.pem"
    target_cns = ["lasstaging.cloud.com", "las.cloud.com"]

    if not os.path.isfile(pem_path):
        logger.warning("Certificate bundle not found: %s", pem_path)
        return None

    try:
        with open(pem_path, "r") as f:
            content = f.read()
    except Exception as e:
        logger.error("Error reading %s: %s", pem_path, e)
        return None

    # Split into individual cert blocks
    raw_parts = content.split("-----BEGIN CERTIFICATE-----")
    cert_blocks = []
    for part in raw_parts:
        part = part.strip()
        if not part:
            continue
        if "-----END CERTIFICATE-----" in part:
            pem = "-----BEGIN CERTIFICATE-----\n" + part.split("-----END CERTIFICATE-----")[0].strip() + "\n-----END CERTIFICATE-----\n"
            cert_blocks.append(pem)

    for idx, pem in enumerate(cert_blocks):
        try:
            cert = x509.load_pem_x509_certificate(pem.encode(), default_backend())
            cn = None
            for attribute in cert.subject:
                if attribute.oid == x509.NameOID.COMMON_NAME:
                    cn = attribute.value
                    break
            logger.info("Bundle cert %d: CN=%s", idx, cn)
            if cn and cn in target_cns:
                logger.info("Matched target CN: %s", cn)
                logger.info("Extracting public key from certificate")
                public_key = cert.public_key()
                return public_key
        except Exception as e:
            logger.warning("Skipping invalid cert index %d: %s", idx, e)

    logger.info("No certificate with target CN found in bundle")
    return None

def find_matching_cert(certchain, logger=LOGGER):
    """Find certificate with CN matching lasstaging.cloud.com or las.cloud.com"""
    target_cns = ["lasstaging.cloud.com", "las.cloud.com"]

    logger.info("Searching for certificates with CN in: %s", target_cns)

    for i, cert_pem in enumerate(certchain["current"]):
        try:
            # Clean up the certificate PEM
            cert_pem_clean = cert_pem.replace('\\n', '\n')

            # Load certificate
            cert = x509.load_pem_x509_certificate(cert_pem_clean.encode(), default_backend())

            # Extract common name from subject
            for attribute in cert.subject:
                if attribute.oid == x509.NameOID.COMMON_NAME:
                    cn = attribute.value
                    logger.info("Certificate %d: CN = %s", i, cn)
                    if cn in target_cns:
                        logger.info("Found matching certificate with CN: %s", cn)
                        logger.info("Extracting public key from certificate")
                        public_key = cert.public_key()
                        return public_key
        except Exception as e:
            logger.error("Error processing certificate %d: %s", i, e)

    logger.warning("No matching certificate found")
    return None

def decode_jwt_with_cert(jwt_token, public_key, logger=LOGGER):
    """Decode JWT token using certificate's public key"""
    try:
        logger.info("Decoding JWT token")
        decoded_payload = jwt.decode(jwt_token, public_key, algorithms=["RS256"])
        logger.info("JWT token decoded successfully")
        return decoded_payload
    except jwt.InvalidTokenError as e:
        logger.error("JWT decode error: %s", e)
        return None
    except Exception as e:
        logger.error("Error decoding JWT: %s", e)
        return None

def is_valid_json_file(file_path, logger=LOGGER):
    """Check if file contains valid JSON"""
    try:
        with open(file_path, 'r') as f:
            json.load(f)
        return True
    except (ValueError, UnicodeDecodeError, IOError):
        logger.debug("File %s is not a valid JSON file", file_path)
        return False

def decode_activation_blob(logger=LOGGER):
    """
    Decode JWT from activation blob file and return the payload

    Returns:
        dict: Decoded JWT payload if successful, None if failed
    """
    activation_blob_path = '/nsconfig/license/offline_activation_blob'
    logger.info("Starting JWT activation blob decoding from: %s", activation_blob_path)

    try:
        # Load the JSON blob
        logger.info("Loading offline_activation_blob file: %s", activation_blob_path)
        with open(activation_blob_path, 'r') as f:
            jwt_token = f.read().strip()

        logger.info("offline_activation_blob file loaded successfully")

        # Find the matching certificate
        pub_key = load_cert_with_target_cn()

        if pub_key:
            decoded_payload = decode_jwt_with_cert(jwt_token, pub_key, logger)

            if decoded_payload:
                logger.info("offline_activation_blob decoded successfully")
                return decoded_payload
            else:
                logger.error("Failed to decode JWT token")
                return None
        else:
            logger.error("No matching certificate found with CN 'lasstaging.cloud.com' or 'las.cloud.com'")
            return None

    except IOError:
        logger.error("File not found: %s", activation_blob_path)
        return None
    except Exception as e:
        logger.error("Unexpected error processing %s: %s", activation_blob_path, e)
        return None

def decode_extension_blob(file_path, logger=LOGGER):
    """
    Decode JWT from extension blob file and return the payload
    
    Args:
        file_path: Path to the extension file

    Returns:
        dict: Decoded JWT payload if successful, None if failed
    """
    logger.info("Starting JWT extension blob decoding from: %s", file_path)

    try:
        # Check if file is valid JSON first
        if not is_valid_json_file(file_path, logger):
            logger.info("Skipping non-JSON file: %s", file_path)
            return None

        # Load the JSON blob
        logger.info("Loading JSON file: %s", file_path)
        with open(file_path, 'r') as f:
            data = json.load(f)

        logger.info("JSON file loaded successfully")

        # Verify it has the expected structure
        if not isinstance(data, dict) or 'extension' not in data or 'certchain' not in data:
            logger.info("File %s doesn't have expected extension structure", file_path)
            return None

        # Find the matching certificate
        pub_key = find_matching_cert(data["certchain"])

        if pub_key:
            # Decode the JWT token
            jwt_token = data["extension"]
            logger.info("JWT token length: %d", len(jwt_token))

            decoded_payload = decode_jwt_with_cert(jwt_token, pub_key, logger)

            if decoded_payload:
                logger.info("Extension blob decoded successfully")
                decoded_payload["source_file"] = os.path.basename(file_path)
                return decoded_payload
            else:
                logger.error("Failed to decode JWT token")
                return None
        else:
            logger.error("No matching certificate found with CN 'lasstaging.cloud.com' or 'las.cloud.com'")
            return None

    except IOError:
        logger.error("File not found: %s", file_path)
        return None
    except ValueError as e:
        logger.error("Invalid JSON in file %s: %s", file_path, e)
        return None
    except Exception as e:
        logger.error("Unexpected error processing %s: %s", file_path, e)
        return None

def decode_all_extensions(extension_dir='/nsconfig/license/extension/', logger=LOGGER):
    """
    Decode all extension files in the specified directory (any file type)
    
    Args:
        extension_dir: Directory containing extension files

    Returns:
        list: Array of decoded extension payloads
    """
    logger.info("Starting bulk extension decoding from directory: %s", extension_dir)

    extensions = []

    # Check if directory exists
    if not os.path.exists(extension_dir):
        logger.error("Extension directory does not exist: %s", extension_dir)
        return extensions

    if not os.path.isdir(extension_dir):
        logger.error("Path is not a directory: %s", extension_dir)
        return extensions

    # Find all files in the directory (excluding hidden files and directories)
    try:
        all_files = []
        for item in os.listdir(extension_dir):
            item_path = os.path.join(extension_dir, item)
            # Skip directories
            if os.path.isfile(item_path):
                all_files.append(item_path)

        logger.info("Found %d files in %s", len(all_files), extension_dir)

        if not all_files:
            logger.warning("No files found in %s", extension_dir)
            return extensions

        # Process each file
        valid_extensions = 0
        for file_path in all_files:
            logger.info("Processing file: %s", file_path)

            decoded_payload = decode_extension_blob(file_path)

            if decoded_payload:
                extensions.append(decoded_payload)
                valid_extensions += 1
                logger.info("Successfully decoded extension from: %s", os.path.basename(file_path))
            else:
                logger.debug("Failed to decode or skipped file: %s", os.path.basename(file_path))

        logger.info("Completed processing %d files, %d valid extensions found", len(all_files), valid_extensions)
        return extensions

    except OSError as e:
        logger.error("Error reading directory %s: %s", extension_dir, e)
        return extensions

def read_hostuid(logger=LOGGER):
    path = "/nsconfig/.nscfg_hostuid"
    if not os.path.isfile(path):
        logger.error("Host UID file missing: %s", path)
        return None
    try:
        with open(path, "r") as f:
            line = f.readline().strip()
        if not line or "," not in line:
            logger.error("Invalid hostuid file format: %s", line)
            return None
        lsid, _ = line.split(",", 1)
        return lsid.strip()
    except Exception as e:
        logger.error("Error reading hostuid file: %s", e)
        return None

def validate_blob(activation_blob, logger=LOGGER):
    """Validate decoded activation_blob payload"""
    if not activation_blob:
        logger.error("No activation_blob data to validate")
        return 1
    try:
        # 1. Current system epoch must be less than exp
        now_epoch = int(time.time())
        exp = activation_blob.get("exp")
        if not isinstance(exp, int):
            logger.error("Missing or invalid exp field")
            return 1
        if now_epoch >= exp:
            logger.error("activation_blob expired: now=%d exp=%d", now_epoch, exp)
            return 1
        logger.info("Expiry OK: now=%d < exp=%d", now_epoch, exp)

        # 2. /nsconfig/.nscfg_hostuid lsid must match binding.licenseserver.lsid
        lsid_payload = (
            activation_blob.get("binding", {})
                    .get("licenseserver", {})
                    .get("lsid")
        )
        if not lsid_payload:
            logger.error("Missing lsid in binding.licenseserver")
            return 1

        host_lsid = read_hostuid()
        if not host_lsid:
            logger.error("Could not obtain host lsid from file")
            return 1

        if host_lsid != lsid_payload:
            logger.error("Host lsid mismatch: file=%s payload=%s", host_lsid, lsid_payload)
            return 1
        logger.info("Host lsid match OK: %s", host_lsid)

        logger.info("activation_blob validation successful")
        return 0
    except Exception as e:
        logger.error("Error during activation_blob validation: %s", e)
        return 1

def parse_args():
    parser = argparse.ArgumentParser(description="Decode activation_blob and optional offline activation blobs")
    parser.add_argument("--check_las_activation_blob",
                        action="store_true",
                        help="Decode offline activation blob /nsconfig/license/offline_activation_blob")
    return parser.parse_args()

def main():
    args = parse_args()
    if args.check_las_activation_blob:
        # Process only the offline activation blob
        blob_json = decode_activation_blob()
        exit_code = validate_blob(blob_json)
        sys.exit(exit_code)

    # Process all extensions in default directory
    extensions = decode_all_extensions()

    # Always output array format
    print(json.dumps(extensions))

    # Exit with error code if no valid extensions found
    if not extensions:
        sys.exit(1)

if __name__ == "__main__":
    main()